# -*- coding: utf-8 -*-

import datetime
import json
import logging
import os
import shutil
from collections import namedtuple

import six

from sandbox import sdk2

from sandbox.common.types import misc as ctm
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.fs import (
    get_unique_file_name,
    make_folder,
)

from sandbox.projects.common.yabs.cachedaemon import CacheDaemonStubSandboxNonIntegrated
import sandbox.projects.common.yabs.server.util.pack_utils as pack_utils

from sandbox.projects.yabs.base_bin_task import BaseBinTaskMixin, base_bin_task_parameters
from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.resource_types import (
    BaseBackupSdk2Resource,
    YABS_SERVER_REQUEST_LOG_GZ,
    YABS_SERVER_TESTENV_DB_FLAGS,
    YABS_SERVER_FT_DOLBILKA_PLAN,
    YabsServerStatStub,
)
from sandbox.projects.yabs.qa.constants import DEBUG_COOKIE
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.yabs_specific.sandbox import AmmoRequestlogModuleYabsSpecificSandboxAdapter
from sandbox.projects.yabs.qa.dolbilo_module.simple.adapters.sandbox import DolbiloModuleSandboxAdapter
from sandbox.projects.yabs.qa.dumper_module.adapters.sandbox import DumperModuleSandboxAdapter
from sandbox.projects.yabs.qa.mutable_parameters import MutableParameters
from sandbox.projects.yabs.qa.sut.bases_provider.adapters.sandbox import BaseStateSandboxAdapter
from sandbox.projects.yabs.qa.sut.solomon_stats import dump_solomon_stats
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox import YabsMetaSandboxAdapter
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox.parameters import YabsSUTParameters
from sandbox.projects.yabs.qa.tasks.duration_measure_task import BaseDurarionMeasureTask, BaseDurarionMeasureTaskParameters
from sandbox.projects.yabs.qa.utils import Contextable
from sandbox.projects.yabs.qa.utils.base import get_bin_bases_unpacked_size, get_max_unpacking_workers


MB_IN_GB = 1 << 10


class RequirementsParameters(sdk2.Task.Parameters):
    shard_space = sdk2.parameters.Integer(
        'Binary base space required for single shard (will account to either disk or ramdrive requirement), GB',
        default_value=185,
    )
    generic_disk_space = sdk2.parameters.Integer(
        'Generic disk space, GB',
        default_value=60,
    )
    ram_space = sdk2.parameters.Integer(
        'Ram space, GB',
        default_value=200,
    )


class PerformancePipelineResults(namedtuple('PerformancePipelineResults', [
    'rps',
    'rps_list',
]), Contextable):
    pass


class YabsServerPerformanceMetaShootResult(BaseBackupSdk2Resource):
    '''Shoot result for YabsServerPerformanceMeta'''
    shoot_index = sdk2.parameters.Integer('Shooting session number')
    mode = sdk2.parameters.String('Mode for shoot')


class YabsServerPerformanceMeta(BaseBinTaskMixin, BaseDurarionMeasureTask, sdk2.Task):
    """Task that runs performance test of meta"""

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = datetime.timedelta(hours=2).total_seconds()
        _base_bin_task_parameters = base_bin_task_parameters(
            release_version_default=ReleaseStatus.STABLE,
            resource_attrs_default={"task_bundle": "yabs_server_meta_load"},
        )

        with sdk2.parameters.Group('General settings') as general_group:
            shoot_sessions = sdk2.parameters.Integer('Shoot sessions', default_value=3)
            use_base_state_generator = sdk2.parameters.Bool(
                'Unpack bases before init modules',
                default_value=True
            )
        with sdk2.parameters.Group('Requirements settings (use these instead of requirements tab!)') as requirements_settings:
            requirements_parameters = RequirementsParameters()
        with sdk2.parameters.Group('Yabs-server module settings') as yabs_server_module_settings:
            server_module_parameters = YabsSUTParameters()
            meta_module_parameters = YabsMetaSandboxAdapter.get_init_parameters_class()()
            update_parameters_resource = sdk2.parameters.Resource(
                'Resource with JSON-dumped parameter update dict',
                resource_type=YABS_SERVER_TESTENV_DB_FLAGS,
            )
            stat_stub_data = sdk2.parameters.Resource(
                'Resource with cachedaemon data dir for stat stubs',
                resource_type=YabsServerStatStub,
                required=True,
            )
        with sdk2.parameters.Group('Ammo generation module settings') as ammo_module_settings:
            ammo_module_parameters = AmmoRequestlogModuleYabsSpecificSandboxAdapter.get_init_parameters_class()()
        with sdk2.parameters.Group('Shoot module settings') as shoot_module_settings:
            shoot_module_parameters = DolbiloModuleSandboxAdapter.get_init_parameters_class()()
        with sdk2.parameters.Group('Dumper module settings') as dumper_module_settings:
            dumper_module = DumperModuleSandboxAdapter.get_init_parameters_class()()

        duration_parameters = BaseDurarionMeasureTaskParameters()

    class Context(sdk2.Task.Context):
        hosts_slots_count = 0
        hosts_slot_index = 0

        rps = 0
        rps_list = []

    def on_save(self):
        super(YabsServerPerformanceMeta, self).on_save()
        self.Requirements.ram = self.Parameters.ram_space * MB_IN_GB
        shard_count = len(self.Parameters.stat_shards)
        bin_bases_unpacked_size = []
        if self.Parameters.use_tmpfs:
            bin_bases_unpacked_size = get_bin_bases_unpacked_size(self.Parameters.meta_binary_base_resources)
            self.Context.ramdrive_size = min(self.Parameters.generic_disk_space * MB_IN_GB, sum(bin_bases_unpacked_size) + 20 * MB_IN_GB)
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.Context.ramdrive_size,
                None
            )
            self.Requirements.disk_space = self.Parameters.generic_disk_space * MB_IN_GB
        else:
            self.Requirements.ramdrive = None
            self.Requirements.disk_space = (
                self.Parameters.generic_disk_space + self.Parameters.shard_space * shard_count
            ) * MB_IN_GB
        self.Context.unpacking_workers = get_max_unpacking_workers(bin_bases_unpacked_size, self.Requirements.ram, self.Context.ramdrive_size)

    def init_parameters(self, parameters):
        self.MutableParameters = MutableParameters.__from_parameters__(parameters)
        if self.MutableParameters.update_parameters_resource:
            update_parameters_resource_path = str(sdk2.ResourceData(self.MutableParameters.update_parameters_resource).path)
            with open(update_parameters_resource_path, 'r') as f_out:
                update_parameters_dict = json.load(f_out)
            self.MutableParameters.__dict__.update(update_parameters_dict)

    def prepare_ammo_resource(self, ammo_path, shoot_index, content_type):
        gzipped_ammo_path = pack_utils.gzip_file(ammo_path)
        ammo_resource = YABS_SERVER_REQUEST_LOG_GZ(
            self,
            'Ammo resource',
            gzipped_ammo_path,
            shoot_index=shoot_index,
            content_type=content_type,
            cachedaemon_dump_res_id=self.MutableParameters.cache_daemon_stub_resource.id,
        )
        sdk2.ResourceData(ammo_resource).ready()

    def prepare_dplan_resource(self, dplan_path, shoot_index, content_type):
        dplan_resource = YABS_SERVER_FT_DOLBILKA_PLAN(
            self,
            'Dplan resource',
            dplan_path,
            shoot_index=shoot_index,
            content_type=content_type,
            cachedaemon_dump_res_id=self.MutableParameters.cache_daemon_stub_resource.id,
        )
        sdk2.ResourceData(dplan_resource).ready()

    def shoot(self, ammo_module, shoot_module, server_module):
        dplan_path = ammo_module.get_dplan_path()
        dump_path = shoot_module.shoot_and_watch(server_module, dplan_path)

        return dump_path

    def init_modules(self):
        self.init_parameters(self.Parameters)
        self.shoot_module = DolbiloModuleSandboxAdapter(self.MutableParameters, self).create_module()
        self.dumper_module = DumperModuleSandboxAdapter(self.MutableParameters, self).create_module()

    def get_params_for_stat_stub(self, server_adapter):
        cachedaemon_dump_path = six.text_type(sdk2.ResourceData(self.MutableParameters.stat_stub_data).path)
        cachedaemon_logs_dir = get_unique_file_name(server_adapter.get_logs_dir(), 'cache_daemon_logs')
        cachedaemon_data_dir = get_unique_file_name(os.path.join(str(self.ramdrive.path), 'stats'), 'cache_daemon_data')
        return dict(
            cache_daemon_executable_path=server_adapter.get_cache_daemon_executable_path(),
            data_dir=cachedaemon_data_dir,
            dump_path=cachedaemon_dump_path,
            log_subdir=cachedaemon_logs_dir,
            services=None,
            start_on_creation=True,
            use_sub_path=True,
        )

    def get_stat_stub_ports(self, stat_stub):
        stat_stub_ports = stat_stub.get_ports_by_tag()
        return {service: port for service, port in stat_stub_ports.iteritems() if service.startswith('yabstat')}

    def run_performance_pipeline(self, parameters, flush_bases_directory=False, mode=None, index_2on1=1):
        self.init_parameters(parameters)
        rps_list = []
        mode_txt = mode + '_' if mode else ''

        stubs_cachedaemon_data_dir = get_unique_file_name(os.path.join(str(self.ramdrive.path), 'stubs'), 'cache_daemon_data')
        cachedaemon_data_dirs = []

        meta_adapter = YabsMetaSandboxAdapter(
            self.MutableParameters,
            task_instance=self,
            work_dir="meta_adapter_{}".format(index_2on1)
        )
        base_state_adapter = BaseStateSandboxAdapter(
            self.MutableParameters,
            task_instance=self,
            base_dir=meta_adapter.get_base_dir(),
            transport_resource_path=meta_adapter.get_transport_resource_path(),
        )
        cachedaemon = meta_adapter.create_cachedaemon(cachedaemon_data_dir=stubs_cachedaemon_data_dir)

        with CacheDaemonStubSandboxNonIntegrated(**self.get_params_for_stat_stub(meta_adapter)) as stat_stub:
            stat_stub_ports = self.get_stat_stub_ports(stat_stub)
            logging.info('Stat stub ports: %s', stat_stub_ports)

            with self.stage_duration('create_modules'):
                base_state = None
                if self.MutableParameters.use_base_state_generator:
                    base_state_generator = base_state_adapter.create_module(
                        unpack_workers=self.Context.unpacking_workers,
                    )
                    base_state = base_state_generator.get_bases(set(self.MutableParameters.meta_binary_base_resources))

                meta_module = meta_adapter.create_module(
                    cachedaemon,
                    stat_instances=list(stat_stub.instances.values()),
                    shared_base_state=base_state,
                    max_unpacking_workers=self.Context.unpacking_workers,
                )

            if parameters.use_default_debug_cookie:
                debug_cookie = DEBUG_COOKIE
            else:
                cookies = meta_module.get_debug_cookies()
                logging.info('Got debug cookies: %s', cookies)
                debug_cookie = cookies[0]

            ammo_module = AmmoRequestlogModuleYabsSpecificSandboxAdapter(self.MutableParameters, self, debug_cookie=debug_cookie).create_module()

            for i in range(self.Parameters.shoot_sessions):
                logging.debug('Start session %s', i)

                solomon_stats_dir = get_unique_file_name(meta_adapter.get_logs_dir(), "solomon_stats")
                make_folder(solomon_stats_dir)

                with cachedaemon, meta_module:
                    dump_path = self.shoot(
                        ammo_module,
                        self.shoot_module,
                        meta_module,
                    )
                    dump_solomon_stats(meta_module.get_server_backend_object(), solomon_stats_dir)

                shoot_result = self.dumper_module.get_processed_stats(dump_path, add_raw_stats=True)
                shoot_result_resource = sdk2.ResourceData(
                    YabsServerPerformanceMetaShootResult(
                        self,
                        'Result for shoot',
                        'shoot_result_{}{}.txt'.format(mode_txt, i),
                        shoot_index=i,
                        mode=mode,
                    )
                )
                shoot_result_resource.path.write_bytes(shoot_result['raw_stats'])
                shoot_result_resource.ready()
                rps_list.append(float(shoot_result['rps']))
                os.remove(dump_path)
                logging.debug('End session %s', i)

            if flush_bases_directory:
                meta_module.base_provider.flush_state()

            cachedaemon_data_dirs = [meta_module.cachedaemon.data_dir, stat_stub.data_dir]

        for cachedaemon_data_dir in cachedaemon_data_dirs:
            shutil.rmtree(cachedaemon_data_dir)

        meta_module.get_server_backend_object().check_access_log(
            ext_sharded={},
            quantiles={90, 99},
            error_rate_thr=1.,
            request_error_rate_thr=1.,
            ext_error_rate_thr=1.,
        )

        return PerformancePipelineResults(
            rps=max(rps_list),
            rps_list=rps_list,
        )

    @track_errors
    def on_execute(self):
        self.init_modules()
        performance_pipeline_results = self.run_performance_pipeline(self.Parameters)
        performance_pipeline_results.__to_context__(self.Context)
