import collections
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm

from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.resource_types import (
    YABS_SERVER_TESTENV_DB_FLAGS,
    YabsServerStatStub,
)
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox import YabsMetaSandboxAdapter
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox.parameters import YabsSUTParameters2On1
from sandbox.projects.yabs.qa.tasks.YabsServerPerformanceMeta import YabsServerPerformanceMeta
from sandbox.projects.yabs.qa.mutable_parameters import MutableParameters
from sandbox.projects.yabs.qa.utils.base import get_max_unpacking_workers
from sandbox.projects.yabs.qa.utils.task_hooks import get_performance_meta_2_on_1_requirements

ParametersFor2On1Batch = collections.namedtuple('ParametersFor2On1', ['parameters', 'mode', 'second_batch'])


MB_IN_GB = B_IN_MB = 1 << 10


class YabsServerPerformanceMeta2on1(YabsServerPerformanceMeta):
    '''
    Task for load b2b test of meta (2 on 1).
    '''

    class Parameters(YabsServerPerformanceMeta.Parameters):
        kill_timeout = 4 * 60 * 60

        with YabsServerPerformanceMeta.Parameters.general_group() as general_group:
            shuffle_run_order = sdk2.parameters.Bool('Shuffle order of runs within 2 on 1', default_value=True)
            flush_bases_directory = sdk2.parameters.Bool('Flush yabs-server bases directory between 2 on 1 runs', default_value=True)
        with sdk2.parameters.Group('Yabs-server module settings') as yabs_server_module_settings:
            server_module_parameters = YabsSUTParameters2On1()
            meta_module_parameters = YabsMetaSandboxAdapter.get_init_parameters_class(for_2_on_1=True)()
            update_parameters_resource_2 = sdk2.parameters.Resource(
                'Resource with JSON-dumped parameter update dict',
                resource_type=YABS_SERVER_TESTENV_DB_FLAGS,
            )
            stat_stub_data_2 = sdk2.parameters.Resource(
                'Resource with cachedaemon data dir for stat stubs',
                resource_type=YabsServerStatStub,
                required=True,
            )

    class Context(YabsServerPerformanceMeta.Context):
        rps_2 = 0
        rps_list_2 = []

    def on_enqueue(self):
        shoot_requirements = get_performance_meta_2_on_1_requirements(self.server, self.Parameters)
        self.Context.shoot_requirements = dict(shoot_requirements.__dict__)

        self.Requirements.ram = shoot_requirements.ram
        self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, shoot_requirements.ramdrive_size, None)
        self.Requirements.disk_space = shoot_requirements.disk_space

        self.Context.unpacking_workers = get_max_unpacking_workers(shoot_requirements.bin_bases_unpacked_size, shoot_requirements.ram, shoot_requirements.ramdrive_size)

    @track_errors
    def on_execute(self):
        parameters_ordered_list = [
            ParametersFor2On1Batch(self.Parameters, 'baseline', False),
            ParametersFor2On1Batch(self.generate_second_server_parameters(), 'testing', True),
        ]
        self.init_modules()
        if self.Parameters.shuffle_run_order and (self.Context.hosts_slot_index % 2):
            logging.info('Swapping test order')
            parameters_ordered_list.reverse()
        self.run_2_on_1_pipeline(*parameters_ordered_list)

    def generate_second_server_parameters(self):
        parameters = MutableParameters.__from_parameters__(self.Parameters)
        for name, value in parameters:
            if name.endswith('_2'):
                target_name = name[:-2]
                setattr(parameters, target_name, value)
        return parameters

    def run_2_on_1_pipeline(self, parameters_1, parameters_2):
        pipeline_results = self.run_performance_pipeline(
            parameters=parameters_1.parameters,
            mode=parameters_1.mode,
            flush_bases_directory=self.Parameters.flush_bases_directory,
            index_2on1=1,
        )
        pipeline_results.__to_context__(self.Context, second_batch=parameters_1.second_batch)

        pipeline_results_2 = self.run_performance_pipeline(
            parameters_2.parameters,
            mode=parameters_2.mode,
            index_2on1=2,
        )
        pipeline_results_2.__to_context__(self.Context, second_batch=parameters_2.second_batch)
