# -*- coding: utf-8 -*-

from collections import namedtuple
from datetime import timedelta
import logging

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.utils import get_task_link
from sandbox.projects.yabs.base_bin_task import BaseBinTaskMixin, base_bin_task_parameters
from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.performance import scheduling
from sandbox.projects.yabs.qa.performance.collect import collect_rps
from sandbox.projects.yabs.qa.performance.compare import compare_rps
from sandbox.projects.yabs.qa.performance.update_parameters import apply_update_parameters_resource
from sandbox.projects.yabs.qa.tasks.YabsServerPerformanceMeta2on1 import YabsServerPerformanceMeta2on1
from sandbox.projects.yabs.qa.tasks.YabsServerPerformanceMeta import RequirementsParameters
from sandbox.projects.yabs.qa.tasks.base_compare_task.parameters import BaseCompareTaskParameters
from sandbox.projects.yabs.qa.tasks.base_compare_task.task import BaseCompareTask
from sandbox.projects.yabs.qa.utils.general import startrek_hyperlink, html_hyperlink
from sandbox.projects.yabs.qa.resource_types import BaseBackupSdk2Resource


DIFF_REPORT_FILENAME = 'diff_report.html'
DIFF_REPORT_TTL = 60


DiffResults = namedtuple("DiffResults", ["rps"])


class YabsServerPerformanceMetaDiffReport(BaseBackupSdk2Resource):
    has_diff = sdk2.Attributes.Bool("Flag for diff in resource", default=False)


def create_comparison_metareport(lines, links, line_sep='\n', link_formatter=startrek_hyperlink):
    task_executing_report_template = line_sep.join(['{lines}', '{links}', ])
    return task_executing_report_template.format(
        lines=line_sep.join(lines),
        links=line_sep.join(link_formatter(link_url, link_text) for link_text, link_url in links)
    )


class YabsServerPerformanceMetaCmp(BaseBinTaskMixin, BaseCompareTask):
    ignore_compare_input_parameters = [
        'binary_base_resources',
        'server_resource',
        'stat_binary_base_resources',
        'stat_server_resource',
        'meta_binary_base_resources',
        'meta_server_resource',
    ]
    critical_compare_input_parameters = [
        'cache_daemon_stub_resource',
        'requestlog_resource',
        'shoot_plan_resource',
    ]

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseCompareTaskParameters):
        kill_timeout = int(timedelta(minutes=20).total_seconds())
        _base_bin_task_parameters = base_bin_task_parameters(
            release_version_default=ctt.ReleaseStatus.STABLE,
            resource_attrs_default={"task_bundle": "yabs_server_meta_load"},
        )
        push_tasks_resource = True

        with sdk2.parameters.Group('Comparison parameters') as comparison_parameters:
            rps_diff_threshold_percent = sdk2.parameters.Float('RPS diff threshold, %', default_value=10.0)
            improvement_is_diff = sdk2.parameters.Bool('Improvement is diff', default_value=True)

        with sdk2.parameters.Group('Batch mode parameters') as batch_parameters:
            need_ram_check = sdk2.parameters.Bool('Need additional RAM check', default_value=False)
            shoot_request_limit = sdk2.parameters.Integer('Shoot request limit', default=250000)
            shoot_threads = sdk2.parameters.Integer('Shoot threads', default=230)  # 250 (queue size) - 15 (threadpool size) - 5 (magic constant) = 230
            queryargs_update_dict = sdk2.parameters.JSON('Queryargs update dict', default={
                "abd": "0",
                "sysconst-update": "request-log-probability:300,match-log-mode:0,ss-skip-token-enabled:0,EnableBigbWithTvm:0,DisableAsyncMatrixnet:1"
            })

        with sdk2.parameters.Group('Batch settings') as batch_settings:
            subtasks_count = sdk2.parameters.Integer('Number of subtasks', default_value=16)
            abandoned_limit = sdk2.parameters.Integer('Abandoned subtask limit', default_value=5)
            failure_limit = sdk2.parameters.Integer('Failed subtask limit', default_value=5)
            other_break_limit = sdk2.parameters.Integer('Otherwise broken subtask limit', default_value=4)

        with sdk2.parameters.Group('Requirements settings for 2on1 tasks') as requirements_settings:
            requirements_parameters = RequirementsParameters()

    class Context(sdk2.Task.Context):
        has_diff = False
        subtasks_2_on_1_data = []

    def subtasks_list(self):
        return scheduling.get_task_list_from_data_iterable(self.Context.subtasks_2_on_1_data)

    def wait_subtasks(self):
        raise sdk2.WaitTask(
            filter(
                lambda task: task.status not in ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                self.subtasks_list()
            ),
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=False
        )

    def do_batch(self):
        parameters_2_on_1, task_count, checker = self.produce_2_on_1_parameters()
        logging.info('Task count:  {}'.format(task_count))
        if not self.Context.subtasks_2_on_1_data:
            scheduling.schedule_tasks(
                self,
                YabsServerPerformanceMeta2on1,
                parameters_2_on_1,
                task_count,
                self.Context.subtasks_2_on_1_data,
                checker,
                description='Child 2 on 1 run' + self.Parameters.description,
                tasks_resource=self.Requirements.tasks_resource,
            )
            need_wait = True
        else:
            need_wait = scheduling.check_and_reschedule_tasks(
                self,
                self.Context.subtasks_2_on_1_data,
                checker,
                description='Child 2 on 1 rerun' + self.Parameters.description,
                tasks_resource=self.Requirements.tasks_resource,
            )
        if need_wait:
            self.wait_subtasks()

        return self.compare_from_subtasks_routine()

    def compare_from_subtasks_routine(self):
        return DiffResults(
            rps=self.compare_from_subtasks(self.subtasks_list(), collect_rps, compare_rps),
        )

    def compare_from_subtasks(self, subtasks, collect_method, compare_method):
        pre_results = collect_method(subtasks)
        pre_results.__to_context__(self.Context)
        test_results = collect_method(subtasks, second_batch=True)
        test_results.__to_context__(self.Context, second_batch=True)
        return compare_method(pre_results, test_results)

    @track_errors
    def on_execute(self):
        terminate_task = self.check_tasks_parameters()
        if terminate_task:
            return

        diff_results = self.do_batch()

        self.Context.has_diff = self.Parameters.has_diff = self.evaluate_results(diff_results)
        html_report, st_report, short_report_text = self.make_report(diff_results)

        self.Context.short_report_text = short_report_text
        self.Context.short_report_link = get_task_link(self.id)
        with open(DIFF_REPORT_FILENAME, 'w') as f:
            f.write(html_report)

        self.set_info(html_report, do_escape=False)
        self.Parameters.st_report = st_report

        YabsServerPerformanceMetaDiffReport(
            self,
            description='Report resource',
            path=DIFF_REPORT_FILENAME,
            ttl=DIFF_REPORT_TTL,
            has_diff=self.Context.has_diff,
        )

    def make_report(self, results):
        report_lines = []
        report_links = []
        short_report_text = 'No report :('
        if results.rps:
            short_report_text = "Max requests/sec changed by {:+.1f}% ({:.2f} -> {:.2f})".format(
                results.rps.diff_percent, results.rps.pre_hl_median, results.rps.test_hl_median,
            )
            report_lines.append(short_report_text)

        html_report = create_comparison_metareport(report_lines, report_links, line_sep="<br>", link_formatter=html_hyperlink)
        st_report = create_comparison_metareport(report_lines, report_links, line_sep="\n", link_formatter=startrek_hyperlink)

        return html_report, st_report, short_report_text

    def evaluate_results(self, results):
        has_diff = False

        if results.rps:
            rps_degraded = results.rps.diff_percent < -1 * self.Parameters.rps_diff_threshold_percent
            rps_improved = results.rps.diff_percent > self.Parameters.rps_diff_threshold_percent
            has_rps_diff = (rps_degraded or (rps_improved and self.Parameters.improvement_is_diff)) and self.Parameters.rps_diff_threshold_percent
            has_diff = has_diff or has_rps_diff

        return bool(has_diff)

    def produce_2_on_1_parameters(self):
        return_parameters = apply_update_parameters_resource(self.pre_task.Parameters)
        test_parameters = apply_update_parameters_resource(self.test_task.Parameters)
        for name, _ in self.pre_task.Parameters.server_module_parameters:
            return_parameters.__dict__[name + '_2'] = test_parameters.__dict__[name]
        for name, _ in self.Parameters.pre_task.Parameters.meta_module_parameters:
            return_parameters.__dict__[name + '_2'] = test_parameters.__dict__[name]
        for name, value in self.Parameters.requirements_parameters:
            return_parameters.__dict__[name] = value
        return_parameters.__dict__['use_tmpfs'] = True
        return_parameters.__dict__['use_tmpfs_2'] = True
        return_parameters.__dict__["need_ram_check"] = self.Parameters.need_ram_check
        return_parameters.__dict__['stat_stub_data'] = self.pre_task.Parameters.yabs_server_stat_stub_resource
        return_parameters.__dict__['stat_stub_data_2'] = self.test_task.Parameters.yabs_server_stat_stub_resource
        return_parameters.__dict__['shoot_request_limit'] = self.Parameters.shoot_request_limit
        return_parameters.__dict__['shoot_threads'] = self.Parameters.shoot_threads
        return_parameters.__dict__['queryargs_update_dict'] = self.Parameters.queryargs_update_dict
        return_parameters.__dict__['circular_session'] = True

        return return_parameters, self.Parameters.subtasks_count, scheduling.TaskChecker(
            failure_limit=self.Parameters.failure_limit,
            abandoned_limit=self.Parameters.abandoned_limit,
            other_break_limit=self.Parameters.other_break_limit
        )
