#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
import logging
import os

from sandbox import sdk2
from sandbox.sandboxsdk import environments
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.fs import (
    make_folder,
    get_unique_file_name,
)

from sandbox.projects.yabs.base_bin_task import BaseBinTaskMixin, base_bin_task_parameters
from sandbox.projects.common.yabs.cachedaemon import CacheDaemonStubSandboxNonIntegrated
import sandbox.projects.common.yabs.server.util.pack_utils as pack_utils

from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.resource_types import (
    YABS_SERVER_CACHE_DAEMON_STUB_DATA,
    YABS_SERVER_FT_DOLBILKA_PLAN,
    YABS_SERVER_REQUEST_LOG_GZ,
    YABS_SERVER_TESTENV_DB_FLAGS,
    YabsResponseDumpUnpacker,
    YabsServerResponseDump,
    YabsServerStatStub,
    BaseBackupSdk2Resource,
)
from sandbox.projects.yabs.qa.constants import DEBUG_COOKIE
from sandbox.projects.yabs.qa.hamster.utils import shoot_task_enabled_hamsters
from sandbox.projects.yabs.qa.sut.bases_provider.adapters.sandbox import BaseStateSandboxAdapter
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox import YabsMetaSandboxAdapter, YabsStatSandboxAdapter
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox.parameters import YabsSUTParameters
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.yabs_specific.sandbox import AmmoRequestlogModuleYabsSpecificSandboxAdapter
from sandbox.projects.yabs.qa.dolbilo_module.simple.adapters.sandbox import DolbiloModuleSandboxAdapter
from sandbox.projects.yabs.qa.mutable_parameters import MutableParameters
from sandbox.projects.yabs.qa.response_tools.unpacker import (
    find_unpacker_resource,
    get_unpacker_executable_path,
    uc_compress,
    unpack_responses,
)
from sandbox.projects.yabs.qa.tasks.duration_measure_task import BaseDurarionMeasureTask, BaseDurarionMeasureTaskParameters
from sandbox.projects.yabs.qa.utils.base import get_max_unpacking_workers
from sandbox.projects.yabs.qa.utils.task_hooks import get_shoot_requirements
from sandbox.projects.yabs.qa.utils.shoot import check_proto

from sandbox.projects.yabs.qa.tasks.YabsServerPrepareStatStub.cachedaemon import (
    fill_cachedaemon_with_exts,
    STAT_HANDLERS,
)


logger = logging.getLogger(__name__)


class YabsServerResponsesFolder(BaseBackupSdk2Resource):
    pass


def check_meta_access_log(meta_module):
    meta_module.get_server_backend_object().check_access_log(
        ext_sharded={},
        quantiles={90, 99},
        error_rate_thr=.005,
        request_error_rate_thr=.005,
        ext_error_rate_thr=.02,
        ignore_handlers={
            'ping',
            'solomon_stats',
            'solomon_stats_spack',
        },
        ignore_ext={
            'bigb_balancer',
            'bigb_balancer_50',
            'bigb_rf_balancer',
            'yacofast_ssp',
            'linear_models_service',
            'linear_models_service_wide',
            'an_to_yabs_redirect_bad',  # @astkachev remove ignorance of this ext call after BSSERVER-16473
            'adv_machine_knn',  # remove after BSSERVER-16586
            'adv_machine_cat_engine',  # remove after BSSERVER-16586
            'adv_machine_offer_online_quorums',  # remove after BSSERVER-16586
            'yabs_banner_models_gpu',  # remove after BSSERVER-17951
            'yabs_hit_models_gpu',  # SHMDUTY-143
            'goalnet',  # SHMDUTY-159
            'yabs_hit_models_heavy',  # SHMDUTY-287
            'yabs_hit_models_heavy_nanny',  # SHMDUTY-287
            'yabs_hit_models_light_nanny',  # SHMDUTY-287
            'rsya_hit_models_light_01',  # SHMDUTY-287
            'rsya_hit_models_heavy_01',  # SHMDUTY-287
            'pnocsy_2',  # SHMDUTY-258
            'pnocsy350',  # SHMDUTY-291
            'pnocsy500',  # SHMDUTY-291
            'offer_match',  # SHMDUTY-327
            'word_net_service',  # SHMDUTY-392
            'panther',  # BSSERVER-22489
        },
        ext_service_thresholds={
            'pnocsy': 0.05,
            'pnocsy350': 1,  # SHMDUTY-291
            'pnocsy500': 1,  # SHMDUTY-291
        },
    )


def check_stat_access_log(stat_module):
    for stat in stat_module.get_instances():
        stat.check_access_log(
            ext_sharded={},
            quantiles={90, 99},
            error_rate_thr=.005,
            request_error_rate_thr=.01,
            ext_error_rate_thr=.005,
            ignore_handlers={
                'ping',
            },
            ignore_ext={
                'bigb_rf_balancer',
                'linear_models_service',
                'linear_models_service_wide',
            },
            ext_service_thresholds={
                'bscount_fbs': 0.1,
                'bscount_fbs2': 0.1,
            },
        )


class YabsServerPrepareStatStub(BaseBinTaskMixin, BaseDurarionMeasureTask, sdk2.Task):
    """Prepare and upload the resource with cached yabs server stats responses (cachedaemon data dir)"""

    class Parameters(sdk2.Task.Parameters):
        _base_bin_task_parameters = base_bin_task_parameters(
            release_version_default=ReleaseStatus.STABLE,
            resource_attrs_default={"task_bundle": "yabs_server_meta_load"},
        )
        kill_timeout = 3 * 60 * 60

        with sdk2.parameters.Group('General settings') as general_group:
            use_base_state_generator = sdk2.parameters.Bool(
                'Unpack bases before init modules',
                default_value=True
            )
            update_parameters_resource = sdk2.parameters.Resource(
                'Resource with JSON-dumped parameter update dict',
                resource_type=YABS_SERVER_TESTENV_DB_FLAGS,
            )
            dump_parser_resource = sdk2.parameters.Resource(
                'Dump parser resource',
                resource_type=YabsResponseDumpUnpacker,
            )
            save_shoot_dump_resource = sdk2.parameters.Bool(
                'Save shoot dump resource',
                default=False,
            )
        with sdk2.parameters.Group('Requirements settings (use these instead of requirements tab!)') as requirements_settings:
            shard_space = sdk2.parameters.Integer(
                'Binary base space required for single shard (will account to either disk or ramdrive requirement), GB',
                default_value=185,
            )
            generic_disk_space = sdk2.parameters.Integer(
                'Generic disk space, GB',
                default_value=350
            )
            ram_space = sdk2.parameters.Integer(
                'Ram space, GB',
                default_value=480,
            )
        with sdk2.parameters.Group('Yabs-server module settings') as yabs_server_module_settings:
            server_module_parameters = YabsSUTParameters()
            meta_module_parameters = YabsMetaSandboxAdapter.get_init_parameters_class()()
            stat_module_parameters = YabsStatSandboxAdapter.get_init_parameters_class()()
        with sdk2.parameters.Group('Ammo generation module settings') as ammo_module_settings:
            ammo_module_parameters = AmmoRequestlogModuleYabsSpecificSandboxAdapter.get_init_parameters_class()()
        with sdk2.parameters.Group('Shoot module settings') as shoot_module_settings:
            shoot_module_parameters = DolbiloModuleSandboxAdapter.get_init_parameters_class()()
        with sdk2.parameters.Group('Debug options') as debug_parameters:
            save_responses_folder = sdk2.parameters.Bool('Save parsed responses data to resource', default_value=False)
            save_data_for_stat_stub = sdk2.parameters.Bool('Save stat data to resource', default_value=False)
        with sdk2.parameters.Output:
            yabs_server_stat_stub_resource = sdk2.parameters.Resource(
                YabsServerStatStub.__doc__,
                resource_type=YabsServerStatStub)
            hamster_stub_resource = sdk2.parameters.Resource(
                "Hamster stubs resource",
                resource_type=YABS_SERVER_CACHE_DAEMON_STUB_DATA)

        duration_parameters = BaseDurarionMeasureTaskParameters()

    class Requirements(sdk2.Requirements):
        client_tags = (ctc.Tag.LINUX_XENIAL | ctc.Tag.YABS) & ctc.Tag.SSD
        environments = (
            environments.PipEnvironment('futures'),
        )

    def on_enqueue(self):
        super(YabsServerPrepareStatStub, self).on_enqueue()

        shoot_requirements = get_shoot_requirements(self.server, self.Parameters)
        self.Context.shoot_requirements = dict(shoot_requirements.__dict__)

        self.Requirements.ram = shoot_requirements.ram
        self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, shoot_requirements.ramdrive_size, None)
        self.Requirements.disk_space = shoot_requirements.disk_space

        self.Context.unpacking_workers = get_max_unpacking_workers(shoot_requirements.bin_bases_unpacked_size, shoot_requirements.ram, shoot_requirements.ramdrive_size)

    # FIXME: avoid copypaste from YabsServerB2BFuncShoot2
    @property
    def dump_parser_resource(self):
        if self.MutableParameters.dump_parser_resource:
            dump_parser_resource = self.MutableParameters.dump_parser_resource
        else:
            dump_parser_resource = find_unpacker_resource(
                global_key=self.MutableParameters.meta_server_resource.global_key,
                global_key_type=self.MutableParameters.meta_server_resource.global_key_type,
                build_mode="release",
            )
        return dump_parser_resource

    # FIXME: avoid copypaste from YabsServerB2BFuncShoot2
    @property
    def dump_parser_path(self):
        return get_unpacker_executable_path(self.dump_parser_resource)

    enabled_hamsters = property(shoot_task_enabled_hamsters)

    def init_parameters(self):
        self.MutableParameters = MutableParameters.__from_parameters__(self.Parameters)
        if self.Parameters.update_parameters_resource:
            update_parameters_resource_path = str(sdk2.ResourceData(self.Parameters.update_parameters_resource).path)
            with open(update_parameters_resource_path, 'r') as f_out:
                update_parameters_dict = json.load(f_out)
            self.MutableParameters.__dict__.update(update_parameters_dict)

    def prepare_ammo_resource(self, ammo_path, shoot_index, content_type):
        gzipped_ammo_path = pack_utils.gzip_file(ammo_path)
        ammo_resource = YABS_SERVER_REQUEST_LOG_GZ(
            self,
            'Ammo resource',
            gzipped_ammo_path,
            shoot_index=shoot_index,
            content_type=content_type,
            cachedaemon_dump_res_id=self.MutableParameters.cache_daemon_stub_resource.id,
        )
        sdk2.ResourceData(ammo_resource).ready()

    def prepare_dplan_resource(self, dplan_path, shoot_index, content_type):
        dplan_resource = YABS_SERVER_FT_DOLBILKA_PLAN(
            self,
            'Dplan resource',
            dplan_path,
            shoot_index=shoot_index,
            content_type=content_type,
            cachedaemon_dump_res_id=self.MutableParameters.cache_daemon_stub_resource.id,
        )
        sdk2.ResourceData(dplan_resource).ready()

    def shoot(self, ammo_module, shoot_module, cachedaemon, stat_module, meta_module, content_type, shoot_index=0):
        self.prepare_ammo_resource(
            ammo_module.get_ammo_path(),
            shoot_index=shoot_index,
            content_type=content_type,
        )
        self.prepare_dplan_resource(
            ammo_module.get_dplan_path(),
            shoot_index=shoot_index,
            content_type=content_type,
        )
        with cachedaemon, stat_module, meta_module:
            dplan_path = ammo_module.get_dplan_path()
            dump_path = shoot_module.shoot_and_watch(meta_module, dplan_path)

        return dump_path

    def shoot_and_check_proto(self, ammo_module, shoot_module, cachedaemon, stat_module, meta_module, ext_tags, content_type='primary_ammo', shoot_index=0):
        dump_path = self.shoot(
            ammo_module, shoot_module, cachedaemon, stat_module, meta_module,
            content_type=content_type,
            shoot_index=shoot_index,
        )
        bad_requests_path, _, _ = check_proto(self.dump_parser_path, dump_path, ext_tags)

        need_reshoot = ammo_module.update_ammo(bad_requests_path, content_type=content_type)
        if not need_reshoot:
            logging.info('Finished {} reshoot session on attempt {}'.format(content_type, shoot_index))

        return dump_path, need_reshoot

    def shoot_with_reshoot(self, ammo_module, shoot_module, cachedaemon, stat_module, meta_module, content_type, attempts=3, ext_tags=STAT_HANDLERS):
        result_dump_paths = []
        for shoot_index in range(attempts):
            dump_path, need_reshoot = self.shoot_and_check_proto(
                ammo_module, shoot_module, cachedaemon, stat_module, meta_module,
                ext_tags,
                content_type=content_type,
                shoot_index=shoot_index,
            )
            result_dump_paths.append(dump_path)
            if not need_reshoot:
                break
        return result_dump_paths

    def prepare_exts_stub(self, adapter, responses_dir):
        cachedaemon_logs_dir = get_unique_file_name(adapter.get_logs_dir(), 'cache_daemon_logs')

        cachedaemon = CacheDaemonStubSandboxNonIntegrated(
            cache_daemon_executable_path=adapter.get_cache_daemon_executable_path(),
            log_subdir=cachedaemon_logs_dir,
            services=None,
            use_sub_path=True,
            key_header='x-yabs-request-id,x-yabs-ext-request-key',
        )

        stat_cachedaemon_data_dir = get_unique_file_name(adapter.get_work_dir(), 'stat_cache_daemon_data')
        stat_instance_types = [
            'yabstat{:02d}'.format(stat_shard)
            for stat_shard in self.Parameters.stat_shards
        ]
        for instance_type in stat_instance_types:
            cachedaemon.create_instance(
                instance_type,
                data_dir=stat_cachedaemon_data_dir,
                start_on_creation=False,
            )

        logger.debug("Enabled hamsters: %s", self.enabled_hamsters)
        enabled_hamster_tags = list(self.enabled_hamsters.keys())
        hamster_cachedaemon_data_dir = get_unique_file_name(adapter.get_work_dir(), 'hamster_cache_daemon_data')
        for instance_type in enabled_hamster_tags:
            cachedaemon.create_instance(
                instance_type,
                data_dir=hamster_cachedaemon_data_dir,
                start_on_creation=False,
            )

        allowed_tags = STAT_HANDLERS + enabled_hamster_tags
        fill_cachedaemon_with_exts(cachedaemon, responses_dir, allowed_tags=allowed_tags)

        return stat_cachedaemon_data_dir, hamster_cachedaemon_data_dir

    def create_stat_stub_resource(self, data_dir):
        resource = YabsServerStatStub(
            self,
            "Cachedaemon stubs for stat shards {}".format(
                ",".join(str(shard) for shard in sorted(self.Parameters.stat_shards))),
            data_dir,
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def create_hamster_stub_resource(self, data_dir):
        resource = YABS_SERVER_CACHE_DAEMON_STUB_DATA(
            self,
            "Cachedaemon stubs for hamsters",
            data_dir,
            ttl=14,
            provided_tags=" ".join(sorted(list(self.enabled_hamsters.keys()))),
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def save_shoot_dump_resource(self, dump_path, shoot_index=0, pack_type='lz4hc', debug_cookie=DEBUG_COOKIE):
        logging.debug("Compress shoot dump")
        compressed_dump_path = uc_compress(self, dump_path, pack_type=pack_type)

        logging.debug("Create shoot dump resources")
        dump_resource = YabsServerResponseDump(
            self,
            'Shoot dump resource',
            compressed_dump_path,
            pack_codec=pack_type,
            debug_cookie=debug_cookie,
            shoot_index=shoot_index,
            debug_mode='proto_binary',
            dump_parser_id=self.dump_parser_resource.id,
            ammo_type=self.MutableParameters.meta_mode,
        )
        sdk2.ResourceData(dump_resource).ready()

    @track_errors
    def on_execute(self):
        self.init_parameters()

        with self.stage_duration('create_modules'):
            stat_adapter = YabsStatSandboxAdapter(
                self.MutableParameters,
                task_instance=self,
                work_dir="stat_adapter",
            )
            meta_adapter = YabsMetaSandboxAdapter(
                self.MutableParameters,
                task_instance=self,
                work_dir="meta_adapter",
            )
            base_state_adapter = BaseStateSandboxAdapter(
                self.MutableParameters,
                task_instance=self,
                base_dir=stat_adapter.get_base_dir(),
                transport_resource_path=stat_adapter.get_transport_resource_path(),
            )

            base_state = None
            if self.MutableParameters.use_base_state_generator:
                base_state_generator = base_state_adapter.create_module(
                    unpack_workers=self.Context.unpacking_workers,
                )
                base_state = base_state_generator.get_bases(set(self.MutableParameters.stat_binary_base_resources + self.MutableParameters.meta_binary_base_resources))

            stubs_cachedaemon_data_dir = get_unique_file_name(os.path.join(str(self.ramdrive.path), 'stubs'), 'cache_daemon_data')
            cachedaemon = stat_adapter.create_cachedaemon(
                cachedaemon_data_dir=stubs_cachedaemon_data_dir,
                threads_per_service={'bscount': self.Parameters.shoot_threads * len(self.Parameters.stat_shards)},
            )
            stat_module = stat_adapter.create_module(
                cachedaemon=cachedaemon,
                shared_base_state=base_state,
                max_unpacking_workers=self.Context.unpacking_workers
            )
            meta_module = meta_adapter.create_module(
                cachedaemon=cachedaemon,
                stat_instances=stat_module.get_instances(),
                shared_base_state=stat_module.shared_base_state,
                max_unpacking_workers=self.Context.unpacking_workers
            )

        if self.Parameters.use_default_debug_cookie:
            debug_cookie = DEBUG_COOKIE
        else:
            cookies = meta_module.get_debug_cookies()
            logger.info('Got debug cookies: %s', cookies)
            debug_cookie = cookies[0]

        ammo_adapter = AmmoRequestlogModuleYabsSpecificSandboxAdapter(self.MutableParameters, self, debug_cookie=debug_cookie)
        ammo_module = ammo_adapter.create_module()
        shoot_adapter = DolbiloModuleSandboxAdapter(self.MutableParameters, self)
        shoot_module = shoot_adapter.create_module()

        dump_paths = self.shoot_with_reshoot(ammo_module, shoot_module, cachedaemon, stat_module, meta_module, content_type='primary_ammo')

        save_shoot_dump_resource = self.Parameters.save_shoot_dump_resource
        try:
            check_meta_access_log(meta_module)
            check_stat_access_log(stat_module)
        except Exception:
            save_shoot_dump_resource = True
            raise
        finally:
            if save_shoot_dump_resource:
                for i, dump_path in enumerate(dump_paths):
                    self.save_shoot_dump_resource(dump_path, shoot_index=i)

        logger.info('Merge shoot dumps')
        merged_dumps_dir = make_folder('merged_dumps', delete_content=True)
        for dump_path in dump_paths:
            unpack_responses(self, dump_path, merged_dumps_dir, self.dump_parser_path)

        if self.Parameters.save_responses_folder:
            sdk2.ResourceData(YabsServerResponsesFolder(
                self,
                'Merged dumps dir',
                merged_dumps_dir,
            )).ready()

        logger.info('Prepare external services stubs')
        stat_cachedaemon_data_dir, hamster_cachedaemon_data_dir = self.prepare_exts_stub(stat_adapter, merged_dumps_dir)

        logger.info('Upload cachedaemon stub resources')
        self.Parameters.yabs_server_stat_stub_resource = self.create_stat_stub_resource(stat_cachedaemon_data_dir)

        # Creating resource with empty directory will raise InvalidResource error
        if os.path.exists(hamster_cachedaemon_data_dir) and os.listdir(hamster_cachedaemon_data_dir):
            self.Parameters.hamster_stub_resource = self.create_hamster_stub_resource(hamster_cachedaemon_data_dir)
