# -*- coding: utf-8 -*-
import copy
import datetime
import itertools
import json
import logging
import os
import time

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.environments import SvnEnvironment, PipEnvironment
from sandbox.sandboxsdk.parameters import (
    SandboxStringParameter,
    SandboxIntegerParameter,
    SandboxBoolParameter,
    LastReleasedResource,
    Container,
)
from sandbox.sandboxsdk.process import run_process
import sandbox.sandboxsdk.paths

from sandbox.projects.common.utils import get_or_default
from sandbox.projects.common.yabs.server.db.task.cs import YabsCSTask, UseSaveInputFromCS, SaveAllInputs, FilterInputArchiveTablesByOrderID, YtPool
from sandbox.projects.common.yabs.server.db.task.mysql import install_yabs_mysql, synpack_tables
from sandbox.projects.common.yabs.server.db import yt_bases
from sandbox.projects.common.yabs.server.db.utils import get_cs_import_ver
from sandbox.projects.common.yabs.server.util.general import check_tasks

from sandbox.projects.yabs.qa.tasks.YabsServerSaveInput import TABLES_TO_BE_FILTERED_BY_ORDER_ID
from sandbox.projects.yabs.qa.resource_types import YabsDevelopmentYtOneshotBinary, YT_ONESHOTS_PACKAGE, BaseBackupSdk2Resource
from sandbox.projects.yabs.qa.utils.base import generate_basenos
from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink
from sandbox.projects.yabs.qa.utils.yt_utils import set_yt_node_ttl, yt_copy, yt_shallow_copy, mount_tables_batched
from sandbox.projects.yabs.qa.utils.importer import get_importers_source_tables

from sandbox.projects.yabs.qa.bases.sample_tables.parameters import (
    SamplingStrategyParameter,
    SamplingYtPool,
    SamplingQueryTemplateResourceParameter,
    SamplingTablesResourceParameter,
    SamplingStrategy,
    FindTablesForSampling,
    SamplingTablesBlackListResourceParameter,
    MinRowsCountForSamplingTables,
    KeysForSamplingTables,
)

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace, trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import enqueue_task


IMPORT_DIGEST_KEY = 'digest'
IMPORT_PREFIX_KEY = 'destination_prefix'
ONESHOT_YT_PROXY = 'hahn'
ONESHOT_ARCHIVE_ROOT = '//home/yabs-cs-sandbox/input-archive-oneshots'


logger = logging.getLogger(__name__)


def get_input_spec_missing_keys(cs_import_info, input_spec_path):
    need_keys = map(
        lambda table: table['id'],
        itertools.chain(
            *map(
                lambda import_item: cs_import_info[import_item]['tables'],
                cs_import_info.keys()
            )
        )
    )
    with open(input_spec_path) as f:
        spec = json.load(f)
    return set(need_keys) - spec.viewkeys()


class YabsLaunchedCsOperations(BaseBackupSdk2Resource):
    """
        List of annotated launched operation ids
    """
    auto_backup = True
    ttl = 28


class ImportDestinationPath(SandboxStringParameter):
    name = 'import_destination_path'
    description = 'Destination YT node to save import results'
    required = True


class BinDBList(SandboxStringParameter):
    name = 'bin_db_list'
    default_value = ''
    description = 'Binary base tags (will run import for all bases if empty)'


class ImportActions(SandboxStringParameter):
    name = 'import_actions'
    description = 'cs_import actions (default: all present in --print_info). Ignored if "use_cs_cycle" option is turned on'
    required = False


class UseCsCycle(SandboxBoolParameter):
    name = 'use_cs_cycle'
    description = 'use cs_cycle for generate imports'
    default_value = False


class LowerReuseableTTLLimit(SandboxIntegerParameter):
    name = 'lower_reuseable_ttl_limit'
    description = 'Lower limit for output TTL, days (does not work for runs with patch)'
    default_value = 1


class OneShotPath(SandboxStringParameter):
    name = 'oneshot_path'
    description = 'Arcadia path to YT oneshot'


class OneShotArgs(SandboxStringParameter):
    name = 'oneshot_args'
    description = 'Arguments for YT oneshot'


class ReuseImportResults(SandboxBoolParameter):
    name = 'reuse_import_results'
    description = 'Reuse import results'
    default_value = True


class WriteYtDebugLog(SandboxBoolParameter):
    name = 'write_yt_debug_log'
    description = 'Write YT Debug Log to the separate file'
    default_value = True


class CalculateDigest(SandboxBoolParameter):
    name = 'calc_digest'
    description = 'Calculate digest for import result'
    default_value = False


class ShallowDataCopy(SandboxBoolParameter):
    name = 'shallow_data_copy'
    description = 'Shallow data copy for YT oneshot'
    default_value = True


class YabsDevelopmentYtOneshotResource(LastReleasedResource):
    name = 'yabs_development_yt_oneshot_resource'
    description = 'Development YT oneshot resource for CS import'
    required = False
    resource_type = YabsDevelopmentYtOneshotBinary


class Container(Container):
    required = False
    # default_value = 2094988479  # https://sandbox.yandex-team.ru/resource/2094988479 YABS_MYSQL_LXC_IMAGE 08.04.2021
    default_value = None
    description = 'Container'


class Parameters(sdk2.Parameters):
    importers = sdk2.parameters.List(
        label='Importers to run',
        description='If empty will run all importers needed to generate bases',
    )
    require_mysql = sdk2.parameters.Bool("Require MySQL", default=True)
    with require_mysql.value[True]:
        install_yabs_mysql = sdk2.parameters.Bool(
            label='Install yabs mysql in task itsel',
            description=' Otherwise it is assumed that the task is running in a LXC container with mysql.',
            default=True,
        )


class YabsServerRealRunCSImport(YabsCSTask):  # pylint: disable=R0904
    execution_space = 40 * 1024

    type = 'YABS_SERVER_REAL_RUN_CS_IMPORT'

    client_tags = Tag.GENERIC & Tag.LINUX_PRECISE

    input_parameters = YabsCSTask.input_parameters + tuple(Parameters) + (
        ImportDestinationPath,
        BinDBList,
        ImportActions,
        UseCsCycle,
        OneShotPath,
        OneShotArgs,
        ReuseImportResults,
        WriteYtDebugLog,
        CalculateDigest,
        YabsDevelopmentYtOneshotResource,
        ShallowDataCopy,
        Container,
    )

    privileged = True

    environment = (
        SvnEnvironment(),
        PipEnvironment('yandex-yt', use_wheel=True),
        PipEnvironment('yql', use_wheel=True),
    ) + YabsCSTask.environment

    def on_enqueue(self):
        self.ctx['kill_timeout'] = 7 * 3600

    def run_shallow_data_copy(self, yt_client, archive_root, oneshot_tables, destination_ypath_prefix, ):
        from yt.wrapper import YtClient, ypath_join
        production_yt_client = YtClient(proxy=yt_bases.YT_PROXY, token=self.get_yt_token())
        dynamic_tables = []
        for original_oneshot_table_path in oneshot_tables:
            src_oneshot_table_path = ypath_join(archive_root,
                                                'yt_tables',
                                                original_oneshot_table_path.lstrip('/'))
            dst_oneshot_table_path = ypath_join(destination_ypath_prefix,
                                                'yt_tables',
                                                original_oneshot_table_path.lstrip('/'))
            logging.info('Copy %s to %s', src_oneshot_table_path, dst_oneshot_table_path)
            yt_copy(yt_client, src_oneshot_table_path, dst_oneshot_table_path)
            if production_yt_client.get_attribute(original_oneshot_table_path, 'dynamic'):
                dynamic_tables.append(dst_oneshot_table_path)

        exclude_tables = [
            ypath_join(archive_root, 'yt_tables', oneshot_table.lstrip('/'))
            for oneshot_table in oneshot_tables
        ]
        logger.info('Shallow copy %s to %s, excluding %s',
                    archive_root, destination_ypath_prefix, exclude_tables)
        yt_shallow_copy(
            yt_client=yt_client,
            src=ypath_join(archive_root, 'yt_tables'),
            dst=ypath_join(destination_ypath_prefix, 'yt_tables'),
            exclude=exclude_tables
        )
        if dynamic_tables:
            from sandbox.projects.yabs.qa.bases.save_input import convert_tables_to_dynamic, get_tables_state_map

            logger.info('Converting tables to dynamic: %s', map(str, dynamic_tables))
            convert_tables_to_dynamic(dynamic_tables, yt_client)

            dst_dynamic_tables_state_map = get_tables_state_map(yt_client, dynamic_tables)
            unmounted_tables = list(filter(
                lambda table_: dst_dynamic_tables_state_map[table_] == 'unmounted',
                dst_dynamic_tables_state_map
            ))

            if unmounted_tables:
                logger.info('Mounting tables: %s', map(str, unmounted_tables))
                mount_tables_batched(unmounted_tables, yt_client)

    def execute_oneshot(self, yt_token, archive_root, oneshot_path=None, oneshot_args=None, oneshots_package=None, oneshot_tables=(), shallow_data_copy=False):
        from yt.wrapper import YtClient, ypath_join

        yt_client = YtClient(proxy=yt_bases.YT_PROXY, token=yt_token)

        logger.info('Shallow copy %s', 'enabled' if shallow_data_copy else 'disabled')

        destination_ypath_prefix = ypath_join(ONESHOT_ARCHIVE_ROOT, str(self.id))
        with self.memoize_stage.prepare_node_for_yt_oneshot(commit_on_entrance=False):
            logger.info('Create node: %s', destination_ypath_prefix)
            yt_client.create('map_node', destination_ypath_prefix, force=True)
            destination_node_ttl = datetime.timedelta(hours=12)
            logger.info('Set TTL=%s for node "%s"', str(destination_node_ttl), destination_ypath_prefix)
            set_yt_node_ttl(destination_ypath_prefix, destination_node_ttl.total_seconds(), yt_client)

        if shallow_data_copy:
            with self.memoize_stage.prepare_data_for_yt_oneshot(commit_on_entrance=False):
                if not oneshot_tables:
                    raise TaskFailure('Empty oneshot tables list')
                logging.info('Oneshot tables: %s', oneshot_tables)
                self.run_shallow_data_copy(yt_client, archive_root, oneshot_tables, destination_ypath_prefix)
        else:
            with self.memoize_stage.prepare_data_for_yt_oneshot(commit_on_entrance=False):
                logger.info('Copy data for oneshot from "%s" to "%s"', archive_root, destination_ypath_prefix)
                yt_copy(
                    yt_client=yt_client,
                    src_path=archive_root,
                    dst_path=destination_ypath_prefix
                )

        # logger.info('Execute development oneshot')
        # self.execute_development_yt_oneshot(ypath_join(destination_ypath_prefix, 'yt_tables'))
        # logger.info('Development oneshot executed successfully')

        logger.info('Execute oneshot')
        self.execute_yt_oneshot(oneshot_path, oneshot_args, oneshots_package, ypath_join(destination_ypath_prefix, 'yt_tables'))
        logger.info('Oneshot executed successfully')

        patched_input_spec = substitute_ypath_prefix_in_input_spec(
            input_spec=self.ctx['updated_input_spec'],
            ypath_prefix=archive_root,
            new_ypath_prefix=destination_ypath_prefix)

        self.ctx['updated_input_spec'].update(patched_input_spec)

    def get_yt_oneshots_package(self):
        return YT_ONESHOTS_PACKAGE.find(attrs=self.global_key).first()

    @trace_calls
    def save_input(self, yt_client, bs_release_yt_dir, yt_token, archive_root):
        from sandbox.projects.yabs.qa.bases.save_input import get_key_and_table_paths, _build_table_key_spec, get_tables_destination, _get_destination_path
        if self.ctx.get(UseSaveInputFromCS.name):
            logger.info('Run "save input" cs action')
            save_input_result_filepath = yt_bases.run_save_input(
                yt_token=yt_token,
                bs_release_yt_dir=bs_release_yt_dir,
                archive_root=archive_root,
                base_tags=self.base_tags,
                settings_spec=self.cs_settings,
            )
            logger.info('Saved "save input" result to %s', save_input_result_filepath)
            with open(save_input_result_filepath) as f:
                self.ctx['updated_input_spec'] = json.load(f)
        else:
            logger.info('Run save_input from shm')
            cs_import_info = yt_bases.get_cs_import_info(bs_release_yt_dir, self.cs_settings)
            tables_to_save = []
            tables_destination = get_tables_destination(archive_root)
            key_paths = {}
            run_save_input = False
            for importer, importer_settings in cs_import_info.items():
                if self.ctx.get(SaveAllInputs.name) or importer in self.ctx.get('importers', []):
                    importer_key_paths, table_paths, _ = get_key_and_table_paths(importer_settings['tables'])
                    key_paths.update(importer_key_paths)
                    tables_to_save.extend(importer_settings['tables'])
                    for path in table_paths:
                        if not yt_client.exists(_get_destination_path(tables_destination, path)):
                            run_save_input = True
                            break
            logger.info('Will save %d tables: %s', len(tables_to_save), tables_to_save)
            baseno_list = sorted(generate_basenos(
                stats=(1, 2, 5,),
                stats_in_clusters=(12,),
            ))

            if run_save_input:
                with self.memoize_stage.run_save_input_task(commit_on_entrance=False):
                    task_class = sdk2.Task['YABS_SERVER_SAVE_INPUT']
                    save_input_task = task_class(
                        task_class.current,
                        kill_timeout=18000,
                        owner=self.owner,
                        priority=self.priority,
                        description='Run save_input for task {}'.format(get_task_html_hyperlink(self.id)),
                        tables_to_save=tables_to_save,
                        filter_tables_by_orderid=get_or_default(self.ctx, FilterInputArchiveTablesByOrderID),
                        input_spec=self.ctx['input_spec'],
                        dynamic_bundle_name='yabs-cs-sandbox-dynamic',
                        no_quota_attrs=True,
                        recreate_links=True,
                        tables_filtred_by_orderid=TABLES_TO_BE_FILTERED_BY_ORDER_ID,
                        baseno_list=baseno_list,  # previous was: self.ctx['baseno_list'],
                        bs_release_yt_resource=self.ctx['bs_release_yt_resource'],
                        settings_archive=self.ctx['settings_archive'],
                        yt_pool=str(get_or_default(self.ctx, SamplingYtPool)),
                    ).save()
                    if self.ctx.get('tasks_archive_resource'):
                        save_input_task.Requirements.tasks_resource = self.ctx['tasks_archive_resource']
                    enqueue_task(save_input_task)
                    logger.info('Run task YabsServerSaveInput #{}'.format(save_input_task.id))
                    self.set_info('Run task YabsServerSaveInput {}'.format(get_task_html_hyperlink(save_input_task.id)), do_escape=False)
                    self.ctx['save_input_task_id'] = save_input_task.id

                check_tasks(self, self.ctx['save_input_task_id'])

                spec = sdk2.Task[self.ctx['save_input_task_id']].Context.updated_input_spec
                logger.debug('Got save_input result spec: %s', spec)
            else:
                logger.info('All tables exist')
                spec = _build_table_key_spec(key_paths, tables_destination)
                logger.debug('Built spec: %s', spec)

            self.ctx['updated_input_spec'] = spec

    @trace_calls
    def sample_tables(self, yt_client, yt_client_lock, yt_client_eventlog, yql_client, archive_root):
        from yt.wrapper import ypath_join
        from sandbox.projects.yabs.qa.bases.sample_tables.run import run_sample_tables
        from sandbox.projects.yabs.qa.bases.sample_tables.tables import switch_symlinks, clone_archive_with_symlinks, get_tables_for_sampling

        sampling_query_template = self.read_text_resource(get_or_default(self.ctx, SamplingQueryTemplateResourceParameter))
        if self.ctx.get(FindTablesForSampling.name):
            importers_info = yt_bases.get_cs_import_info(self.get_yabscs(), self.cs_settings)
            source_tables, need_mount_tables = get_importers_source_tables(importers_info)
            sampling_tables_black_list = self.read_json_resource(self.ctx.get(SamplingTablesBlackListResourceParameter.name))
            sampling_tables = get_tables_for_sampling(
                yt_client,
                archive_root,
                get_or_default(self.ctx, MinRowsCountForSamplingTables),
                source_tables,
                get_or_default(self.ctx, KeysForSamplingTables),
                sampling_tables_black_list,
                need_mount_tables)
        else:
            sampling_tables = {}
            sampling_tables_black_list = []
            need_mount_tables = set()

        sampling_tables.update(self.read_json_resource(get_or_default(self.ctx, SamplingTablesResourceParameter)))

        logger.info("Final sampling tables config: %s", sampling_tables)
        sampling_hash = self.sampling_parameters_hash
        new_archive_root = ypath_join(archive_root, 'shallow_copy', sampling_hash)
        shallow_copy_path = ypath_join(new_archive_root, 'yt_tables')
        sampled_tables_path = ypath_join(archive_root, 'sampled_tables', sampling_hash, 'yt_tables')

        clone_archive_with_symlinks(
            ypath_join(archive_root, 'yt_tables'),
            ypath_join(archive_root, 'yt_tables'),
            shallow_copy_path,
            yt_client,
        )

        sampling_is_successful, did_sampling, sampling_winning_task_id, sampled_table_configs = run_sample_tables(
            archive_root,
            shallow_copy_path,
            sampled_tables_path,
            sampling_query_template,
            sampling_tables,
            yt_client=yt_client,
            yt_client_lock=yt_client_lock,
            yt_client_eventlog=yt_client_eventlog,
            yt_pool=str(get_or_default(self.ctx, SamplingYtPool)),
            yql_client=yql_client,
            yt_cluster='hahn',
            task_id=self.id,
            dry_run=False,
            need_mount_tables=need_mount_tables,
        )

        if did_sampling:
            self.set_info('Tables were sampled')

        if not sampling_is_successful:
            logger.info('Wait for sampling task #{}'.format(sampling_winning_task_id))
            check_tasks(self, int(sampling_winning_task_id))

        switch_symlinks(sampled_table_configs, yt_client)

        patched_input_spec = substitute_ypath_prefix_in_input_spec(
            input_spec=self.ctx['updated_input_spec'],
            ypath_prefix=archive_root,
            new_ypath_prefix=new_archive_root)

        self.ctx['updated_input_spec'].update(patched_input_spec)

        return new_archive_root

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        from yt.wrapper import YtClient

        if self.ctx["require_mysql"]:
            with trace('require_mysql'):
                with trace('MySQL-python'):
                    logger.debug("Install MySQL-python")
                    PipEnvironment('MySQL-python', '1.2.5', use_wheel=True).prepare()

                logger.debug("Prepare MySQL data")
                self._prepare_data()

        yt_token = self.get_jailed_yt_token()
        yt_client = YtClient(proxy=yt_bases.YT_PROXY, token=yt_token)
        bs_release_yt_dir = self.get_yabscs()

        input_spec = self.read_json_resource(self.input_spec_res_id)
        archive_root = input_spec[yt_bases.AUXILLARY_DATA_IN_SPEC_KEY][yt_bases.ARCHIVE_ROOT_KEY]
        logging.info('Using archive_root %s', archive_root)

        if not self.ctx.get('updated_input_spec'):
            self.save_input(yt_client, bs_release_yt_dir, self.get_yt_token(), archive_root)

        if get_or_default(self.ctx, SamplingStrategyParameter) == SamplingStrategy.sampled.value:
            from yql.api.v1.client import YqlClient

            # yql_client and yt_client_eventlog have access to event log
            yql_token = self.get_yql_token()
            yql_client = YqlClient(token=yql_token)
            yt_client_eventlog = YtClient(proxy=yt_bases.YT_PROXY, token=yql_token)

            # Separate yt client is needed because alter_table cannot be run in transaction
            yt_client_lock = YtClient(proxy=yt_bases.YT_PROXY, token=yt_token)

            new_archive_root = self.sample_tables(
                yt_client=yt_client,
                yt_client_lock=yt_client_lock,
                yt_client_eventlog=yt_client_eventlog,
                yql_client=yql_client,
                archive_root=archive_root,
            )
            logging.info('Using new archive_root %s', new_archive_root)
            archive_root = new_archive_root

        if self.oneshot_path or self.exec_common_yt_oneshots:
            oneshots_package = self.get_yt_oneshots_package() if self.exec_common_yt_oneshots else None

            shallow_data_copy = get_or_default(self.ctx, ShallowDataCopy)
            oneshot_tables = self.get_oneshot_tables(oneshot_path=self.oneshot_path, oneshots_package=oneshots_package) if shallow_data_copy else []
            self.ctx['oneshot_tables'] = oneshot_tables
            self.execute_oneshot(
                yt_token,
                archive_root,
                oneshot_path=self.oneshot_path,
                oneshot_args=self.oneshot_args,
                oneshots_package=oneshots_package,
                oneshot_tables=oneshot_tables,
                shallow_data_copy=shallow_data_copy,
            )

        logs_folder = sandbox.sandboxsdk.paths.get_logs_folder()
        updated_input_spec_path = os.path.abspath(os.path.join(logs_folder, 'updated_input_spec.json'))
        with open(updated_input_spec_path, 'w') as f:
            json.dump(self.ctx['updated_input_spec'], f, indent=2)
        logger.info('Save updated input_spec to %s', updated_input_spec_path)

        destination_prefix = self.run_import(updated_input_spec_path, yt_token)
        self.ctx[IMPORT_PREFIX_KEY] = destination_prefix

    def run_import(self, input_spec_path, yt_token):
        self.report_operations()

        bs_release_yt_dir = self.get_yabscs()

        cs_import_info = yt_bases.get_cs_import_info(bs_release_yt_dir, self.cs_settings)
        missing_keys = get_input_spec_missing_keys(cs_import_info, input_spec_path)
        if missing_keys:
            self.set_info('Keys {} not in import spec'.format(', '.join(missing_keys)))

        try:
            cs_import_ver = get_cs_import_ver(bs_release_yt_dir)
        except Exception as e:
            logger.exception(e)
            cs_import_ver = None
        logger.info('cs_import_ver: %s', cs_import_ver)

        use_cs_cycle = self.ctx.get(UseCsCycle.name)
        if not use_cs_cycle:
            actions_str = get_or_default(self.ctx, ImportActions) or ''
            actions = actions_str.split() or cs_import_info.keys()

        logger.info('Run import')
        run_import_kwargs = dict(
            import_destination_path=self.ctx['import_destination_path'],
            yt_token=yt_token,
            bs_release_yt_dir=bs_release_yt_dir,
            input_spec_path=input_spec_path,
            date=self.get_db_date(),
            base_tags=self.base_tags,
            settings_spec=self.cs_settings,
            write_yt_debug_log=self.ctx.get(WriteYtDebugLog.name),
            task_id=self.id,
            prepare_mysql=self.ctx["require_mysql"],
            yt_pool=get_or_default(self.ctx, YtPool),
        )
        if use_cs_cycle:
            run_import_kwargs.update(
                log_path=self.log_path(),
                importers=self.ctx.get('importers', None)
            )
            run_import = yt_bases.run_cs_cycle_import
        else:
            run_import = yt_bases.run_cs_import
            run_import_kwargs['actions'] = actions
        return run_import(
            **run_import_kwargs
        )

    @trace_calls
    def _prepare_data(self):
        """Sync tables, install mysqld"""
        table_providers, _ = self.process_archive_contents([], run_switcher=False, run_cs_import=True)
        if self.ctx.get('install_yabs_mysql', True):
            logging.debug('Started installing YABS mysql')
            start_time = time.time()
            mysql_instances = frozenset(tp.inst for tp in table_providers)
            install_yabs_mysql(switcher=False, mysql_instances=mysql_instances)
            logging.debug('Finished installing YABS mysql. Time elapsed: {: 0.3f}', time.time() - start_time)
        else:
            logging.debug('Starting YABS mysql')
            try:
                run_process(['/etc/init.d/mysql.yabs', 'yabs', 'start'], log_prefix='mysql.yabs_start')
            except Exception as err:
                raise RuntimeError(err)
            logging.debug('Started YABS mysql')

        synpack_tables(table_providers)

    def execute_yt_oneshot(self, oneshot_path, oneshot_args, oneshots_package, ypath_prefix):
        with self.memoize_stage.run_execute_yt_oneshot_task(commit_on_entrance=False):
            task_class = sdk2.Task['EXECUTE_YT_ONESHOT']
            execute_yt_oneshot_task = task_class(
                task_class.current,
                owner=self.owner,
                priority=self.priority,
                description='Execute YT oneshot "{}"'.format(oneshot_path),
                oneshot_path=oneshot_path,
                oneshot_args=oneshot_args,
                oneshots_package=oneshots_package,
                run_in_test_mode=False,
                ypath_prefix=ypath_prefix,
                yt_proxy=ONESHOT_YT_PROXY,
                allow_access_to_production_data=False,
                force_data_flush=True,
            )
            enqueue_task(execute_yt_oneshot_task)
            logger.info('Run task ExecuteYTOneshot #{}'.format(execute_yt_oneshot_task.id))
            self.set_info('Run task {}'.format(get_task_html_hyperlink(execute_yt_oneshot_task.id)), do_escape=False)
            self.ctx['execute_yt_oneshot_task_id'] = execute_yt_oneshot_task.id

        check_tasks(self, self.ctx['execute_yt_oneshot_task_id'])

    @trace_calls
    def run_get_oneshot_tables(self, oneshot_path, oneshots_package):
        def create_description():
            description = 'Get tables from YT'
            if oneshot_path:
                description += ' oneshot: {}'.format(oneshot_path)
            if oneshots_package:
                description += ' oneshots_package: {}'.format(oneshots_package.id)
            return description

        task_class = sdk2.Task['EXECUTE_YT_ONESHOT']
        execute_yt_oneshot_task = task_class(
            task_class.current,
            owner=self.owner,
            priority=self.priority,
            description=create_description(),
            oneshot_path=oneshot_path,
            oneshots_package=oneshots_package,
            print_tables_only=True,
        )
        enqueue_task(execute_yt_oneshot_task)
        return execute_yt_oneshot_task.id

    @trace_calls
    def get_oneshot_tables(self, oneshot_path=None, oneshots_package=None):
        with self.memoize_stage.run_get_oneshot_tables(commit_on_entrance=False):
            task_id = self.run_get_oneshot_tables(oneshot_path, oneshots_package)
            logging.info('Run task ExecuteYTOneshot to get oneshot tables: #%s', str(task_id))
            self.ctx['get_oneshot_tables_task_id'] = task_id

        check_tasks(self, self.ctx['get_oneshot_tables_task_id'])

        return sdk2.Task[self.ctx['get_oneshot_tables_task_id']].Parameters.oneshot_tables

    def execute_development_yt_oneshot(self, ypath_prefix):
        development_oneshot_resource_id = get_or_default(self.ctx, YabsDevelopmentYtOneshotResource)
        if development_oneshot_resource_id:
            development_oneshot_resource = sdk2.Resource[development_oneshot_resource_id]
            with self.memoize_stage.run_execute_development_yt_oneshot_task(commit_on_entrance=False):
                task_class = sdk2.Task['EXECUTE_YT_ONESHOT']
                execute_development_yt_oneshot_task = task_class(
                    task_class.current,
                    owner=self.owner,
                    priority=self.priority,
                    description='Execute YABS development YT oneshot',
                    oneshot_path='<fake_path>',
                    run_in_test_mode=False,
                    ypath_prefix=ypath_prefix,
                    yt_proxy=ONESHOT_YT_PROXY,
                    allow_access_to_production_data=False,
                    force_data_flush=True,
                    oneshot_binary_resource=development_oneshot_resource
                )
                enqueue_task(execute_development_yt_oneshot_task)
                logger.info('Run task ExecuteYTOneshot #{}'.format(execute_development_yt_oneshot_task.id))
                self.set_info('Run task ExecuteYTOneshot (development YABS oneshot) {}'.format(get_task_html_hyperlink(execute_development_yt_oneshot_task.id)), do_escape=False)
                self.ctx['execute_development_yt_oneshot_task_id'] = execute_development_yt_oneshot_task.id

            check_tasks(self, self.ctx['execute_development_yt_oneshot_task_id'])

    @property
    def base_tags(self):
        """Should be callable from on_enqueue"""
        try:
            self.ctx['base_tags']
        except KeyError:
            tags = []

            tags += get_or_default(self.ctx, BinDBList).split()
            self.ctx['base_tags'] = list(set(tags))

        return self.ctx['base_tags']

    @property
    def oneshot_path(self):
        return self.ctx.get(OneShotPath.name)

    @property
    def oneshot_args(self):
        return self.ctx.get(OneShotArgs.name)

    def get_jailed_yt_token(self):
        """Get YT token without access to production data"""
        return self.get_vault_data('robot-yabs-cs-sbjail', 'yabscs_jailed_yt_token')

    def get_yql_token(self):
        return self.get_vault_data('robot-yabs-cs-sb', 'yql_token')


def substitute_ypath_prefix_in_input_spec(input_spec, ypath_prefix, new_ypath_prefix):
    patched_input_spec = copy.deepcopy(input_spec)

    for key, path in input_spec.iteritems():
        if isinstance(path, basestring) and path.startswith(ypath_prefix):
            relative_path = os.path.relpath(path, ypath_prefix)
            updated_path = os.path.join(new_ypath_prefix, relative_path)
            patched_input_spec[key] = updated_path
            logger.debug('%s: change "%s" to "%s"', key, path, updated_path)

    return patched_input_spec


class InvalidInputSpec(Exception):
    pass


__Task__ = YabsServerRealRunCSImport
