import json
import logging
import time
from collections import defaultdict

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_SAAS_FREEZE_DATA
from sandbox.projects.yabs.qa.tasks.YabsServerFreezeSaasSnapshots import SNAPSHOTS_TABLE_PROXY, SNAPSHOTS_TABLE_PATH

logger = logging.getLogger(__name__)


TESTENV_RESOURCE_STATUSES = ('OK', 'OK_OLD', 'NOT_CHECKED', 'CHECKING', 'CHECKED', 'SWITHCING', )


def get_testenv_resources(database_name='yabs-2.0'):
    filtered_resources = defaultdict(list)
    testenv_resources = TEClient.get_resources(database_name)
    for testenv_resource in testenv_resources:
        if testenv_resource['name'] == 'YABS_SERVER_SAAS_FREEZE_DATA' and testenv_resource['status'] in TESTENV_RESOURCE_STATUSES:
            filtered_resources[testenv_resource['status']].append(testenv_resource)

    resources_to_ping = []
    for status, resources in filtered_resources.items():
        resource = max(resources, key=lambda x: x['resource_timestamp'])
        logger.debug('Will ping %s %s #%s', status, resource['sandbox_name'], resource['resource_id'])
        resources_to_ping.append(resource['resource_id'])

    logger.debug('Got resources: %s', json.dumps(resources_to_ping, indent=2))
    return resources_to_ping


def ping_saas_freeze_data(yt_client, resource_ids):
    snapshots_to_ping = []
    for resource_id in resource_ids:
        freeze_data_resource = sdk2.ResourceData(YABS_SERVER_SAAS_FREEZE_DATA[resource_id])
        with open(str(freeze_data_resource.path), 'r') as freeze_data_file:
            freeze_data = json.load(freeze_data_file)
        snapshots_to_ping.extend([{"stream": snapshot['stream'], "id": snapshot['id']} for snapshot in freeze_data])

    logger.debug('Will ping snapshots: %s', json.dumps(snapshots_to_ping, indent=2))
    existing_snapshots = list(yt_client.lookup_rows(SNAPSHOTS_TABLE_PATH, snapshots_to_ping, format='json'))
    logger.debug('Found snapshots: %s', json.dumps(existing_snapshots, indent=2))
    now = int(time.time())
    for snapshot in existing_snapshots:
        snapshot['accessed'] = now
    logger.debug('Will insert snapshots: %s', json.dumps(existing_snapshots, indent=2))
    yt_client.insert_rows(SNAPSHOTS_TABLE_PATH, existing_snapshots, format='json')


class YabsServerResourcePinger(sdk2.Task):

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group('Auth') as auth_params:
            yav_secret_id = sdk2.parameters.String("YaV secret id", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

    class Requirements(sdk2.Requirements):
        ram = 4 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

        environments = (
            environments.PipEnvironment('yandex-yt', use_wheel=True),
        )

    def on_execute(self):
        resources_to_ping = get_testenv_resources()

        from yt.wrapper import YtClient
        secret = sdk2.yav.Secret(self.Parameters.yav_secret_id)
        yt_token = secret.data()['yt_token']
        yt_client = YtClient(proxy=SNAPSHOTS_TABLE_PROXY, token=yt_token)

        ping_saas_freeze_data(yt_client, resources_to_ping)
