import functools
import logging
import sys
import traceback

from sandbox import sdk2
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.common.types.misc import NotExists
from sandbox.projects.common.build.parameters import ArcadiaPatch
from sandbox.projects.yabs.qa.brave_tests.common import TestType
from sandbox.projects.yabs.qa.tasks.YabsServerFindBraveTestsBaseRevision import OutputParameters as FindBraveTestsBaseRevisionOutputParameters
from sandbox.projects.yabs.qa.tasks.YabsServerRunBraveTests.stages import (
    build_yabs_server_release,
    build_yabs_server_sanitize_address,
    build_yabs_server_sanitize_memory,
    generate_ft_report,
    generate_meta_performance_report,
    generate_sanitize_report,
    generate_stat_performance_report,
    run_ft_shoot_cmp_tasks,
    run_ft_meta_shoot_cmp_tasks,
    run_ft_shoot_tasks,
    run_ft_stability_tasks,
    run_ft_meta_shoot_tasks,
    run_sanitize_address_shoot_tasks,
    run_sanitize_memory_shoot_tasks,
    run_meta_performance_shoot_cmp_tasks,
    run_meta_performance_shoot_tasks,
    run_stat_performance_shoot_cmp_tasks,
    run_stat_performance_shoot_tasks,
)
from sandbox.projects.yabs.qa.utils.arcanum import comment_review as comment_arcanum_review
from sandbox.projects.yabs.qa.utils.general import html_hyperlink, get_task_html_hyperlink


logger = logging.getLogger(__name__)
_STAGES = (
    build_yabs_server_release,
    build_yabs_server_sanitize_address,
    build_yabs_server_sanitize_memory,
    generate_ft_report,
    generate_meta_performance_report,
    generate_sanitize_report,
    generate_stat_performance_report,
    run_ft_shoot_cmp_tasks,
    run_ft_meta_shoot_cmp_tasks,
    run_ft_shoot_tasks,
    run_ft_stability_tasks,
    run_ft_meta_shoot_tasks,
    run_sanitize_address_shoot_tasks,
    run_sanitize_memory_shoot_tasks,
    run_meta_performance_shoot_cmp_tasks,
    run_meta_performance_shoot_tasks,
    run_stat_performance_shoot_cmp_tasks,
    run_stat_performance_shoot_tasks,
)


def suppress(*exceptions):
    def decorator(f):
        @functools.wraps(f)
        def wrapper(self, *args, **kwags):
            try:
                return f(self, *args, **kwags)
            except exceptions:
                ex_type, ex, tb = sys.exc_info()
                exception_lines = traceback.format_exception(ex_type, ex, tb)
                self.set_info("".join(exception_lines).strip())

                self.Context.has_diff = True
        return wrapper
    return decorator


class YabsServerRunBraveTests(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        max_restarts = 0
        push_tasks_resource = True

        tokens = sdk2.parameters.YavSecret("YAV secret identifier", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

        test_type = sdk2.parameters.String(
            "Test type",
            choices=[(test_type.value, test_type.value) for test_type in TestType],
            default=TestType.FT.value)
        arcadia_patch = ArcadiaPatch()
        base_revision_parameters = FindBraveTestsBaseRevisionOutputParameters()

        with sdk2.parameters.Group("Developer options") as developer_group:
            debug_mode = sdk2.parameters.Bool('Debug mode', default=False)
            with debug_mode.value[True]:
                reusable_stages = sdk2.parameters.CheckGroup(
                    'Reusable stages',
                    choices=[
                        (_stage.__name__, _stage.__name__)
                        for _stage in _STAGES
                    ],
                    default=None,
                )

    def add_comment_to_arcanum_review(self):
        arcanum_review_id = self.Context.arcanum_review_id
        if arcanum_review_id is NotExists:
            return

        arcanum_token = self.Parameters.tokens.data()["arc_token"]
        comment = "{test_type} brave test task {task_link}".format(
            test_type=self.Parameters.test_type.upper(),
            task_link=get_task_html_hyperlink(self.id),
        )
        comment_arcanum_review(
            arcanum_token=arcanum_token,
            review_id=arcanum_review_id,
            comment=comment,
        )

        arcanum_review_link = "https://a.yandex-team.ru/review/{}/details".format(arcanum_review_id)
        arcanum_review_hyperlink = html_hyperlink(arcanum_review_link, arcanum_review_id)
        self.set_info("Posted comment to arcanum review {}".format(arcanum_review_hyperlink), do_escape=False)

    @suppress(TaskError, TaskFailure)
    def on_execute(self):
        if self.Parameters.debug_mode:
            self.Context.__reusable_stages = self.Parameters.reusable_stages

        with self.memoize_stage.comment_arcanum_review(commit_on_entrance=False):
            try:
                self.add_comment_to_arcanum_review()
            except Exception as e:
                logger.exception(e)
                self.set_info("Failed to post comment to arcanum review")

        baseline_shoot_task_ids = {
            "bs": self.Parameters.bs_task_id,
            "yabs": self.Parameters.yabs_task_id,
            "bsrank": self.Parameters.bsrank_task_id,
        }

        if self.Parameters.test_type in [TestType.FT.value, TestType.FT_SAAS.value]:
            build_yabs_server_release(
                self,
                base_revision=self.Parameters.base_revision,
                arcadia_patch=self.Parameters.arcadia_patch,
            )

        if self.Parameters.test_type == TestType.FT.value:
            meta_baseline_shoot_task_ids = {
                "bs": self.Parameters.meta_bs_task_id,
                "yabs": self.Parameters.meta_yabs_task_id,
                "bsrank": self.Parameters.meta_bsrank_task_id,
            }

            run_ft_shoot_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )
            run_ft_meta_shoot_tasks(
                self,
                baseline_shoot_task_ids=meta_baseline_shoot_task_ids,
            )
            run_ft_stability_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )

            run_ft_shoot_cmp_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
                arcanum_review_id=0 if self.Context.arcanum_review_id is NotExists else self.Context.arcanum_review_id,
            )
            run_ft_meta_shoot_cmp_tasks(
                self,
                baseline_shoot_task_ids=meta_baseline_shoot_task_ids,
                arcanum_review_id=0 if self.Context.arcanum_review_id is NotExists else self.Context.arcanum_review_id,
            )
            report_html, has_diff = generate_ft_report(self)

        if self.Parameters.test_type == TestType.PERFORMANCE.value:
            build_yabs_server_release(
                self,
                base_revision=self.Parameters.base_revision,
                arcadia_patch=self.Parameters.arcadia_patch,
            )

            run_stat_performance_shoot_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )
            run_stat_performance_shoot_cmp_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )
            report_html, has_diff = generate_stat_performance_report(self)

        if self.Parameters.test_type == TestType.PERFORMANCE_META.value:
            build_yabs_server_release(
                self,
                base_revision=self.Parameters.base_revision,
                arcadia_patch=self.Parameters.arcadia_patch,
            )

            run_meta_performance_shoot_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )
            run_meta_performance_shoot_cmp_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )
            report_html, has_diff = generate_meta_performance_report(self)

        if self.Parameters.test_type == TestType.SANITIZE.value:
            build_yabs_server_sanitize_address(
                self,
                base_revision=self.Parameters.base_revision,
                arcadia_patch=self.Parameters.arcadia_patch,
            )
            build_yabs_server_sanitize_memory(
                self,
                base_revision=self.Parameters.base_revision,
                arcadia_patch=self.Parameters.arcadia_patch,
            )

            run_sanitize_address_shoot_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )

            run_sanitize_memory_shoot_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )
            report_html, has_diff = generate_sanitize_report(self)

        if self.Parameters.test_type == TestType.FT_SAAS.value:
            run_ft_shoot_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )
            run_ft_stability_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
            )

            run_ft_shoot_cmp_tasks(
                self,
                baseline_shoot_task_ids=baseline_shoot_task_ids,
                arcanum_review_id=0 if self.Context.arcanum_review_id is NotExists else self.Context.arcanum_review_id,
            )
            report_html, has_diff = generate_ft_report(self)

        self.set_info(report_html, do_escape=False)
        self.Context.has_diff = has_diff
