import itertools
import logging
import re

import sandbox.projects.yabs.qa.utils.importer as importer_utils
from sandbox.projects.yabs.qa.utils.general import get_json_md5

from sandbox.projects.yabs.sandbox_task_tracing import trace_calls


logger = logging.getLogger(__name__)


def get_importers_settings_version(bases, importers_info, mkdb_info, cs_settings_version):
    """Calculate the checksum of settings of importers that the database depends on

    :param bases: binary bases
    :type bases: list
    :param importers_info: importers info, output of 'cs import --print-info'
    :type importers_info: dict
    :param cs_settings_version: content system settings version, output of 'cs import --settings-version'
    :type cs_settings_version: dict
    :return: importers with settings versions
    :rtype: dict
    """
    logger.debug('Bases: %s', bases)
    importers = importer_utils.get_bases_importers_with_dependencies(bases, importers_info, mkdb_info)
    logger.debug('Importers: %s', importers)

    settings_version = {
        importer: cs_settings_version[importer]
        for importer in importers
    }

    logger.debug('Importers settings version: %s', settings_version)

    return settings_version


def get_importers_code_version(bases, importers_info, mkdb_info, importer_code_version):
    """Calculate the checksum of code of importers that the database depends on

    :param bases: binary bases
    :type bases: list
    :param importers_info: importers info, output of 'cs import --print-info'
    :type importers_info: dict
    :param importer_code_version: content system importers code version, output of 'dbtool importer_code_ver'
    :type importer_code_version: dict
    :return: importers with code versions
    :rtype: dict
    """
    logger.debug('Bases: %s', bases)
    importers = importer_utils.get_bases_importers_with_dependencies(bases, importers_info, mkdb_info)
    logger.debug('Importers: %s', importers)

    code_version = {
        importer: importer_code_version[importer]
        for importer in importers
    }

    logger.debug('Importers code version: %s', code_version)

    return code_version


def separate_base_tags(base_tags):
    separated_tags = set()
    for tag in base_tags:
        match = re.match(r'(yabs|bs)_(st\d+|lmng_\d+|lmcounters_\d+|dbe)', tag)
        if match:
            separated_tags |= set(['yabs_' + match.group(2), 'bs_' + match.group(2)])
    return list(set(base_tags) | separated_tags)


def get_importers_after(importers, importers_info):
    importer_graph = importer_utils.build_importer_graph_from_importers_info(importers_info)
    importers_after = set(itertools.chain.from_iterable(
        importer_utils.get_importers_after(importer, importer_graph) for importer in importers))
    return importers_after


def is_st_update_mode(bases):
    return any(basename.startswith('st_update_') for basename in bases)


@trace_calls(save_arguments='all')
def get_importers_to_run(importers, existing_imports, importers_info):
    not_found_importers = set(importers) - set(existing_imports)
    importers_after_not_found = get_importers_after(not_found_importers, importers_info)
    return list(
        not_found_importers | (importers_after_not_found & set(importers))
    )


def get_importer_with_dependencies_version(importer, importers_info, versions):
    importers = importer_utils.get_importers_with_dependencies([importer], importers_info)
    importer_versions = {
        importer: versions.get(importer)
        for importer in importers
    }
    logger.debug("Importer %s with dependencies versions: %s", importer, importer_versions)
    for importer_name, importer_version in importer_versions.items():
        if not importer_version:
            raise importer_utils.CSDataError('Missing version for importer {}'.format(importer_name))
    return get_json_md5(importer_versions)


def import_node_is_reusable(oneshot_path, drop_import_result, tags):
    reuse_import_node_conditions = {
        "no oneshot": not oneshot_path,
        "drop_import_results is disabled": not drop_import_result,
        "not precommit check with patch": "WITH-PATCH" not in tags,
    }
    logger.debug("Conditions for allowing reuse of import node: %s", reuse_import_node_conditions)
    return all(reuse_import_node_conditions.values())
