import logging
import json
import time

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.yabs.qa.resource_types import (
    BS_RELEASE_YT,
    YABS_CS_INPUT_SPEC,
    YABS_CS_SETTINGS_ARCHIVE,
)
from sandbox.projects.yabs.qa.tasks.YabsServerRunCS import YabsServerRunCS, YabsServerRunCSStdOut
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.common.yabs.server.db import yt_bases
from sandbox.projects.yabs.qa.utils import yt_utils


logger = logging.getLogger(__name__)

SECONDS_IN_HOUR = 60 * 60
WORKDIR_ROOT = '//home/yabs-cs-sandbox/editpage-parse'
CS_ARGS = 'partner_interface editpage_parse'


def single(value):
    return [{'value': value}]


class YabsServerRunEditPageParse(sdk2.Task):
    name = 'YABS_SERVER_RUN_EDITPAGE_PARSE'
    description = 'Run EditPage parse'

    class Parameters(sdk2.Parameters):
        input_spec = sdk2.parameters.Resource(
            'Input spec for CS', resource_type=YABS_CS_INPUT_SPEC, required=True)
        bs_release_yt_resource = sdk2.parameters.Resource(
            'BS release yt resource', resource_type=BS_RELEASE_YT)
        settings_archive = sdk2.parameters.Resource(
            'CS settings', resource_type=YABS_CS_SETTINGS_ARCHIVE)
        yt_token_vault_name = sdk2.parameters.String(
            'Vault name for YT token', required=True, default='yabs-cs-sb-yt-token')
        yt_proxy = sdk2.parameters.String(
            'YT cluster', required=True, default='hahn')
        workdir_root = sdk2.parameters.String(
            'Root directory of workdirs', required=True, default=WORKDIR_ROOT)
        workdir_ttl = sdk2.parameters.Integer(
            'TTL for workdir in hours', required=True, default=24)
        mode = sdk2.parameters.RadioGroup(
            'EditPageParse mode',
            choices=(
                ('Copy replicas', 0),
                ('Mixed (copy protected from replicas)', 1),
                ('Standalone (use only LB stream)', 2),
            ),
            required=True, default=1)

        with sdk2.parameters.Output:
            workdir = sdk2.parameters.String('Workdir')

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024
        environments = (
            environments.PipEnvironment('yandex-yt', use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        workdir = None
        archive_root = None
        updated_settings_spec = None

    @property
    def workdir(self):
        if self.Context.workdir is None:
            self.Context.workdir = self._create_workdir()
            self.Context.save()
        return self.Context.workdir

    @property
    def archive_root(self):
        if self.Context.archive_root is None:
            spec_resource = sdk2.ResourceData(self.Parameters.input_spec)
            with open(str(spec_resource.path)) as f:
                input_spec = json.load(f)
                self.Context.archive_root = input_spec[yt_bases.AUXILLARY_DATA_IN_SPEC_KEY][yt_bases.ARCHIVE_ROOT_KEY]
        return self.Context.archive_root

    def _create_workdir(self):
        from yt.wrapper import YtClient, ypath_join

        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault_name)
        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)
        workdir = ypath_join(self.Parameters.workdir_root, str(self.id))

        logger.info('Create workdir "%s"', workdir)
        yt_client.create('map_node', workdir)

        ttl = self.Parameters.workdir_ttl * SECONDS_IN_HOUR
        logger.info('Set TTL=%s seconds for node "%s"', ttl, workdir)
        yt_utils.set_yt_node_ttl(workdir, ttl, yt_client)
        return workdir

    def get_input_spec_unixtime(self):
        dtt = self.Parameters.input_spec.created.timetuple()
        return int(time.mktime(dtt))

    def get_settings_spec(self, **extra):
        spec = {
            'cooldown': 0,
            'workdir': self.workdir,
            'now_unixtime': self.get_input_spec_unixtime(),
        }
        if self.Context.updated_settings_spec:
            spec.update(json.loads(self.Context.updated_settings_spec))

        spec.update(extra)
        settings = {}
        for k, v in spec.iteritems():
            settings[k] = single(v)
        return settings

    def get_cs_task_description(self, extra_settings):
        params = []
        for k, v in extra_settings.iteritems():
            params.append('{} {}'.format(k, v))
        return self.description + ': ' + ', '.join(params)

    def run_editpage_parse(self, **extra_settings):
        run_cs_task = YabsServerRunCS(
            self,
            description=self.get_cs_task_description(extra_settings),
            tool='cs',
            bs_release_yt_resource=self.Parameters.bs_release_yt_resource,
            settings_archive=self.Parameters.settings_archive,
            yt_proxy=self.Parameters.yt_proxy,
            args=CS_ARGS,
            settings_spec=self.get_settings_spec(**extra_settings),
            save_stdout='save_input' in extra_settings
        )
        run_cs_task.enqueue()
        return run_cs_task

    def on_execute(self):
        with self.memoize_stage.save_input(commit_on_entrance=False):
            task = self.run_editpage_parse(save_input=self.archive_root)
            self.Context.save_input_task_id = task.id
        check_tasks(self, [self.Context.save_input_task_id])

        with self.memoize_stage.update_settings_spec(commit_on_entrance=False):
            save_input_task = self.find(id=self.Context.save_input_task_id).first()
            cs_log_resource = YabsServerRunCSStdOut.find(task=save_input_task).first()
            cs_log_resource = sdk2.ResourceData(cs_log_resource)
            with open(str(cs_log_resource.path)) as f:
                spec = json.dumps(json.load(f), indent=2)
                logger.info('Updated settings spec: %s', spec)
                self.Context.updated_settings_spec = spec

        with self.memoize_stage.execute_editpage_parse(commit_on_entrance=False):
            task = self.run_editpage_parse(mode=self.Parameters.mode)
            self.Context.execute_editpage_parse_task_id = task.id
        check_tasks(self, [self.Context.execute_editpage_parse_task_id])

        self.Parameters.workdir = self.Context.workdir
