import logging

from sandbox import sdk2

from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.common.errors import TaskFailure
from sandbox.projects.yabs.qa.tasks.cs_helpers import CSHelper
from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink
from sandbox.projects.yabs.qa.resource_types import BS_RELEASE_YT, YABS_CS_SETTINGS_ARCHIVE, YABS_CS_INPUT_SPEC
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.qa.tasks.YabsServerRunCS import YabsServerRunCS
from sandbox.projects.yabs.qa.pipeline.stage import stage
from sandbox.common.types.misc import NotExists

import sandbox.common.types.task as ctt

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import enqueue_task


logger = logging.getLogger(__name__)

TABLES_TO_BE_FILTERED_BY_ORDER_ID = ['//home/bigb/caesar/stable/Banners', '//home/yabs/dict/MysqlBannerDontUse', '//home/bigb/caesar/stable/AdGroups']


class YabsServerSaveInput(sdk2.Task, CSHelper):
    description = 'Save production data for tests'

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        environments = (
            PipEnvironment('jsondiff', version='1.2.0'),
            PipEnvironment('yandex-yt', use_wheel=True),
            PipEnvironment('yql', use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 5 * 3600
        tables_to_save = sdk2.parameters.JSON(
            'List of tables to save from production',
            required=True,
            description="""tables - list of dicts: [
                {
                    'id': 'table1',
                    'path': '//home/Table1'
                },
                {
                    'id': 'table2',
                    'path': '//home',
                    'table': 'Table2',
                    'need_mount': False
                }
                ]"""
        )
        input_spec = sdk2.parameters.Resource("Input spec for cs_import", resource_type=YABS_CS_INPUT_SPEC, required=True)
        dynamic_bundle_name = sdk2.parameters.String('Dynamic bundle name', default='yabs-cs-sandbox-dynamic')
        no_quota_attrs = sdk2.parameters.Bool('No quota attributes', default=True)
        recreate_links = sdk2.parameters.Bool('Recreate table links', default=True)
        filter_tables_by_orderid = sdk2.parameters.Bool('Filter tables by OrderID', default=False, description='A table must contain OrderID field')
        with filter_tables_by_orderid.value[True]:
            tables_filtred_by_orderid = sdk2.parameters.List('Tables to be filtred by OrderID', required=True, default=[], description='Path in production')
            baseno_list = sdk2.parameters.List('BaseNo for filtering', required=True)
            bs_release_yt_resource = sdk2.parameters.Resource('BS release yt resource', resource_type=BS_RELEASE_YT, required=True)
            settings_archive = sdk2.parameters.Resource("cs settings", resource_type=YABS_CS_SETTINGS_ARCHIVE)
            yt_token_vault_name = sdk2.parameters.String('Vault name for YT token', required=True, default='yabs-cs-sb-yt-token')
            yt_proxy = sdk2.parameters.String('YT cluster', required=True, default='hahn')
            yt_pool = sdk2.parameters.String('YT pool for yql operation', required=True)
            yql_token_vault_name = sdk2.parameters.String('Vault name for YQL token', required=True, default='yql_token')

    def on_save(self):
        super(YabsServerSaveInput, self).on_save()
        if self.Parameters.input_spec:
            sem_name = "yabs_server_save_input_{}".format(self.Parameters.input_spec.id)
        else:
            sem_name = "yabs_server_save_input"

        if self.Parameters.owner != 'YABS_SERVER_SANDBOX_TESTS' and self.author in self.server.group['YABS_SERVER_SANDBOX_TESTS'].read()["members"]:
            self.Parameters.owner = 'YABS_SERVER_SANDBOX_TESTS'

        if self.Parameters.owner != 'YABS_SERVER_SANDBOX_TESTS':
            # SANDBOX-8829 SHMDUTY-93
            raise TaskFailure('{} is not a member of YABS_SERVER_SANDBOX_TESTS, cannot take auto semaphore'.format(self.author))

        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name=sem_name, weight=1, capacity=1),
            ],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        )

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        from yt.wrapper import YtClient
        from sandbox.projects.yabs.qa.bases.save_input import save_input

        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault_name)
        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)

        kwargs = dict(
            dynamic_bundle_name=self.Parameters.dynamic_bundle_name,
            no_quota_attrs=self.Parameters.no_quota_attrs,
            recreate_links=self.Parameters.recreate_links,
        )
        if self.Parameters.filter_tables_by_orderid:
            kwargs.update(
                tables_filtred_by_orderid=self.Parameters.tables_filtred_by_orderid,
                baseno_list=self.Parameters.baseno_list,
                baseno_by_orederid_table=self._get_baseno_by_orederid_table(yt_client),
                yt_proxy=self.Parameters.yt_proxy,
                yql_token=sdk2.Vault.data(self.owner, self.Parameters.yql_token_vault_name),
                yt_pool=self.Parameters.yt_pool,
            )

        spec = save_input(
            yt_client,
            self.Parameters.tables_to_save,
            self.archive_root_path,
            **kwargs
        )
        logger.debug('Got save_input result spec: %s', spec)
        self.Context.updated_input_spec = spec

    def _get_baseno_by_orederid_table(self, yt_client):
        from yt.wrapper import ypath_join
        from sandbox.projects.yabs.qa.bases.save_input import get_tables_destination

        tables_destination = get_tables_destination(self.archive_root_path)
        import_path = ypath_join(tables_destination, 'import')
        import_table = ypath_join(import_path, 'BaseNoByOrderId')

        if yt_client.exists(import_table):
            logger.info('Table %s already exists', import_table)
        else:
            settings_for_importer = {'destination_prefix': [{
                'value': import_path}]}
            self._run_baseno_by_orederid_import(yt=yt_client, tables_destination=tables_destination, settings_spec=settings_for_importer, import_table=import_table)

        if self.Context.subtask_run_cs_id is not NotExists:
            check_tasks(self, self.Context.subtask_run_cs_id)

        return import_table

    @stage(provides='subtask_run_cs_id')
    @trace_calls
    def _run_baseno_by_orederid_import(self, yt=None, tables_destination=None, settings_spec=None, import_table=None, kill_timeout=3600):
        logger.info('Run importer BaseNoByOrderId')
        subtask = enqueue_task(YabsServerRunCS(
            self,
            kill_timeout=kill_timeout if kill_timeout else self.Parameters.kill_timeout,
            description="Run import BaseNoByOrderId for task {}".format(get_task_html_hyperlink(self.id)),
            notifications=self.Parameters.notifications,
            tool='cs',
            args='import BaseNoByOrderId',
            bs_release_yt_resource=self.Parameters.bs_release_yt_resource,
            yt_token_vault_name=self.Parameters.yt_token_vault_name,
            yt_proxy=self.Parameters.yt_proxy,
            settings_spec=settings_spec,
            settings_archive=self.Parameters.settings_archive,
        ))
        logger.info('Run cs task #%d', subtask.id)
        self.set_info('Run cs export task {}'.format(get_task_html_hyperlink(subtask.id)), do_escape=False)
        return subtask.id
