import json
import logging
import time

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import ReleaseStatus, Semaphores

from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.common.yabs.server.db.yt_bases import YT_PROXY, DEFAULT_CS_INPUT_ARCHIVE_TTL
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.hamster.record import ENDPOINTS_TABLE_PROXY
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.hamster import get_active_hamster_endpoints, ping_active_hamster_endpoints, remove_expired_hamster_endpoints
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.saas import (
    SNAPSHOTS_TABLE_PROXY,
    SEMAPHORE_NAME,
)
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.kvrs_saas import (
    get_active_kvrs_saas_freeze_data,
    ping_active_kvrs_saas_freeze_data,
    remove_kvrs_frozen_states
)
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.cs_input_spec import get_active_cs_input_spec_resources, ping_active_cs_input_spec_resources
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.binary_bases import get_active_binary_base_resources, ping_active_binary_base_resources


logger = logging.getLogger(__name__)
DEFAULT_HAMSTER_TTL = 12 * 60 * 60
DEFAULT_SAAS_FREEZE_DATA_TTL = 12 * 60 * 60


class YabsServerShmResourceGC(BaseBinTask):
    """Checks active hamster services and removes unused ones. """

    class Parameters(BaseBinTask.Parameters):

        resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_type": "YABS_SERVER_SHM_RESOURCE_GC"})
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)

        with sdk2.parameters.Group('Ping settings') as ping_settings:
            do_ping = sdk2.parameters.Bool("Actually ping anything and make writing changes", default=True)
            do_freeze_saas_snapshots = sdk2.parameters.Bool('Freeze saas snapshots', default=False)
            ttl = sdk2.parameters.Integer('TTL in seconds (deprecated)', default=DEFAULT_HAMSTER_TTL)  # deprecated
            hamster_ttl = sdk2.parameters.Integer('Hamster TTL in seconds', default=DEFAULT_HAMSTER_TTL)
            saas_freeze_data_ttl = sdk2.parameters.Integer('Saas freeze data TTL in seconds', default=DEFAULT_SAAS_FREEZE_DATA_TTL)
            cs_input_spec_ttl = sdk2.parameters.Integer('CS input spec TTL in seconds', default=DEFAULT_CS_INPUT_ARCHIVE_TTL)

        with sdk2.parameters.Group('OK_OLD settings') as ok_old_settings:
            ok_old_resources_to_keep = sdk2.parameters.List(
                "What types of OK_OLD resources to keep (service tags, only one resource per type is kept)",
                default=['goalnet', 'rsya_hit_models_heavy_01'])
        with sdk2.parameters.Group('Auth') as auth_params:
            tokens = sdk2.parameters.YavSecret("YAV secret identifier", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

    class Requirements(sdk2.Requirements):
        ram = 4 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

        environments = (
            environments.PipEnvironment('yandex-yt', use_wheel=True),
        )

    def on_save(self):
        super(YabsServerShmResourceGC, self).on_save()
        if self.Parameters.do_freeze_saas_snapshots:
            self.Requirements.semaphores = Semaphores(
                acquires=[
                    Semaphores.Acquire(name=SEMAPHORE_NAME, weight=1),
                ],
            )

    def on_execute(self):
        from yt.wrapper import YtClient
        tokens = self.Parameters.tokens.data()
        yt_token = tokens['yt_token']

        yt_client_hamster_endpoints = YtClient(proxy=ENDPOINTS_TABLE_PROXY, token=yt_token)
        yt_client_saas_shapshots = yt_client_hamster_endpoints if ENDPOINTS_TABLE_PROXY == SNAPSHOTS_TABLE_PROXY else YtClient(proxy=SNAPSHOTS_TABLE_PROXY, token=yt_token)
        yt_client_cs_input_spec = YtClient(proxy=YT_PROXY, token=yt_token)

        testenv_resources = TEClient.get_resources('yabs-2.0')
        logger.debug('Got resources from testenv: %s', json.dumps(testenv_resources, indent=2))

        now = int(time.time())

        failures = []
        active_hamster_endpoints = {}
        hamster_endpoints_to_ping = {}
        try:
            active_hamster_endpoints = get_active_hamster_endpoints(
                testenv_resources, self.server,
                ok_old_resources_to_keep=self.Parameters.ok_old_resources_to_keep
            )
            logger.debug("Active hamster endpoints: %s", active_hamster_endpoints)
            for service_tag, resource_ids in active_hamster_endpoints.items():
                for resource_id in resource_ids:
                    with open(str(sdk2.ResourceData(sdk2.Resource[resource_id]).path), 'r') as resource_data:
                        hamster_endpoints_to_ping[(resource_id, service_tag)] = json.load(resource_data)
        except Exception:
            logger.error('Got exception while trying to determine hamster_endpoints to ping', exc_info=True)
            failures.append('determine_hamster_endpoints')
        if not self.Parameters.do_ping:
            self.set_info("Dry run, finishing.")
            return
        try:
            ping_active_hamster_endpoints(yt_client_hamster_endpoints, hamster_endpoints_to_ping, now, self.Parameters.ttl, insert=True)
        except Exception:
            logger.error('Got exception while trying to ping hamster_endpoints', exc_info=True)
            failures.append('hamster_endpoints')

        active_kvrs_saas_freeze_data_resources = get_active_kvrs_saas_freeze_data(testenv_resources, self.server)
        logger.info('Ping active KVRS SaaS topology resources %s', active_kvrs_saas_freeze_data_resources)
        try:
            ping_active_kvrs_saas_freeze_data(yt_client_saas_shapshots, active_kvrs_saas_freeze_data_resources, now)
        except Exception:
            logger.error('Got exception while trying to ping kvrs_saas_freeze_data', exc_info=True)
            failures.append('kvrs_saas_freeze_data')

        try:
            active_cs_input_spec_resources = get_active_cs_input_spec_resources(testenv_resources, self.server)
            ping_active_cs_input_spec_resources(yt_client_cs_input_spec, self.server, active_cs_input_spec_resources, self.Parameters.cs_input_spec_ttl)
        except Exception:
            logger.error('Got exception while trying to ping CS input spec', exc_info=True)
            failures.append('cs_input_spec')

        try:
            active_binary_base_resources = get_active_binary_base_resources(self.server)
            ping_active_binary_base_resources(self.server, active_binary_base_resources)
        except Exception:
            logger.error('Got exception while trying to ping binary bases', exc_info=True)
            failures.append('binary_bases')

        if failures:
            raise TaskFailure('Failed to ping [{}], see logs for more details'.format(', '.join(failures)))

        failures = []
        successfully_removed_endpoints, failed_to_remove_endpoints = remove_expired_hamster_endpoints(
            nanny_token=tokens['nanny_token'],
            yp_token=tokens['yp_token'],
            yt_client=yt_client_hamster_endpoints,
            now_timestamp=now,
            active_endpoints=active_hamster_endpoints,
        )

        if successfully_removed_endpoints:
            self.set_info('Successfully removed [{}]'.format(_concat_endpoints(successfully_removed_endpoints)))
            self.Parameters.tags = self.Parameters.tags + list({'REMOVED_{endpoint.service_name}'.format(endpoint=endpoint) for endpoint in successfully_removed_endpoints})

        if failed_to_remove_endpoints:
            failures.extend(['{endpoint.service_name} #{endpoint.resource_id}'.format(endpoint=endpoint) for endpoint in failed_to_remove_endpoints])

        try:
            remove_kvrs_frozen_states(yt_client_saas_shapshots, active_kvrs_saas_freeze_data_resources)
        except:
            logger.error('Failed to unfreeze snapshots', exc_info=True)
            failures.append('saas_kvrs_frozen_data')

        if failures:
            raise TaskFailure('Failed to remove [{}], see logs for more details'.format(', '.join(failures)))


def _concat_endpoints(endpoints):
    """ Generates description string from list of ExpiredEndpoints
    """
    return ', '.join('{endpoint.service_name} #{endpoint.resource_id}'.format(endpoint=endpoint) for endpoint in endpoints)
