import itertools
import json
import logging
from collections import namedtuple

from sandbox.projects.yabs.qa.hamster.record import ENDPOINTS_TABLE_PATH
from sandbox.projects.yabs.qa.hamster.remove import remove_hamster
from sandbox.projects.yabs.qa.hamster.testenv import get_testenv_hamster_endpoints
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.spec import get_last_released_spec_data

logger = logging.getLogger(__name__)
ExpiredEndpoint = namedtuple("ExpiredEndpoint", ("service_name", "resource_id"))


def _get_hamster_endpoints_from_spec(spec_data, sandbox_client):
    ext_service_endpoint_resource_ids = spec_data["ext_service_endpoint_resource_ids"]
    result = {}
    for resource_id in ext_service_endpoint_resource_ids:
        service_tag = sandbox_client.resource[resource_id].read()["attributes"]["service_tag"]
        result[service_tag] = [resource_id]
    return result


def _get_active_one_shot_spec_hamster_endpoints(sandbox_client):
    spec_data = get_last_released_spec_data(
        resource_type="YABS_SERVER_HOUSE_SPEC",
        sandbox_client=sandbox_client,
    )

    return _get_hamster_endpoints_from_spec(spec_data, sandbox_client)


def _get_active_ab_experiment_spec_hamster_endpoints(sandbox_client):
    spec_data = get_last_released_spec_data(
        resource_type="YABS_SERVER_ABEXPERIMENT_SPEC",
        sandbox_client=sandbox_client,
    )

    return _get_hamster_endpoints_from_spec(spec_data, sandbox_client)


def get_active_hamster_endpoints(testenv_resources, sandbox_client, ok_old_resources_to_keep=None):
    testenv_endpoints = get_testenv_hamster_endpoints(
        testenv_resources,
        ok_old_resources_to_keep=ok_old_resources_to_keep)
    logger.debug("Testenv hamster endpoint resources: %s", testenv_endpoints)

    oneshot_endpoints = _get_active_one_shot_spec_hamster_endpoints(sandbox_client)
    logger.debug("Oneshot spec hamster endpoint resources: %s", oneshot_endpoints)

    ab_experiment_endpoints = _get_active_ab_experiment_spec_hamster_endpoints(sandbox_client)
    logger.debug("AB experiment spec hamster endpoint resources: %s", ab_experiment_endpoints)

    active_endpoints = {}
    for service_tag, resource_ids in itertools.chain(
            testenv_endpoints.items(),
            oneshot_endpoints.items(),
            ab_experiment_endpoints.items(),
    ):
        active_endpoints.setdefault(service_tag, set()).update(resource_ids)

    return active_endpoints


def ping_active_hamster_endpoints(yt_client, active_endpoint_resources, now_timestamp, ttl, insert=False):
    endpoints_to_ping_set = set(active_endpoint_resources.keys())
    endpoints_to_ping = [
        {"resource_id": resource_id, "service_name": service_tag}
        for resource_id, service_tag in endpoints_to_ping_set
    ]
    logger.debug('Will ping endpoints: %s', json.dumps(endpoints_to_ping, indent=2))

    existing_endpoints = list(yt_client.lookup_rows(ENDPOINTS_TABLE_PATH, endpoints_to_ping, format='json'))
    logger.debug('Found endpoints in %s: %s', ENDPOINTS_TABLE_PATH, json.dumps(existing_endpoints, indent=2))

    existing_endpoints_set = set(
        (e["resource_id"], e["service_name"])
        for e in existing_endpoints
    )
    endpoints_to_insert = endpoints_to_ping_set - existing_endpoints_set
    if insert and endpoints_to_insert:
        for resource_id, service_name in endpoints_to_insert:
            enpoint_resource_data = active_endpoint_resources[(resource_id, service_name)]
            existing_endpoints.append({
                "resource_id": resource_id,
                "service_name": service_name,
                "created": now_timestamp,
                "service_id": enpoint_resource_data["service_id"],
                "cluster": enpoint_resource_data["cluster"],
                "endpoint_set": enpoint_resource_data["endpoint_set"],
                "service_type": enpoint_resource_data["service_type"],
            })

    for endpoint in existing_endpoints:
        endpoint.update(
            accessed=now_timestamp,
            expired=max(now_timestamp + ttl, endpoint.get("expired", 0)),
        )

    logger.debug('Will insert endpoints into %s: %s', ENDPOINTS_TABLE_PATH, json.dumps(existing_endpoints, indent=2))
    yt_client.insert_rows(ENDPOINTS_TABLE_PATH, existing_endpoints, format='json')


def _get_expired_hamster_endpoints(yt_client, now_timestamp):
    query = '* from [{}] where expired < {}'.format(ENDPOINTS_TABLE_PATH, now_timestamp)
    logger.debug('Running query: "{}"'.format(query))
    return list(yt_client.select_rows(query, format='json'))


def _is_active_endpoint(endpoint, active_endpoint_resources):
    return endpoint['resource_id'] in active_endpoint_resources


def remove_expired_hamster_endpoints(nanny_token, yp_token, yt_client, now_timestamp, active_endpoints={}):
    expired_hamster_endpoints = _get_expired_hamster_endpoints(yt_client, now_timestamp)
    logger.debug('Expired endpoints: %s', expired_hamster_endpoints)

    active_endpoint_resources = list(itertools.chain.from_iterable(active_endpoints.values()))
    logger.debug('Active hamster endpoint resources: %s', active_endpoint_resources)

    successfully_removed_endpoints = set()
    failed_to_remove_endpoints = set()
    for expired_endpoint in expired_hamster_endpoints:
        if _is_active_endpoint(expired_endpoint, active_endpoint_resources):
            logger.error('Tried to remove active hamster: %s', expired_endpoint)
            failed_to_remove_endpoints.add(ExpiredEndpoint(expired_endpoint['service_name'], expired_endpoint['resource_id']))
            continue

        logger.info(
            'Will remove %s service %s at %s from resource_id=%s',
            expired_endpoint['service_name'], expired_endpoint['service_id'], expired_endpoint['cluster'], expired_endpoint['resource_id']
        )
        logger.debug('Remove endpoint: %s', expired_endpoint)
        try:
            remove_hamster(
                service_id=expired_endpoint['service_id'],
                service_type=expired_endpoint['service_type'],
                cluster=expired_endpoint['cluster'],
                nanny_token=nanny_token,
                yp_token=yp_token,
                force=True,
            )
            successfully_removed_endpoints.add(ExpiredEndpoint(expired_endpoint['service_name'], expired_endpoint['resource_id']))
        except Exception:
            logger.error('Got exception while trying to remove %s', expired_endpoint, exc_info=True)
            failed_to_remove_endpoints.add(ExpiredEndpoint(expired_endpoint['service_name'], expired_endpoint['resource_id']))

    yt_client.delete_rows(ENDPOINTS_TABLE_PATH, successfully_removed_endpoints, format='json')
    return successfully_removed_endpoints, failed_to_remove_endpoints
