import datetime
import json
import logging
from collections import defaultdict

from sandbox import sdk2
from sandbox.projects.yabs.qa.saas.coordinator import SaasKVRSCoordinatorClient
from sandbox.projects.yabs.qa.saas.record import extend_expiration_time as extend_saas_record_expiration_time, get_snapshot_metas_kvrs_to_delete, delete_record
from sandbox.projects.yabs.qa.saas.topology import (
    topology_from_string,
    topology_to_states,
)
from sandbox.projects.yabs.qa.tasks.YabsServerFreezeCurrentKvrsSaasSnapshot import DEFAULT_CLUSTER, DEFAULT_ENDPOINT_SET_ID


logger = logging.getLogger(__name__)


DEFAULT_TOPOLOGY_TTL = datetime.timedelta(days=1).total_seconds()


def _get_active_testenv_kvrs_saas_freeze_data(testenv_resources, ignore_statuses=('BAD',)):
    saas_freeze_data_resources = defaultdict(list)
    for testenv_resource in testenv_resources:
        if (
            testenv_resource['name'] == 'YABS_SERVER_SAAS_FROZEN_TOPOLOGY'
            and testenv_resource['status'] not in ignore_statuses
        ):
            saas_freeze_data_resources[testenv_resource['status']].append(testenv_resource)

    active_saas_freeze_data_resources = []
    for status, resources in saas_freeze_data_resources.items():
        resource = max(resources, key=lambda x: x['resource_timestamp'])
        logger.debug('Found active %s %s, resource_id=%s', status, resource['name'], resource['resource_id'])
        active_saas_freeze_data_resources.append(resource['resource_id'])

    logger.debug('Got resources: %s', json.dumps(active_saas_freeze_data_resources, indent=2))
    return active_saas_freeze_data_resources


def get_active_kvrs_saas_freeze_data(testenv_resources, sandbox_client):
    active_saas_freeze_data = _get_active_testenv_kvrs_saas_freeze_data(testenv_resources)
    # active_saas_freeze_data.extend(_get_active_one_shot_spec_saas_freeze_data(sandbox_client))
    # active_saas_freeze_data.extend(_get_active_ab_experiment_spec_saas_freeze_data(sandbox_client))
    return list(set(active_saas_freeze_data))


def ping_active_kvrs_saas_freeze_data(yt_client, active_saas_freeze_data_resources, now_timestamp):
    for resource_id in active_saas_freeze_data_resources:
        extend_saas_record_expiration_time(yt_client, resource_id, DEFAULT_TOPOLOGY_TTL)


def remove_kvrs_frozen_states(yt_client, active_snapshots):
    saas_coordinator_client = SaasKVRSCoordinatorClient(DEFAULT_CLUSTER, DEFAULT_ENDPOINT_SET_ID)

    resource_ids_to_delete = get_snapshot_metas_kvrs_to_delete(yt_client)
    for resource_id in resource_ids_to_delete:
        logger.debug("Remove SaaS states from resource %s")

        if resource_id in active_snapshots:
            logger.warning("Trying to delete active KVRS SaaS snapshot resource %s, skipping", resource_id)
            continue

        try:
            topology_resource = sdk2.Resource[resource_id]
        except:
            logger.warning("Resource %s not found", resource_id)
            delete_record(yt_client, resource_id)
            continue

        topology_resource_path = str(sdk2.ResourceData(topology_resource).path)
        with open(topology_resource_path, 'rb') as topology_file:
            topology_str = topology_file.read()

        topology = topology_from_string(topology_str)
        states = topology_to_states(topology)
        saas_coordinator_client.unfreeze_state(states)
        delete_record(yt_client, resource_id)
