import json
import time
import logging

from collections import defaultdict

import requests

from sandbox import sdk2
from sandbox.projects.common.decorators import retries
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_SAAS_FREEZE_DATA
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.spec import get_last_released_spec_data

logger = logging.getLogger(__name__)

SAAS_COORDINATOR_BASE_URL = 'https://ctrl.clusterstate.yandex-team.ru/saas2/coordinator/saas2/coordinator'
SNAPSHOTS_TABLE_PROXY = 'hahn'
SNAPSHOTS_TABLE_PATH = '//home/yabs-cs-sandbox/saas/Snapshots'
SEMAPHORE_NAME = 'yabs_server_freeze_saas_snapshots_creative'


class SaasCoordinatorClient(object):

    def __init__(self, base_url=SAAS_COORDINATOR_BASE_URL):
        self.__base_url = base_url
        self.__session = requests.Session()

    def __do_request(self, method, **kwargs):
        request = requests.Request(method.upper(), self.__base_url, **kwargs)
        prepared_request = request.prepare()
        logger.debug(
            'Sending request:\n%s\n%s%s',
            prepared_request.method + ' ' + prepared_request.url,
            '\n'.join('{}: {}'.format(k, v) for k, v in prepared_request.headers.items()),
            '\n\n{}'.format(prepared_request.body) if prepared_request.body else '',
        )
        response = self.__session.send(prepared_request)
        logger.debug('Got response from %s: %d\n%s', response.url, response.status_code, response.content)
        response.raise_for_status()
        return response

    @retries(max_tries=3, delay=2)
    def get_snapshots(self, **kwargs):
        kwargs.setdefault('params', {}).update(handler='/snapshots')
        return self.__do_request('GET', **kwargs).json()

    @retries(max_tries=3, delay=2)
    def post_freeze(self, **kwargs):
        kwargs.setdefault('params', {}).update(handler='/freeze')
        return self.__do_request('POST', **kwargs).json()


def _get_active_testenv_saas_freeze_data(testenv_resources, ignore_statuses=('BAD', 'OK_OLD', )):
    saas_freeze_data_resources = defaultdict(list)
    for testenv_resource in testenv_resources:
        if testenv_resource['name'] == 'YABS_SERVER_SAAS_FREEZE_DATA' and testenv_resource['status'] not in ignore_statuses:
            saas_freeze_data_resources[testenv_resource['status']].append(testenv_resource)

    active_saas_freeze_data_resources = []
    for status, resources in saas_freeze_data_resources.items():
        resource = max(resources, key=lambda x: x['resource_timestamp'])
        logger.debug('Found active %s %s, resource_id=%s', status, resource['name'], resource['resource_id'])
        active_saas_freeze_data_resources.append(resource['resource_id'])

    logger.debug('Got resources: %s', json.dumps(active_saas_freeze_data_resources, indent=2))
    return active_saas_freeze_data_resources


def _get_active_one_shot_spec_saas_freeze_data(sandbox_client):
    spec_data = get_last_released_spec_data(
        resource_type="YABS_SERVER_HOUSE_SPEC",
        sandbox_client=sandbox_client,
    )
    return [spec_data['saas_freeze_data_resource_id']]


def _get_active_ab_experiment_spec_saas_freeze_data(sandbox_client):
    spec_data = get_last_released_spec_data(
        resource_type="YABS_SERVER_ABEXPERIMENT_SPEC",
        sandbox_client=sandbox_client,
    )
    return [spec_data['saas_freeze_data_resource_id']]


def get_active_saas_freeze_data(testenv_resources, sandbox_client):
    active_saas_freeze_data = _get_active_testenv_saas_freeze_data(testenv_resources)
    active_saas_freeze_data.extend(_get_active_one_shot_spec_saas_freeze_data(sandbox_client))
    active_saas_freeze_data.extend(_get_active_ab_experiment_spec_saas_freeze_data(sandbox_client))
    return list(set(active_saas_freeze_data))


def get_active_snapshot_metas(yt_client, table_path):
    rows = list(yt_client.read_table(table_path, format='json'))
    logger.debug('Got active snapshots data from %s: %s', table_path, rows)
    return [(row["stream"], row["id"]) for row in rows]


def insert_snapshot_metas(yt_client, table_path, new_snapshots_to_freeze):
    now = int(time.time())
    rows = [
        {
            "stream": snapshot["Meta"]["Stream"],
            "id": snapshot["Meta"]["Id"],
            "created": now,
            "accessed": now,
        }
        for snapshot in new_snapshots_to_freeze
    ]
    yt_client.insert_rows(table_path, rows, format='json')


def get_new_snapshots_to_freeze(available_saas_snapshots, streams_to_freeze):
    new_snapshots_to_freeze = []
    processed_streams = set()

    for snapshot in reversed(available_saas_snapshots):
        if all([
            snapshot["Meta"]["Stream"] not in processed_streams,
            snapshot["Meta"]["Stream"] in streams_to_freeze,
            not snapshot["Status"]["Frozen"],
            snapshot["Status"]["Active"],
        ]):
            processed_streams.add(snapshot["Meta"]["Stream"])
            new_snapshots_to_freeze.append(snapshot)

    logger.info('New snapshots available for freezing: %s', json.dumps(new_snapshots_to_freeze, indent=2))

    return new_snapshots_to_freeze


def get_old_snapshots_to_freeze(available_saas_snapshots, active_snapshot_metas):
    old_snapshots_to_freeze = [
        snapshot
        for snapshot in available_saas_snapshots
        if all([
            (snapshot["Meta"]["Stream"], snapshot["Meta"]["Id"]) in active_snapshot_metas,
            snapshot["Status"]["Frozen"],
            snapshot["Status"]["Active"],
        ])
    ]
    logger.info('Old active snapshots we have to keep: %s', json.dumps(old_snapshots_to_freeze, indent=2))
    return old_snapshots_to_freeze


@retries(max_tries=4, delay=10, backoff=5)
def freeze_snapshot_metas(saas_coordinator_client, yt_client, new_snapshots_to_freeze, old_snapshots_to_freeze, do_freeze=True):
    """
    Inserts new snapshot's meta into the table, then tries to freeze snapshots.
    In case of failure of freezing process record in the table persists, but it will be removed by TTL eviction policy due to unability to use this snapshot.
    """
    snapshots_to_freeze = new_snapshots_to_freeze + old_snapshots_to_freeze
    logger.debug('Will freeze %d snapshots: %s', len(snapshots_to_freeze), json.dumps(snapshots_to_freeze, indent=2))

    if do_freeze:
        insert_snapshot_metas(yt_client, SNAPSHOTS_TABLE_PATH, new_snapshots_to_freeze)
        saas_coordinator_client.post_freeze(json={'SnapshotMetas': [snapshot["Meta"] for snapshot in snapshots_to_freeze]})


def ping_active_saas_freeze_data(yt_client, active_saas_freeze_data_resources, now_timestamp):

    snapshots_to_ping = []
    for resource_id in active_saas_freeze_data_resources:
        freeze_data_resource = sdk2.ResourceData(YABS_SERVER_SAAS_FREEZE_DATA[resource_id])
        with open(str(freeze_data_resource.path), 'r') as freeze_data_file:
            freeze_data = json.load(freeze_data_file)
        snapshots_to_ping.extend([
            {"stream": snapshot['stream'], "id": snapshot['id']}
            for snapshot in freeze_data
        ])

    logger.debug('Will ping snapshots: %s', json.dumps(snapshots_to_ping, indent=2))
    existing_snapshots = list(yt_client.lookup_rows(SNAPSHOTS_TABLE_PATH, snapshots_to_ping, format='json'))
    logger.debug('Found snapshots in %s: %s', SNAPSHOTS_TABLE_PATH, json.dumps(existing_snapshots, indent=2))
    for snapshot in existing_snapshots:
        snapshot['accessed'] = now_timestamp
    logger.debug('Will insert snapshots into %s: %s', SNAPSHOTS_TABLE_PATH, json.dumps(existing_snapshots, indent=2))
    yt_client.insert_rows(SNAPSHOTS_TABLE_PATH, existing_snapshots, format='json')
