from __future__ import division

from datetime import timedelta
from collections import namedtuple

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.performance import (
    collect,
    scheduling,
)
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_META_NETWORK_STATISTICS
from sandbox.projects.yabs.qa.performance.update_parameters import apply_update_parameters_resource
from sandbox.projects.yabs.qa.tasks.YabsServerStatPerformance2 import YabsServerStatPerformance2, YabsServerStatPerformanceParameters
from sandbox.projects.yabs.qa.utils import general

PerfResults = namedtuple("PerfResults", ["rps", "rps_corrected", "rss", "perf"])
PerfResults.__new__.__defaults__ = (None, ) * len(PerfResults._fields)


class YabsServerStatPerformanceBest2(BaseBinTask):

    class Parameters(YabsServerStatPerformanceParameters):
        kill_timeout = int(timedelta(minutes=20).total_seconds())
        push_tasks_resource = True

        with sdk2.parameters.Group('Batch settings') as batch_settings:
            subtasks_without_corrections_count = sdk2.parameters.Integer('Number of subtasks allowed to run on "unknown" hosts', default_value=16)
            subtasks_count = sdk2.parameters.Integer('Number of subtasks', default_value=16)

            abandoned_limit = sdk2.parameters.Integer('Abandoned subtask limit', default_value=10)
            failure_limit = sdk2.parameters.Integer('Failed subtask limit', default_value=0)
            other_break_limit = sdk2.parameters.Integer('Otherwise broken subtask limit', default_value=2)
            use_2on1 = sdk2.parameters.Bool('Run actual performance tests under CMP task. Use one host for pre and test shoots')

        with sdk2.parameters.Output:
            meta_network_statistics = sdk2.parameters.Resource(
                'Resource with network statistics for meta',
                resource_type=YABS_SERVER_META_NETWORK_STATISTICS,
            )

    class Context(sdk2.Task.Context):
        subtasks_without_corrections = []
        need_2_on_1 = False
        use_2on1_in_cmp_task = False

    def run_subtasks(self, parameters):
        scheduling.schedule_tasks(
            self,
            YabsServerStatPerformance2,
            parameters,
            parameters.subtasks_without_corrections_count,
            self.Context.subtasks_without_corrections,
            self.checker,
            description='Child run\n' + self.Parameters.description,
            tasks_resource=self.Requirements.tasks_resource,
        )

    def check_and_rerun_subtasks(self, parameters):
        need_wait = False
        need_wait |= scheduling.check_and_reschedule_tasks(
            self,
            self.Context.subtasks_without_corrections,
            self.checker,
            description='Child rerun\n' + self.Parameters.description,
            tasks_resource=self.Requirements.tasks_resource,
        )
        return need_wait

    def subtask_list(self):
        return scheduling.get_task_list_from_data_iterable(self.Context.subtasks_without_corrections)

    def wait_subtasks(self):
        raise sdk2.WaitTask(
            filter(
                lambda task: task.status not in ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                self.subtask_list()
            ),
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=False
        )

    @track_errors
    def on_execute(self):
        parameters = apply_update_parameters_resource(self.Parameters)
        self.checker = scheduling.TaskChecker(
            failure_limit=parameters.failure_limit,
            abandoned_limit=parameters.abandoned_limit,
            other_break_limit=parameters.other_break_limit
        )
        if parameters.use_2on1:
            self.Context.need_2_on_1 = True
            self.Context.use_2on1_in_cmp_task = True
            self.set_info('2 on 1 dummy')
            return
        if not self.subtask_list():
            self.run_subtasks(parameters)
            need_wait = True
        else:
            need_wait = self.check_and_rerun_subtasks(parameters)
        if need_wait:
            self.wait_subtasks()

        results = self.collect_results(self.subtask_list())
        report = self.make_report(results)
        self.set_info(report, do_escape=False)

    def collect_results(self, subtasks):
        rps_result = collect.collect_rps(subtasks, corrected=False)
        rps_result.__to_context__(self.Context)

        rps_result_corrected = collect.collect_rps(subtasks, corrected=True)
        rps_result_corrected.__to_context__(self.Context)

        rss_result = collect.collect_rss(subtasks)
        rss_result.__to_context__(self.Context)

        perf_result = collect.collect_perf(subtasks)
        perf_result.__to_context__(self.Context)

        return PerfResults(
            rps=rps_result,
            rps_corrected=rps_result_corrected,
            rss=rss_result,
            perf=perf_result,
        )

    def make_report(self, results):
        report = '<br>'.join([
            'RPS 80th percentile (host-corrected): {rps}'.format(rps=results.rps_corrected.rps_percentile_80_corrected),
            'RPS Hodges-Lehman median (host-corrected): {rps}'.format(rps=results.rps_corrected.rps_hl_median_corrected),
            'RPS 80th percentile: {rps}'.format(rps=results.rps.rps_percentile_80),
            'RPS Hodges-Lehman median: {rps}'.format(rps=results.rps.rps_hl_median),
        ] + [
            'RSS {} Hodges-Lehmann median for yabs-server {}: {:.3f} GB'.format(
                result_type, engine_mode, getattr(results.rss, collect.get_rss_result_key(result_type, engine_mode, hl_median=True)) / (1 << 20))
            for result_type in (collect.ResultType.baseline, collect.ResultType.increase)
            for engine_mode in (collect.EngineMode.meta, collect.EngineMode.stat)
        ] + [
            general.html_hyperlink(results.perf.perf_link, 'Perf resource link')
        ])
        return report
