from __future__ import division

import jinja2
import json
import logging
import os
import subprocess
from collections import namedtuple
from datetime import timedelta
from functools import partial

from sandbox import sdk2
from sandbox.common.fs import get_unique_file_name, make_folder
from sandbox.common.utils import get_task_link
from sandbox.common.types import task as ctt
from sandbox.common.format import size2str

from sandbox.projects.yabs.base_bin_task import BaseBinTaskMixin, base_bin_task_parameters
from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.utils.general import startrek_hyperlink, html_hyperlink
from sandbox.projects.yabs.qa.tasks.base_compare_task import (
    parameters as cmp_parameters,
    task as cmp_task,
)
from sandbox.projects.yabs.qa.performance import (
    collect,
    compare,
    scheduling,
)

from sandbox.projects.yabs.qa.performance.stats.access import IntervalMetrics
from sandbox.projects.yabs.qa.performance.update_parameters import apply_update_parameters_resource
from sandbox.projects.yabs.qa.tasks.YabsServerStatPerformance2On1_2 import YabsServerStatPerformance2On1_2
from sandbox.projects.yabs.qa.tasks.YabsServerStatPerformance2 import StatLoadShootParameters
from sandbox.projects.yabs.qa.resource_types import BaseBackupSdk2Resource


class YabsServerStatPerformanceDiffReport(BaseBackupSdk2Resource):
    has_diff = sdk2.Attributes.Bool("Flag for diff in resource", default=False)


DIFF_REPORT_FILENAME = 'diff_report.html'
DIFF_REPORT_TTL = 60
ALL_HANDLERS = '__ALL_HANDLERS__'
ACCESS_LOG_DIFF_FOLDER = 'access_log_diff'


DiffResults = namedtuple("DiffResults", ["rps", "avg_rps", "rss_stat", "rss_meta", "perf"])
DiffResults.__new__.__defaults__ = (None, ) * len(DiffResults._fields)


HTML_NETWORK_STATS_TABLE = jinja2.Template('''{{ header }}
<table>
    <tr>
    {% for col_name in stats[0] %}
        <th>{{ col_name }}</th>
    {% endfor %}
    </tr>
    {% for row in stats[1:] %}
    <tr>
        {% for val in row %}
        <td>{{ val }}</td>
        {% endfor %}
    </tr>
    {% endfor %}
</table>''')
ST_NETWORK_STATS_TABLE = jinja2.Template('''{{ header }}
#|
||{% for col_name in stats[0] %}**{{ col_name }}**|{% endfor %}|
{% for row in stats[1:] %}
||{% for val in row %}{{ val }}|{% endfor %}|
{% endfor %}
|#''')


def create_comparison_metareport(lines, links, line_sep='\n', link_formatter=startrek_hyperlink, stats_lines=None, stats_template=ST_NETWORK_STATS_TABLE):
    if stats_lines is None:
        stats_lines = []
    task_executing_report_template = line_sep.join(['{lines}', '{links}', '{stats}'])
    return task_executing_report_template.format(
        lines=line_sep.join(lines),
        links=line_sep.join(link_formatter(link_url, link_text) for link_text, link_url in links),
        stats=line_sep.join(stats_template.render(header=header, stats=stats) for header, stats in stats_lines),
    )


def compare_network_statistics_by_handlers(pre_stats, test_stats):
    DIFF_FORMAT = '{:+.3f}%'
    handlers = set(pre_stats.keys() + test_stats.keys())
    res = []
    res.append(['handler', 'pre mean', 'test mean', '% diff'])

    def get_mean(v):
        if v is None:
            return {
                'req': 0.,
                'res': 0.,
            }
        return {
            'req': float(v['request_size']) / v['count'],
            'res': float(v['response_size']) / v['count'],
        }

    def add_row(name, pre, test):
        diff = (test / pre - 1) * 100 if pre > 0 else 100
        res.append([
            name,
            size2str(pre),
            size2str(test),
            DIFF_FORMAT.format(diff),
        ])

    for handler in handlers:
        pre = get_mean(pre_stats.get(handler))
        test = get_mean(test_stats.get(handler))
        add_row('req ' + handler, pre['req'], test['req'])
        add_row('res ' + handler, pre['res'], test['res'])

    return res


def get_network_meta_statistics(task):
    if task is None:
        return None

    resource = task.Parameters.meta_network_statistics
    if resource is None:
        return None

    with open(str(sdk2.ResourceData(resource).path)) as f:
        return json.load(f)


class YabsServerStatPerformanceBestCmp2(BaseBinTaskMixin, cmp_task.BaseCompareTask):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(cmp_parameters.BaseCompareTaskParameters):
        kill_timeout = int(timedelta(minutes=21).total_seconds())
        push_tasks_resource = True

        _base_bin_task_parameters = base_bin_task_parameters(
            release_version_default=ctt.ReleaseStatus.STABLE,
            resource_attrs_default={"task_bundle": "yabs_server_stat_load"},
        )

        with sdk2.parameters.Group('Comparison parameters') as comparison_parameters:
            rps_diff_threshold_percent = sdk2.parameters.Float('RPS diff threshold, %', default_value=10.0)
            rss_diff_threshold_percent = sdk2.parameters.Float('RSS diff threshold, %', default_value=10.0)
            improvement_is_diff = sdk2.parameters.Bool('Improvement is diff', default_value=True)

            compare_input_parameters = cmp_parameters.BaseCompareTaskParameters.compare_input_parameters()
            with compare_input_parameters.value[True]:
                critical_parameters_mode = cmp_parameters.BaseCompareTaskParameters.critical_parameters_mode(
                    default=cmp_parameters.CriticalParametersDiffHandleMode.mark_as_diff,
                )

        use_check_task_shoot_parameters = sdk2.parameters.Bool('Get shoot parameters from check task', default=True)
        with use_check_task_shoot_parameters.value[False]:
            with sdk2.parameters.Group('Shoot module settings') as shoot_module_settings:
                stat_load_shoot_parameters = StatLoadShootParameters()
                stat_shoot_mode = sdk2.parameters.String('Stat shoot mode', choices=[(x, x) for x in ['finger', 'plan', 'binary', 'fuckup']], default_value='finger')

            with sdk2.parameters.Group('Scheduling parameters') as scheduling_settings:
                task_count = sdk2.parameters.Integer('Task count', default=16)
                abandoned_limit = sdk2.parameters.Integer('Abandoned subtask limit', default_value=10)
                failure_limit = sdk2.parameters.Integer('Failed subtask limit', default_value=2)
                other_break_limit = sdk2.parameters.Integer('Otherwise broken subtask limit', default_value=2)
        use_splitted_dplan = sdk2.parameters.Bool('Use spliited dplan for child tasks', default=False)
        with use_splitted_dplan.value[True]:
            default_task_count = sdk2.parameters.Integer('Default task count', default=6)
            task_count_per_handler = sdk2.parameters.JSON('Task count per handler', default={'pmatch': 10})
            shoot_sessions_per_handler = sdk2.parameters.JSON('Shoot sessions per handler', default={})
            shoot_threads_per_handler = sdk2.parameters.JSON('Shoot threads per handler', default={})
            shoot_request_limit_per_handler = sdk2.parameters.JSON('Shoot request limit per handler', default={})

        with sdk2.parameters.Group('Batch mode parameters') as batch_parameters:
            need_ram_check = sdk2.parameters.Bool('Need additional RAM check', default_value=False)

        with sdk2.parameters.Output:
            collected_results = sdk2.parameters.JSON('Collected metrics', default={})

    class Context(sdk2.Task.Context):
        has_diff = False
        subtasks_2_on_1_data = []
        collected_results = {}

    ignore_compare_input_parameters = [
        'binary_base_resources',
        'server_resource',
        'stat_binary_base_resources',
        'stat_server_resource',
        'meta_binary_base_resources',
        'meta_server_resource',
    ]
    critical_compare_input_parameters = [
        'cache_daemon_stub_resource',
        'shoot_plan_resource',
        'requestlog_resource',
        'use_requestlog',
    ]

    def compare_pre_test(self, results_class, compare_method):
        pre_results = results_class.__from_context__(self.Parameters.pre_task.Context)
        test_results = results_class.__from_context__(self.Parameters.test_task.Context)
        return compare_method(pre_results, test_results)

    def compare_from_subtasks(self, subtasks, collect_method, compare_method):
        pre_results = collect_method(subtasks)
        pre_results.__to_context__(self.Context)
        test_results = collect_method(subtasks, second_batch=True)
        test_results.__to_context__(self.Context, second_batch=True)
        return compare_method(pre_results, test_results)

    def compare_from_subtasks_with_splitted_dplans(self, subtasks, handlers, collect_method, compare_method):
        pre_results = collect_method(subtasks)
        test_results = collect_method(subtasks, second_batch=True)

        for results, batch in (
            (pre_results, 'pre'),
            (test_results, 'test'),
        ):
            for key, value in results._asdict().iteritems():
                self.Context.collected_results[handlers][batch][key] = value

        return compare_method(pre_results, test_results)

    def evaluate_results(self, results):
        has_diff = False
        if results.rss_stat:
            rss_degraded = results.rss_stat.diff_percent < -1 * self.Parameters.rss_diff_threshold_percent
            rss_improved = results.rss_stat.diff_percent > self.Parameters.rss_diff_threshold_percent
            has_rss_stat_diff = (rss_degraded or (rss_improved and self.Parameters.improvement_is_diff)) and bool(self.Parameters.rss_diff_threshold_percent)
            has_diff = has_diff or has_rss_stat_diff

        if results.rss_meta:
            rss_degraded = results.rss_meta.diff_percent < -1 * self.Parameters.rss_diff_threshold_percent
            rss_improved = results.rss_meta.diff_percent > self.Parameters.rss_diff_threshold_percent
            has_rss_meta_diff = (rss_degraded or (rss_improved and self.Parameters.improvement_is_diff)) and bool(self.Parameters.rss_diff_threshold_percent)
            has_diff = has_diff or has_rss_meta_diff

        if results.rps:
            rps_degraded = results.rps.diff_percent < -1 * self.Parameters.rps_diff_threshold_percent
            rps_improved = results.rps.diff_percent > self.Parameters.rps_diff_threshold_percent
            has_rps_diff = (rps_degraded or (rps_improved and self.Parameters.improvement_is_diff)) and bool(self.Parameters.rps_diff_threshold_percent)
            has_diff = has_diff or has_rps_diff

        return has_diff

    def on_save(self):
        super(YabsServerStatPerformanceBestCmp2, self).on_save()

        if self.Parameters.use_splitted_dplan:
            for pre_dplan, test_dplan in zip(
                self.Parameters.pre_task.Parameters.splitted_dplan_resources,
                self.Parameters.test_task.Parameters.splitted_dplan_resources
            ):
                if sdk2.Resource[pre_dplan].handlers != sdk2.Resource[test_dplan].handlers:
                    self.Parameters.use_splitted_dplan = False
                    break

    @track_errors
    def on_execute(self):
        terminate_task = self.check_tasks_parameters()
        if terminate_task:
            return

        do_2_on_1 = any(task.Context.need_2_on_1 for task in (self.Parameters.pre_task, self.Parameters.test_task))
        if do_2_on_1:
            diff_results = self.do_batch()
        else:
            diff_results = self.compare_pre_test_routine()

        if not self.Parameters.use_splitted_dplan:
            self.Context.has_diff = self.Parameters.has_diff = self.evaluate_results(list(diff_results.values())[0])
        else:
            pre_request_count = test_request_count = 0
            pre_weighted_rps = test_weighted_rps = rps_diff_percent = 0.0
            for splitted_dplan, splitted_dplan_2 in map(
                lambda dplans: [sdk2.Resource[id] for id in dplans],
                zip(
                    self.Parameters.pre_task.Parameters.splitted_dplan_resources,
                    self.Parameters.test_task.Parameters.splitted_dplan_resources
                )
            ):
                pre_request_count += int(splitted_dplan.request_count)
                test_request_count += int(splitted_dplan_2.request_count)
                pre_weighted_rps += diff_results[splitted_dplan.handlers].rps.pre_hl_median * int(splitted_dplan.request_count)
                test_weighted_rps += diff_results[splitted_dplan_2.handlers].rps.test_hl_median * int(splitted_dplan_2.request_count)
                rps_diff_percent += diff_results[splitted_dplan.handlers].rps.diff_percent * int(sdk2.Resource[splitted_dplan].request_count)
            self.Context.rps_hl_median = pre_weighted_rps / pre_request_count
            self.Context.rps_hl_median_2 = test_weighted_rps / test_request_count

            rps_diff_percent /= pre_request_count
            rps_degraded = rps_diff_percent < -1 * self.Parameters.rps_diff_threshold_percent
            rps_improved = rps_diff_percent > self.Parameters.rps_diff_threshold_percent
            all_diff = (rps_degraded or (rps_improved and self.Parameters.improvement_is_diff)) and bool(self.Parameters.rps_diff_threshold_percent)

            diff_results[ALL_HANDLERS] = DiffResults(
                rps=compare.CompareRpsResults(
                    pre_hl_median=self.Context.rps_hl_median,
                    test_hl_median=self.Context.rps_hl_median_2,
                    diff_percent=rps_diff_percent,
                ),
            )

            self.Context.has_diff = self.Parameters.has_diff = all_diff
            self.Parameters.collected_results = self.Context.collected_results
        html_report, st_report, short_report_text = self.make_report(diff_results)

        self.Context.short_report_text = short_report_text
        self.Context.short_report_link = get_task_link(self.id)
        with open(DIFF_REPORT_FILENAME, 'w') as f:
            f.write(html_report)

        self.set_info(html_report, do_escape=False)
        self.Parameters.st_report = st_report

        YabsServerStatPerformanceDiffReport(self, 'Report resource',
                                            DIFF_REPORT_FILENAME,
                                            ttl=DIFF_REPORT_TTL,
                                            has_diff=self.Context.has_diff)

    def do_batch(self):
        parameters_2_on_1, task_count, checker = self.produce_2_on_1_parameters()
        logging.info('Task count:  {}'.format(task_count))
        if not self.Context.subtasks_2_on_1_data:
            if self.Parameters.use_splitted_dplan:
                for splitted_dplan, splitted_dplan_2 in zip(
                    self.Parameters.pre_task.Parameters.splitted_dplan_resources,
                    self.Parameters.test_task.Parameters.splitted_dplan_resources
                ):
                    parameters_2_on_1.__dict__.update({
                        'stat_dplan_resource': splitted_dplan,
                        'stat_dplan_resource_2': splitted_dplan_2,
                    })

                    task_count = max([
                        self.Parameters.task_count_per_handler.get(handler, self.Parameters.default_task_count)
                        for handler in sdk2.Resource[splitted_dplan].handlers.split(',')
                    ])
                    parameters_2_on_1.__dict__['stat_shoot_sessions'] = max([
                        self.Parameters.shoot_sessions_per_handler.get(handler, self.Parameters.stat_shoot_sessions)
                        for handler in sdk2.Resource[splitted_dplan].handlers.split(',')
                    ])
                    parameters_2_on_1.__dict__['stat_shoot_threads'] = min([
                        self.Parameters.shoot_threads_per_handler.get(handler, self.Parameters.stat_shoot_threads)
                        for handler in sdk2.Resource[splitted_dplan].handlers.split(',')
                    ])
                    parameters_2_on_1.__dict__['stat_shoot_request_limit'] = max([
                        self.Parameters.shoot_request_limit_per_handler.get(handler, self.Parameters.stat_shoot_request_limit)
                        for handler in sdk2.Resource[splitted_dplan].handlers.split(',')
                    ])

                    scheduling.schedule_tasks(  # TODO: BSSERVER-21752
                        self,
                        YabsServerStatPerformance2On1_2,
                        parameters_2_on_1,
                        task_count,
                        self.Context.subtasks_2_on_1_data,
                        checker,
                        description='Child 2 on 1 run ' + self.Parameters.description,
                        tasks_resource=self.Requirements.tasks_resource,
                    )
            else:
                scheduling.schedule_tasks(
                    self,
                    YabsServerStatPerformance2On1_2,
                    parameters_2_on_1,
                    task_count,
                    self.Context.subtasks_2_on_1_data,
                    checker,
                    description='Child 2 on 1 run ' + self.Parameters.description,
                    tasks_resource=self.Requirements.tasks_resource,
                )
            need_wait = True
        else:
            need_wait = scheduling.check_and_reschedule_tasks(
                self,
                self.Context.subtasks_2_on_1_data,
                checker,
                description='Child 2 on 1 rerun ' + self.Parameters.description,
                tasks_resource=self.Requirements.tasks_resource,
            )
        if need_wait:
            self.wait_subtasks()

        if not self.Parameters.use_splitted_dplan:
            return self.compare_from_subtasks_routine(self.subtasks_list())
        else:
            subtasks_per_handlers = {}
            for task in self.subtasks_list():
                handlers = sdk2.Resource[task.Parameters.stat_dplan_resource].handlers
                subtasks_per_handlers.setdefault(handlers, [])
                subtasks_per_handlers[handlers].append(task)

            diff_per_handlers = {}
            for handlers, subtasks in subtasks_per_handlers.items():
                diff_per_handlers[handlers] = self.compare_from_subtasks_with_splitted_dplans_routine(subtasks, handlers)
            return diff_per_handlers

    def compare_pre_test_routine(self):
        compare_perf = partial(compare.compare_perf, self)
        return {ALL_HANDLERS: DiffResults(
            rps=self.compare_pre_test(collect.RpsResults, compare.compare_rps),
            rss_stat=self.compare_pre_test(collect.RssResults, compare.compare_rss_stat),
            perf=self.compare_pre_test(collect.PerfResults, compare_perf),
        )}

    def compare_from_subtasks_routine(self, subtasks):
        self.compare_access_log(subtasks)
        compare_perf = partial(compare.compare_perf, self)
        return {ALL_HANDLERS: DiffResults(
            rps=self.compare_from_subtasks(subtasks, collect.collect_rps, compare.compare_rps),
            rss_stat=self.compare_from_subtasks(subtasks, collect.collect_rss, compare.compare_rss_stat),
            perf=self.compare_from_subtasks(subtasks, collect.collect_perf, compare_perf),
            avg_rps=self.compare_from_subtasks(subtasks, collect.collect_avg_rps, compare.compare_rps),
        )}

    def compare_from_subtasks_with_splitted_dplans_routine(self, subtasks, handlers):
        self.Context.collected_results[handlers] = {
            'pre': {},
            'test': {},
        }
        self.compare_access_log(subtasks, handlers)
        compare_perf = partial(compare.compare_perf, self)
        return DiffResults(
            rps=self.compare_from_subtasks_with_splitted_dplans(subtasks, handlers, collect.collect_rps, compare.compare_rps),
            rss_stat=self.compare_from_subtasks_with_splitted_dplans(subtasks, handlers, collect.collect_rss, compare.compare_rss_stat),
            perf=self.compare_from_subtasks_with_splitted_dplans(subtasks, handlers, collect.collect_perf, compare_perf),
            avg_rps=self.compare_from_subtasks_with_splitted_dplans(subtasks, handlers, collect.collect_avg_rps, compare.compare_rps),
        )

    def compare_access_log(self, subtasks, handlers=ALL_HANDLERS, quantiles=(90, 99)):
        def is_swapped_test_order(subtask):
            return subtask.Parameters.shuffle_run_order and (subtask.Context.hosts_slot_index % 2)
        full_metrics = {
            'pre': IntervalMetrics(),
            'test': IntervalMetrics(),
        }
        best_metrics = {
            'pre': IntervalMetrics(),
            'test': IntervalMetrics(),
        }
        for subtask in subtasks:
            for index_2on1, suffix in (
                (1, ''),
                (2, '_2'),
            ):
                shoot_data_path = get_unique_file_name('', 'shoot_data_folder')
                subprocess.check_call([
                    'unzip',
                    str(sdk2.ResourceData(sdk2.Resource.find(
                        task=subtask,
                        attrs={
                            "index_2on1": index_2on1
                        }
                    ).first()).path),
                    '-d',
                    str(shoot_data_path)
                ])
                batch = 'pre' if (index_2on1 == 1 and not is_swapped_test_order(subtask)) or (index_2on1 == 2 and is_swapped_test_order(subtask)) else 'test'

                shoot_session_with_max_rps = 0
                for session_id in range(subtask.Parameters.stat_shoot_sessions):
                    with open(os.path.join(shoot_data_path, str(session_id), 'access-log.json')) as fp:
                        full_metrics[batch].update_from_dict(json.load(fp))
                    if getattr(subtask.Context, 'rps' + suffix) == getattr(subtask.Context, 'rps_list' + suffix)[session_id]:
                        shoot_session_with_max_rps = session_id
                with open(os.path.join(shoot_data_path, str(shoot_session_with_max_rps), 'access-log.json')) as fp:
                    best_metrics[batch].update_from_dict(json.load(fp))
        folder_path = get_unique_file_name(ACCESS_LOG_DIFF_FOLDER, handlers)
        make_folder(folder_path)
        for prefix, interval_metrics in (
            ('full', full_metrics),
            ('best', best_metrics),
        ):
            with open(os.path.join(folder_path, 'diff_{}.json'.format(prefix)), 'w') as fp:
                json.dump(
                    compare.dict_metrics_diff(
                        interval_metrics['pre'].prepare_metrics(quantiles),
                        interval_metrics['test'].prepare_metrics(quantiles)
                    ),
                    fp,
                    sort_keys=True,
                    indent=4
                )
        YabsServerStatPerformanceDiffReport(self, 'Access log diff folder', folder_path, handlers=handlers)

    def make_report(self, diff_results):
        report_lines = []
        report_links = []
        short_report_text = 'No report :('
        for handlers, results in diff_results.items():
            report_lines.append('Handlers: {}'.format(handlers))
            if results.rps:
                short_report_text = "Max requests/sec changed by {:+.1f}% ({:.2f} -> {:.2f})".format(
                    results.rps.diff_percent, results.rps.pre_hl_median, results.rps.test_hl_median,
                )
                report_lines.append(short_report_text)
            if results.avg_rps:
                report_lines.append("Avg requests/sec changed by {:+.1f}% ({} -> {})".format(
                    results.avg_rps.diff_percent, results.avg_rps.pre_hl_median, results.avg_rps.test_hl_median,
                ))
            if results.rss_stat:
                report_lines.append("Stat RSS changed by {:+.1f}% ({:.3f}GB -> {:.3f}GB)".format(
                    results.rss_stat.diff_percent,
                    results.rss_stat.pre_hl_median / (1 << 20),
                    results.rss_stat.test_hl_median / (1 << 20),
                ))
            if results.rss_meta:
                report_lines.append("Meta RSS changed by {:+.1f}% ({:.3f}GB -> {:.3f}GB)".format(
                    results.rss_meta.diff_percent,
                    results.rss_meta.pre_hl_median / (1 << 20),
                    results.rss_meta.test_hl_median / (1 << 20),
                ))
            if results.perf:
                report_links.append(('Diff flamegraph link', results.perf.diff_flamegraph_url))
            report_lines.append("")

        pre_meta_statistics = get_network_meta_statistics(self.Parameters.pre_task)
        test_meta_statistics = get_network_meta_statistics(self.Parameters.test_task)
        stats_lines = []
        if pre_meta_statistics and test_meta_statistics:
            header = 'Network statistics for yabstat'
            stats = compare_network_statistics_by_handlers(pre_meta_statistics, test_meta_statistics)
            stats_lines.append((header, stats))

        html_report = create_comparison_metareport(report_lines, report_links, line_sep="<br>", link_formatter=html_hyperlink, stats_lines=stats_lines, stats_template=HTML_NETWORK_STATS_TABLE)
        st_report = create_comparison_metareport(report_lines, report_links, line_sep="\n", link_formatter=startrek_hyperlink, stats_lines=stats_lines, stats_template=ST_NETWORK_STATS_TABLE)

        return html_report, st_report, short_report_text

    def subtasks_list(self):
        return scheduling.get_task_list_from_data_iterable(self.Context.subtasks_2_on_1_data)

    def wait_subtasks(self):
        raise sdk2.WaitTask(
            filter(
                lambda task: task.status not in ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                self.subtasks_list()
            ),
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=False
        )

    def produce_2_on_1_parameters(self):
        return_parameters = apply_update_parameters_resource(self.Parameters.pre_task.Parameters)
        test_parameters = apply_update_parameters_resource(self.Parameters.test_task.Parameters)
        for name, _ in self.Parameters.pre_task.Parameters.server_module_parameters:
            return_parameters.__dict__[name + '_2'] = test_parameters.__dict__[name]
        for name, _ in self.Parameters.pre_task.Parameters.stat_module_parameters:
            return_parameters.__dict__[name + '_2'] = test_parameters.__dict__[name]

        for task_parameters, parameter_suffix in (
            (self.Parameters.pre_task.Parameters, ''),
            (self.Parameters.test_task.Parameters, '_2'),
        ):
            return_parameters.__dict__['prepared_usage_data_meta' + parameter_suffix] = task_parameters.usage_data_meta
            return_parameters.__dict__['prepare_stat_dplan' + parameter_suffix] = False
            return_parameters.__dict__['stat_dplan_resource' + parameter_suffix] = task_parameters.dplan_resource
            return_parameters.__dict__['stat_store_request_log' + parameter_suffix] = False

        return_parameters.__dict__["need_ram_check"] = self.Parameters.need_ram_check

        if not self.Parameters.use_check_task_shoot_parameters:
            return_parameters.__dict__["stat_shoot_sessions"] = self.Parameters.stat_shoot_sessions
            return_parameters.__dict__["stat_shoot_threads"] = self.Parameters.stat_shoot_threads
            return_parameters.__dict__["stat_shoot_request_limit"] = self.Parameters.stat_shoot_request_limit
            return_parameters.__dict__["stat_circular_session"] = self.Parameters.stat_circular_session
            return_parameters.__dict__["store_dumps"] = self.Parameters.store_dumps
            return_parameters.__dict__["stat_shoot_mode"] = self.Parameters.stat_shoot_mode

            task_count = self.Parameters.task_count
            abandoned_limit = self.Parameters.abandoned_limit
            other_break_limit = self.Parameters.other_break_limit
            failure_limit = self.Parameters.failure_limit
        else:
            task_count = max((parameters.subtasks_without_corrections_count for parameters in (return_parameters, test_parameters))) or 1
            abandoned_limit = max(parameters.abandoned_limit for parameters in (return_parameters, test_parameters))
            other_break_limit = max(parameters.other_break_limit for parameters in (return_parameters, test_parameters))
            failure_limit = max(parameters.failure_limit for parameters in (return_parameters, test_parameters))

        return return_parameters, task_count, scheduling.TaskChecker(
            failure_limit=failure_limit,
            abandoned_limit=abandoned_limit,
            other_break_limit=other_break_limit
        )
