import json
import logging

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.common.types import misc as ctm
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common.yabs.server.requestlog import get_statistics
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.response_tools.unpacker import uc_compress, find_unpacker_resource
from sandbox.projects.yabs.qa.resource_types import YabsServerResponseDump
from sandbox.projects.yabs.qa.tasks.YabsServerStatPerformance2 import (
    YabsServerStatPerformance2,
    MB_IN_GB
)
from sandbox.projects.yabs.qa.tasks.YabsServerStatPerformanceBest2 import (
    YabsServerStatPerformanceBest2,
)
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox import YabsMetaSandboxAdapter, YabsStatSandboxAdapter
from sandbox.projects.yabs.qa.ammo_module.dplan.adapters.sandbox import AmmoDplanModuleSandboxAdapter
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.yabs_stat.sandbox import AmmoRequestlogModuleYabsStatSandboxAdapter
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.yabs_specific.sandbox import AmmoRequestlogModuleYabsStatPerformanceSandboxAdapter
from sandbox.projects.yabs.qa.dolbilo_module.simple.adapters.sandbox import DolbiloModuleSandboxAdapter
from sandbox.projects.yabs.qa.performance.stat_test import prepare_request_log
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_DOLBILKA_PLAN, YABS_SERVER_REQUEST_LOG_GZ, YABS_SERVER_META_NETWORK_STATISTICS
from sandbox.projects.yabs.qa.utils.base import get_bin_bases_unpacked_size, get_max_unpacking_workers


logger = logging.getLogger(__name__)


class NestedList(sdk2.parameters.JSON):
    default_value = []

    @classmethod
    def cast(cls, value):
        value = sdk2.parameters.JSON.cast(value)
        if not isinstance(value, list):
            raise ValueError("This parameter supports only list")
        if any([not isinstance(item, list) for item in value]):
            raise ValueError("This parameter supports only nested lists")
        return value


class YabsServerStatPerformancePrepareDplan(YabsServerStatPerformance2):
    """ Creates request log resource for future use in stat performance pipeline
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.YABS & ctc.Tag.SSD

    class Parameters(YabsServerStatPerformanceBest2.Parameters):
        kill_timeout = 2 * 60 * 60  # 2h

        with sdk2.parameters.Group("Version and task resource") as version_and_task_resource:
            auto_search = BaseBinTask.Parameters.auto_search(default=True)
            resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_bundle": "yabs_server_stat_load"})
            release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)
        meta_module_parameters = YabsMetaSandboxAdapter.get_init_parameters_class()()

        sort_requestlog = sdk2.parameters.Bool('Sort request log for better stability', default=False)
        split_handlers = NestedList('Split dplan by this handlers', default_value=[['pmatch']])
        with sdk2.parameters.Output:
            dplan_resource = sdk2.parameters.Resource('Resource with prepared dplan for stat shoot session')
            splitted_dplan_resources = sdk2.parameters.Resource('Resources with prepared splitted dplan for stat shoot session', multiple=True)
            usage_data_meta = sdk2.parameters.JSON('Data usage info for meta')
            meta_network_statistics = sdk2.parameters.Resource(
                'Resource with network statistics for meta',
                resource_type=YABS_SERVER_META_NETWORK_STATISTICS,
            )

    class Context(YabsServerStatPerformance2.Context):
        need_2_on_1 = True
        use_2on1_in_cmp_task = True

    @property
    def dump_parser_resource(self):
        dump_parser_resource = find_unpacker_resource(
            global_key=self.Parameters.meta_server_resource.global_key,
            global_key_type=self.Parameters.meta_server_resource.global_key_type,
            build_mode="release",
        )
        return dump_parser_resource

    def save_shoot_dump_resource(self, dump_path, shoot_index=0, pack_type='lz4hc'):
        logging.debug("Compress shoot dump")
        compressed_dump_path = uc_compress(self, dump_path, pack_type=pack_type)

        logging.debug("Create shoot dump resources")
        dump_resource = YabsServerResponseDump(
            self,
            'Shoot dump resource',
            compressed_dump_path,
            pack_codec=pack_type,
            shoot_index=shoot_index,
            debug_mode='proto_binary',
            dump_parser_id=self.dump_parser_resource.id,
            ammo_type=self.Parameters.meta_mode,
        )
        sdk2.ResourceData(dump_resource).ready()

    def prepare_stat_dplan(self, parameters, shoot_module, stat_index, shared_base_state=None, cachedaemon_data_dir=None, use_sandbox_config=True, index=1):
        with self.stage_duration("create_modules"):
            stat_adapter = YabsStatSandboxAdapter(
                parameters,
                task_instance=self,
                work_dir="prepare_dplan_stat_server_adapter_{}".format(index),
            )
            cachedaemon = stat_adapter.create_cachedaemon(cachedaemon_data_dir=cachedaemon_data_dir)
            stat_module = stat_adapter.create_module(
                cachedaemon=cachedaemon,
                shard_no=stat_index,
                shared_base_state=shared_base_state,
                maximum_keys_per_tag=1,
                use_sandbox_config=use_sandbox_config,
                max_unpacking_workers=self.Context.unpacking_workers,
            )

            meta_adapter = YabsMetaSandboxAdapter(
                parameters,
                task_instance=self,
                work_dir="prepare_dplan_meta_server_adapter_{}".format(index),
            )
            meta_module = meta_adapter.create_module(
                cachedaemon=cachedaemon,
                stat_instances=stat_module.get_instances(),
                shared_base_state=stat_module.shared_base_state,
                maximum_keys_per_tag=1,
                use_sandbox_config=use_sandbox_config,
                max_unpacking_workers=self.Context.unpacking_workers,
            )

        if parameters.use_requestlog:
            ammo_module = AmmoRequestlogModuleYabsStatPerformanceSandboxAdapter(parameters, self).create_module()
        else:
            ammo_module = AmmoDplanModuleSandboxAdapter(parameters, self).create_module()

        dplan_path = ammo_module.get_dplan_path()
        with cachedaemon, stat_module, meta_module, meta_module.get_usage_data() as usage_data_meta:
            dump_path = shoot_module.shoot_and_watch(meta_module, dplan_path)
            stat = stat_module.get_server_backend_object()
        request_log_path = stat.get_phantom_log_path(stat.LOG_REQUEST)
        request_log_path, handlers = prepare_request_log.do_sort_request_log(request_log_path)

        save_shoot_dump_resource = self.Parameters.store_dumps
        try:
            prepare_request_log.check_meta_access_log(meta_module)
            prepare_request_log.check_cachedaemon_access_log(meta_module.cachedaemon)
        except Exception:
            save_shoot_dump_resource = True
            raise
        finally:
            if save_shoot_dump_resource:
                self.save_shoot_dump_resource(dump_path, shoot_index=0)

        stat_ammo_module = AmmoRequestlogModuleYabsStatSandboxAdapter(parameters, self, request_log_path).create_module()

        dplan_paths = {}
        request_counts = {}
        for split_handlers in self.Parameters.split_handlers:
            splitted_request_log_path, request_count = prepare_request_log.split_request_log(request_log_path, split_handlers)
            if not request_count:
                self.set_info('Did not find any requests to {} handlers'.format(self.handlers_to_string(split_handlers)))
                continue
            request_counts[self.handlers_to_string(split_handlers)] = request_count
            dplan_paths[self.handlers_to_string(split_handlers)] = AmmoRequestlogModuleYabsStatSandboxAdapter(
                parameters,
                self,
                splitted_request_log_path
            ).create_module().get_dplan_path()
            for handler in split_handlers:
                handlers.remove(handler)
        other_handlers_request_log_path, request_count = prepare_request_log.split_request_log(request_log_path, handlers)
        request_counts[self.handlers_to_string(handlers)] = request_count
        dplan_paths[self.handlers_to_string(handlers)] = AmmoRequestlogModuleYabsStatSandboxAdapter(
            parameters,
            self,
            other_handlers_request_log_path
        ).create_module().get_dplan_path()

        return (
            stat_ammo_module.get_dplan_path(),
            dplan_paths,
            request_log_path,
            request_counts,
            usage_data_meta,
            stat_module.shared_base_state
        )

    def handlers_to_string(self, handlers):
        return ','.join(sorted(handlers))

    def on_save(self):
        super(YabsServerStatPerformancePrepareDplan, self).on_save()
        self.Requirements.ram = self.Parameters.ram_space * MB_IN_GB
        shard_count = len(self.Parameters.stat_shards)
        bin_bases_unpacked_size = []
        if self.Parameters.use_tmpfs:
            bin_bases_unpacked_size = get_bin_bases_unpacked_size(self.Parameters.meta_binary_base_resources + self.Parameters.stat_binary_base_resources)
            self.Context.ramdrive_size = min((self.Parameters.shard_space * shard_count) * MB_IN_GB, sum(bin_bases_unpacked_size) + 10 * MB_IN_GB)
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                self.Context.ramdrive_size,
                None
            )
            self.Requirements.disk_space = self.Parameters.generic_disk_space * MB_IN_GB
        else:
            self.Requirements.ramdrive = None
            self.Requirements.disk_space = (self.Parameters.generic_disk_space + (self.Parameters.shard_space * shard_count)) * MB_IN_GB
        self.Context.unpacking_workers = get_max_unpacking_workers(bin_bases_unpacked_size, self.Requirements.ram, self.Context.ramdrive_size)

    @track_errors
    def on_execute(self):
        if self.Parameters.prepare_stat_dplan:
            self.set_info('Will prepare stat dplan for 2 on 1')
            stat_index = self.Parameters.stat_shards[0]
            shoot_module = DolbiloModuleSandboxAdapter(self.Parameters, self).create_module()

            dplan_path, splitted_dplan_paths, requestlog_path, request_counts, usage_data_meta, _ = self.prepare_stat_dplan(
                self.Parameters,
                shoot_module,
                stat_index,
                shared_base_state=None,
                use_sandbox_config=False
            )

            dplan_resource = YABS_SERVER_DOLBILKA_PLAN(
                self,
                'Stat dolbilka plan',
                dplan_path,
                meta_mode=self.Parameters.meta_mode,
                shard=stat_index,
            )
            sdk2.ResourceData(dplan_resource).ready()

            splitted_dplan_resources = []
            for handlers, path in splitted_dplan_paths.items():
                splitted_dplan_resource = YABS_SERVER_DOLBILKA_PLAN(
                    self,
                    'Stat dolbilka plan, {}'.format(handlers),
                    path,
                    meta_mode=self.Parameters.meta_mode,
                    shard=stat_index,
                    handlers=handlers,
                    request_count=request_counts[handlers],
                )
                sdk2.ResourceData(splitted_dplan_resource).ready()
                splitted_dplan_resources.append(splitted_dplan_resource)

            if requestlog_path is not None:
                requestlog_resource = YABS_SERVER_REQUEST_LOG_GZ(
                    self,
                    'Stat requestlog',
                    requestlog_path,
                    meta_mode=self.Parameters.meta_mode,
                    shard=stat_index,
                )
                requestlog_resource_data = sdk2.ResourceData(requestlog_resource)
                requestlog_resource_data.ready()

                network_statistics = get_statistics(requestlog_path, ignore_handlers=['ping'])
                network_statistics_resource = YABS_SERVER_META_NETWORK_STATISTICS(
                    self,
                    'meta network statistics',
                    'network_statistics.json',
                )
                network_statistics_resource_data = sdk2.ResourceData(network_statistics_resource)
                network_statistics_resource_data.path.write_text(unicode(json.dumps(network_statistics, indent=4)))
                network_statistics_resource_data.ready()
                try:
                    self.Parameters.meta_network_statistics = network_statistics_resource.id
                except AttributeError:
                    pass

            try:
                self.Parameters.dplan_resource = dplan_resource
            except AttributeError:
                pass
            try:
                self.Parameters.splitted_dplan_resources = splitted_dplan_resources
            except AttributeError:
                pass
            try:
                self.Parameters.usage_data_meta = usage_data_meta
            except AttributeError:
                pass
        else:
            self.set_info('Run as 2 on 1 dummy')
