import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.yabs.server.util.general import find_last_ready_resource
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.yabs_specific.sandbox.parameters import AmmoRequestlogYabsSpecificParameters
from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink
from sandbox.projects.yabs.qa.tasks.YabsServerValidateABExperiment import handler_bigb_and_has_bs_parameters, get_ab_experiment
from sandbox.projects.yabs.qa.tasks.YabsServerCreateABExperimentSpec.spec import YabsServerABExperimentSpec
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_AB_EXPERIMENT_SETTINGS

from sandbox.projects.yabs.qa.spec.parameters import ShootParameters
from sandbox.projects.yabs.qa.pipeline_test_framework.stages import (
    get_shard_map,
    launch_stat_load_shoot_tasks,
    launch_meta_load_shoot_tasks,
    launch_stat_load_cmp_tasks,
    launch_meta_load_cmp_tasks,
)
from sandbox.projects.yabs.qa.tasks.YabsServerTestABExperiment.stages import (
    get_spec_data,
    get_ab_experiment_settings,
    launch_ab_experiment_validation_task,
    check_ab_experiment_validation_task,
    launch_base_producing_tasks,
    get_binary_bases,

    launch_ft_shoot_tasks,
    launch_ft_cmp_tasks,
    launch_ft_cmp_painted_tasks,

    set_version_report_data,
    set_ft_report_data,
    set_stat_load_report_data,
    set_meta_load_report_data,
)
from sandbox.projects.yabs.qa.tasks.YabsServerTestABExperiment.report import (
    create_report,
    TEST_REPORT_TEMPLATE,
    VERSION_REPORT_TEMPLATE,
)


logger = logging.getLogger(__name__)

_STAGES = [
    get_spec_data,
    get_shard_map,
    get_ab_experiment_settings,
    launch_ab_experiment_validation_task,
    check_ab_experiment_validation_task,
    launch_base_producing_tasks,
    get_binary_bases,
    launch_ft_shoot_tasks,
    launch_stat_load_shoot_tasks,
    launch_meta_load_shoot_tasks,
    launch_ft_cmp_tasks,
    launch_ft_cmp_painted_tasks,
    launch_stat_load_cmp_tasks,
    launch_meta_load_cmp_tasks,
    set_version_report_data,
    set_ft_report_data,
    set_stat_load_report_data,
    set_meta_load_report_data,
]


class YabsServerTestABExperiment(sdk2.Task):
    """New task for AB-experiment tests"""

    name = 'YABS_SERVER_TEST_AB_EXPERIMENT'

    class Requirements(sdk2.Requirements):
        ram = 4 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        push_tasks_resource = True

        validate_ab_experiment = sdk2.parameters.Bool('Run YABS_SERVER_VALIDATE_AB_EXPERIMENT', default=True)
        ab_experiment_settings = AmmoRequestlogYabsSpecificParameters.ab_experiment_settings()
        ab_experiment_settings_resource = sdk2.parameters.Resource('Resource with json AB experiment settings', resource_type=YABS_SERVER_AB_EXPERIMENT_SETTINGS)
        hash_type = sdk2.parameters.RadioGroup(
            'TESTID hash type',
            choices=(
                ('REQID', 'HT_REQID'),
                ('ADSUSERID', 'HT_ADS_USER_ID'),
                ('CUID', 'HT_CUID'),
                ('YUID', 'HT_YUID'),
                ('ICOOKIE', 'HT_ICOOKIE'),
                ('CRYPTAID', 'HT_CRYPTAID'),
                ('NO_IDENTIFIER', ''),
            ),
            default='HT_REQID',
            required=False,
        )

        ab_experiment_spec = sdk2.parameters.Resource('AB experiment spec', resource_type=YabsServerABExperimentSpec, default=None, do_not_copy=True)
        use_ab_experiment_db = sdk2.parameters.Bool('Generate ab_experiment binary base with AB-experiment configs', default=True)
        download_latest_bases = sdk2.parameters.Bool('Download latest bases from production and ignore stability checks', default=False)
        dry_run = sdk2.parameters.Bool('Dry run', default=False)
        with download_latest_bases.value[True]:
            bases_to_download = sdk2.parameters.JSON('List of bases to download', default=['formula'])

        shoot_parametars = ShootParameters()

        with sdk2.parameters.Group('Developer options') as developer_options:
            debug_mode = sdk2.parameters.Bool('Debug mode', default=False)
            with debug_mode.value[True]:
                reusable_stages = sdk2.parameters.CheckGroup(
                    'Reusable stages',
                    choices=[
                        (_stage.__name__, _stage.__name__)
                        for _stage in _STAGES
                    ],
                    default=None,
                )

        with sdk2.parameters.Output():
            cmp_result = sdk2.parameters.JSON("CMP child tasks results")

    def check_ab_experiment_for_content_autobudget_experiment_id(self):
        ab_experiments = get_ab_experiment(self.Parameters.ab_experiment_settings_resource, self.Parameters.ab_experiment_settings)
        for exp in ab_experiments:
            try:
                settings = exp['CONTEXT']['MAIN']['BIGB_MAIN']
                if settings.get("AutobudgetSettings") and settings["AutobudgetSettings"].get("AutobudgetExperimentID"):
                    return True
            except KeyError:
                continue
        return False

    def on_save(self):
        if any(
            map(handler_bigb_and_has_bs_parameters, self.Parameters.ab_experiment_settings) + [self.Parameters.stat_load, self.Parameters.meta_load]
        ):
            self.Parameters.use_ab_experiment_db = True

        if self.author in self.server.group['YABS_SERVER_SANDBOX_TESTS'].read()["members"]:
            self.Parameters.owner = 'YABS_SERVER_SANDBOX_TESTS'

        if self.Parameters.dry_run:
            self.Parameters.tags = self.Parameters.tags + ['dry_run']

    def on_enqueue(self):
        if not self.Parameters.ab_experiment_spec:
            attributes_to_search = {
                'released_spec': True,
                'ft': True,
            }
            if self.Parameters.stat_load:
                attributes_to_search['stat_load'] = True
            if self.Parameters.meta_load:
                attributes_to_search['meta_load'] = True
            self.Parameters.ab_experiment_spec = find_last_ready_resource(YabsServerABExperimentSpec, attributes_to_search)

    def on_execute(self):
        if self.Parameters.dry_run:
            return

        if self.check_ab_experiment_for_content_autobudget_experiment_id():
            msg = "Won't test, settings contain AutobudgetExperimentID"
            self.set_info(msg)
            return

        if self.Parameters.debug_mode:
            self.Context.__reusable_stages = self.Parameters.reusable_stages

        with self.memoize_stage.set_tags:
            self.Parameters.tags = list(set(self.Parameters.tags + ['AB-EXPERIMENT-TEST']))

        get_spec_data(self)

        # Report version asap
        version_report_data = set_version_report_data(self)
        with self.memoize_stage.set_version_report_data:
            self.set_info(create_report(version_report_data, VERSION_REPORT_TEMPLATE), do_escape=False)

        get_shard_map(
            self,
            ft_shards=self.Parameters.ft_shards,
            stat_load=self.Parameters.stat_load,
            stat_load_shards=self.Parameters.stat_load_shards,
            meta_load=self.Parameters.meta_load,
            meta_load_shards=self.Parameters.meta_load_shards,
        )

        # parameters
        get_ab_experiment_settings(self)

        # Validate
        launch_ab_experiment_validation_task(self, validate_ab_experiment=self.Parameters.validate_ab_experiment, ab_experiment_settings_resource=self.Parameters.ab_experiment_settings_resource)
        check_ab_experiment_validation_task(self, validate_ab_experiment=self.Parameters.validate_ab_experiment)

        # Base gen
        launch_base_producing_tasks(
            self,
            use_ab_experiment_db=self.Parameters.use_ab_experiment_db,
            download_latest_bases=self.Parameters.download_latest_bases,
            bases_to_download=self.Parameters.bases_to_download,
        )
        get_binary_bases(self)

        # Shoot
        launch_ft_shoot_tasks(self, meta_modes=self.Parameters.ft_meta_modes, test_description=self.Parameters.description, shoot_baseline_tasks=self.Parameters.download_latest_bases)
        launch_stat_load_shoot_tasks(self, meta_modes=self.Parameters.stat_load_meta_modes, stat_load=self.Parameters.stat_load, test_description=self.Parameters.description)
        launch_meta_load_shoot_tasks(self, meta_modes=self.Parameters.meta_load_meta_modes, meta_load=self.Parameters.meta_load, test_description=self.Parameters.description)

        # CMP
        launch_ft_cmp_tasks(self, ttl_for_dump_with_diff=4)
        launch_ft_cmp_painted_tasks(self, ttl_for_dump_with_diff=4)
        launch_stat_load_cmp_tasks(self, stat_load=self.Parameters.stat_load, test_description=self.Parameters.description)
        launch_meta_load_cmp_tasks(self, meta_load=self.Parameters.meta_load, test_description=self.Parameters.description)

        # Report

        cmp_result = []
        failed_cmp_tasks = []

        failed_ft_cmp_tasks, ft_report_data = set_ft_report_data(self)
        failed_cmp_tasks.extend(failed_ft_cmp_tasks)
        if ft_report_data:
            cmp_result.extend(ft_report_data)
        with self.memoize_stage.set_ft_report_data:
            self.set_info(create_report(ft_report_data, TEST_REPORT_TEMPLATE), do_escape=False)

        failed_stat_load_cmp_tasks, stat_load_report_data = set_stat_load_report_data(self, stat_load=self.Parameters.stat_load)
        failed_cmp_tasks.extend(failed_stat_load_cmp_tasks)
        if stat_load_report_data:
            cmp_result.extend(stat_load_report_data)
            with self.memoize_stage.set_stat_load_report_data:
                self.set_info(create_report(stat_load_report_data, TEST_REPORT_TEMPLATE), do_escape=False)

        failed_meta_load_cmp_tasks, meta_load_report_data = set_meta_load_report_data(self, meta_load=self.Parameters.meta_load)
        failed_cmp_tasks.extend(failed_meta_load_cmp_tasks)
        if meta_load_report_data:
            cmp_result.extend(meta_load_report_data)
            with self.memoize_stage.set_meta_load_report_data:
                self.set_info(create_report(meta_load_report_data, TEST_REPORT_TEMPLATE), do_escape=False)

        with self.memoize_stage.set_cmp_result_output_param(commit_on_entrance=False):
            self.Parameters.cmp_result = cmp_result

        # Final check
        if failed_cmp_tasks:
            msg = 'Following tasks failed: ' + ', '.join([get_task_html_hyperlink(t) for t in failed_cmp_tasks])
            self.set_info(msg, do_escape=False)
            raise TaskFailure('Tasks {} failed'.format(failed_cmp_tasks))
