from jinja2 import Environment

from sandbox import sdk2
from sandbox.common.types.task import Status as TaskStatus
from sandbox.common.utils import get_task_link
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp import YabsServerB2BFuncShootCmp
from sandbox.projects.yabs.qa.tasks.YabsServerPerformanceMetaCmp import YabsServerPerformanceMetaCmp
from sandbox.projects.yabs.qa.tasks.YabsServerStatPerformanceBestCmp2 import YabsServerStatPerformanceBestCmp2

SB_STYLE = '''.table {
    white-space: normal;
}
.table table {
    border-collapse: collapse; }
.table th {
    padding: 2px 8px;
    text-align: left;
    vertical-align: top;
    background-color: #f2f2f2;
    border: 1px solid #ddd;
}
.table tr {
    padding: 2px 8px;
    text-align: left;
    vertical-align: top;
}
.table td {
    padding: 2px 8px;
    border: 1px solid #ddd;
}
.table pre {
    white-space: pre-wrap;
}'''


class Status(object):
    OK = "OK"
    FAIL = "FAIL"


COLORS_BY_STATUS = {
    Status.FAIL: "red",
    Status.OK: "green",
}

VERSION_REPORT_TEMPLATE = '''
<style>{{ style }}</style>
<div class="table">
<h2>Stat server version</h2>
<pre>{{ report_data.stat_server }}</pre>
<h2>Meta server version</h2>
<pre>{{ report_data.meta_server }}</pre>
</div>
'''

TEST_REPORT_TEMPLATE = '''
<style>{{ style }}</style>
<div class="table">
{% for report in report_data -%}
<h2>{{ report.section_name }} tests</h2>
<table>
    <thead>
        <tr>
            <th>Role/Shard</th>
            <th>Status</th>
            <th>Result</th>
            <th>Ammo created</th>
        </tr>
    </thead>
    <tbody>
    {% for row in report.rows -%}
        <tr>
            <td>{{ row.role }}/{{ row.shard }}</td>
            <td><a href="{{ row.cmp_task_link }}" target="_blank">{{ row.cmp_task_id }}</a>&nbsp;<span style="color: {{ color_map.get(row.cmp_status) }}">{{ row.cmp_status }}</span></td>
            <td><a href="{{ row.cmp_report_link }}" target="_blank">{{ row.cmp_report_text }}</a></td>
            <td>Created at {{ row.ammo_creation_time }}</a></td>
        </tr>
    {% endfor -%}
    </tbody>
</table>
{% endfor -%}
'''


def _get_cmp_task_data(cmp_task_id):
    cmp_task = sdk2.Task[cmp_task_id]
    cmp_task_ctx = cmp_task.Context
    log_stat_diff_resourse_id = None
    has_diff = getattr(cmp_task_ctx, 'has_diff', True)
    short_report_text = 'Diff' if has_diff else 'No diff'
    short_report_link = get_task_link(cmp_task_id)
    if cmp_task.status != TaskStatus.SUCCESS:
        return True, 'CMP task failed', short_report_link, None
    if isinstance(cmp_task, (YabsServerB2BFuncShootCmp, YabsServerPerformanceMetaCmp)):
        if isinstance(cmp_task, YabsServerB2BFuncShootCmp):
            log_stat_diff_resourse_id = cmp_task.Parameters.log_stat_diff_resource_id
        short_report_text = getattr(cmp_task_ctx, 'short_report_text')
        short_report_link = getattr(cmp_task_ctx, 'short_report_link')
    elif isinstance(cmp_task, (YabsServerStatPerformanceBestCmp2, )):
        short_report_text = 'Max requests/sec changed by {diff_percent:+.2f}% ({pre_rps:.2f} -> {test_rps:.2f})'.format(
            diff_percent=(cmp_task_ctx.rps_hl_median_2 - cmp_task_ctx.rps_hl_median) * 100 / cmp_task_ctx.rps_hl_median,
            pre_rps=cmp_task_ctx.rps_hl_median,
            test_rps=cmp_task_ctx.rps_hl_median_2,
        )
    return has_diff, short_report_text, short_report_link, log_stat_diff_resourse_id


def _create_report_data(cmp_tasks, ammo_resources, section_name):
    report_data = {
        'section_name': section_name,
        'rows': []
    }
    for meta_mode, meta_mode_cmp_tasks in cmp_tasks.items():
        if meta_mode in ammo_resources:
            ammo_creation_time = sdk2.Resource[ammo_resources[meta_mode]].created.strftime('%Y-%m-%d %H:%M:%S')
        else:
            ammo_creation_time = 'unknown time'
        for shard_num, cmp_task_id in cmp_tasks[meta_mode].items():
            cmp_task_url = get_task_link(cmp_task_id)
            has_diff, short_report_text, short_report_link, log_stat_diff_resource_id = _get_cmp_task_data(cmp_task_id)
            report_data['rows'].append({
                "role": meta_mode,
                "shard": shard_num,
                "cmp_task_link": cmp_task_url,
                "cmp_task_id": cmp_task_id,
                "cmp_report_link": short_report_link,
                "cmp_report_text": short_report_text,
                "cmp_status": Status.FAIL if has_diff else Status.OK,
                "ammo_creation_time": ammo_creation_time,
                "log_stat_diff_resource_id": log_stat_diff_resource_id,
            })

    return report_data


def _create_meta_load_report_data(cmp_tasks, ammo_resources, meta_load_shard_nums):
    report_data = {
        'section_name': 'Meta load',
        'rows': []
    }
    for meta_mode, cmp_task_id in cmp_tasks.items():
        if meta_mode in ammo_resources:
            ammo_creation_time = sdk2.Resource[ammo_resources[meta_mode]].created.strftime('%Y-%m-%d %H:%M:%S')
        else:
            ammo_creation_time = 'unknown time'

        cmp_task_url = get_task_link(cmp_task_id)
        has_diff, short_report_text, short_report_link, log_stat_diff_resource_id = _get_cmp_task_data(cmp_task_id)

        report_data['rows'].append({
            "role": meta_mode,
            "shard": '/'.join(meta_load_shard_nums),
            "cmp_task_link": cmp_task_url,
            "cmp_task_id": cmp_task_id,
            "cmp_report_link": short_report_link,
            "cmp_report_text": short_report_text,
            "cmp_status": Status.FAIL if has_diff else Status.OK,
            "ammo_creation_time": ammo_creation_time,
            "log_stat_diff_resource_id": log_stat_diff_resource_id,
        })

    return report_data


def create_report(report_data, report_template):
    env = Environment()
    html_report_template = env.from_string(report_template)
    return html_report_template.render(report_data=report_data, style=SB_STYLE, color_map=COLORS_BY_STATUS)
