import json
import logging
from itertools import chain
from copy import deepcopy

from sandbox import sdk2
from sandbox.common.types.task import Status as TaskStatus

from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.qa.pipeline.stage import stage, memoize_stage
from sandbox.projects.yabs.qa.spec.constants import META_MODES
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.default_paint import (
    AB_LOG_TRANSFORMATION,
    AB_RESPONSE_TRANSFORMATION,
    AB_EXT_TRANSFORMATION,
    AB_EXT_IGNORES,
)
from sandbox.projects.yabs.qa.tasks.YabsServerValidateABExperiment import YabsServerValidateAbExperiment, BS_AB_HANDLERS
from sandbox.projects.yabs.qa.tasks.YabsServerGenerateConfigWithNewAbExperiment import YabsServerGenerateConfigWithNewAbExperiment
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShoot2 import YabsServerB2BFuncShoot2
from sandbox.projects.yabs.qa.tasks.YabsServerGetProdBases import YabsServerGetProdBases
from sandbox.projects.yabs.qa.tasks.YabsServerCreateABExperimentSpec.spec import ABExperimentSpec

from sandbox.projects.yabs.qa.tasks.YabsServerTestABExperiment.report import (
    _create_report_data,
    _create_meta_load_report_data,
)
from sandbox.projects.yabs.qa.pipeline_test_framework.helpers import (
    _launch_task,
    _launch_shoot_tasks,
)
from sandbox.projects.yabs.qa.tasks.YabsServerTestABExperiment.helpers import (
    _launch_ft_cmp_tasks,
)


logger = logging.getLogger(__name__)


SPEC_CONTEXT_FIELDS = tuple('spec_{}'.format(k) for k in ABExperimentSpec.__slots__)


@stage(provides=SPEC_CONTEXT_FIELDS, result_is_dict=True)
def get_spec_data(task):
    ab_experiment_spec = sdk2.ResourceData(task.Parameters.ab_experiment_spec)
    with open(str(ab_experiment_spec.path), 'r') as spec_file:
        ab_experiment_spec_data = json.load(spec_file)
    return {
        'spec_{}'.format(key): ab_experiment_spec_data.get(key)
        for key in ABExperimentSpec.__slots__
    }


@stage(provides='ab_experiment_validation_task_id')
def launch_ab_experiment_validation_task(task, base_gen_ab_experiment_settings, spec_validate_ab_experiment_bin_id,
                                         validate_ab_experiment=False, ab_experiment_settings_resource=None):
    if validate_ab_experiment:
        return _launch_task(
            task,
            YabsServerValidateAbExperiment,
            ab_experiment=base_gen_ab_experiment_settings or {},
            ab_experiment_resource=ab_experiment_settings_resource,
            validate_ab_experiment_bin_id=spec_validate_ab_experiment_bin_id,
        )
    return -1


@stage(provides='validation_passed')
def check_ab_experiment_validation_task(task, ab_experiment_validation_task_id, validate_ab_experiment=False):
    if validate_ab_experiment:
        check_tasks(task, [ab_experiment_validation_task_id])
    return True


def _launch_base_producing_tasks(
        task,
        cs_input_spec_resource_id,
        cs_settings_archive_resource_id,
        mysql_archive_resource_id,
        bs_release_yt_resource_id,
        bs_release_tar_resource_id,
        setup_ya_make_task_id,
        drop_other_experiments,
        base_gen_ab_experiment_settings,
        use_ab_experiment_db=True,
        download_latest_bases=False,
        bases_to_download=(),
        bsfrontend_dir='//home/yabs-transport/transport/bsfrontend_dir',
):
    base_producing_task_ids = []
    if use_ab_experiment_db:
        params = dict(
            description='Make ab_experiment | {}'.format(task.Parameters.description),
            ab_experiment=base_gen_ab_experiment_settings,
            ab_experiment_resource=task.Parameters.ab_experiment_settings_resource,
            hash_type=task.Parameters.hash_type,
            ab_handlers=BS_AB_HANDLERS,
            mbb_params={
                'input_spec': cs_input_spec_resource_id,
                'settings_archive': cs_settings_archive_resource_id,
                'mysql_archive_contents': mysql_archive_resource_id,
                'bs_release_yt_resource': bs_release_yt_resource_id,
                'server_resource': bs_release_tar_resource_id,
                'use_cs_cycle': True,
                'use_save_input_from_cs': False,
                'filter_input_archive_tables_by_orderid': False,
                'save_all_inputs': False,
                'yt_pool': 'yabs-cs-sandbox-ab-experiment',
            },
            release_version='stable',
            drop_other_experiments=drop_other_experiments,
        )
        if task.Requirements.tasks_resource:
            params.update({
                "auto_search": False,
                "__requirements__": {
                    "tasks_resource": task.Requirements.tasks_resource,
                }
            })
        generate_ab_config_task_id = _launch_task(task, YabsServerGenerateConfigWithNewAbExperiment, **params)
        base_producing_task_ids.append(generate_ab_config_task_id)
    if download_latest_bases:
        base_download_task_id = _launch_task(
            task,
            YabsServerGetProdBases,
            description='Download bases: {} | {}'.format(', '.join(sorted(list(task.Parameters.bases_to_download))), task.Parameters.description),
            base_ver=sdk2.Task[setup_ya_make_task_id].Context.db_ver,
            bases=bases_to_download,
            yt_path=bsfrontend_dir,
            __requirements__={'tasks_resource': task.Requirements.tasks_resource},
        )
        base_producing_task_ids.append(base_download_task_id)

    return base_producing_task_ids


@stage(provides=(
    'stat_base_producing_task_ids',
    'meta_base_producing_task_ids',
))
def launch_base_producing_tasks(
        task,
        spec_cs_input_spec_resource_id,
        spec_cs_settings_archive_resource_id,
        spec_mysql_archive_resource_id,
        spec_stat_bs_release_yt_resource_id,
        spec_stat_bs_release_tar_resource_id,
        spec_meta_bs_release_yt_resource_id,
        spec_meta_bs_release_tar_resource_id,
        spec_stat_setup_ya_make_task_id,
        spec_meta_setup_ya_make_task_id,
        spec_drop_other_experiments,
        spec_stat_base_tags_map,
        spec_meta_base_tags_map,
        base_gen_ab_experiment_settings,
        use_ab_experiment_db=True,
        download_latest_bases=False,
        bases_to_download=(),
):
    stat_bases_to_download = []
    meta_bases_to_download = []
    if download_latest_bases and bases_to_download:
        stat_bases_to_download = set(bases_to_download) & set(chain(*spec_stat_base_tags_map.values()))
        meta_bases_to_download = set(bases_to_download) & set(chain(*spec_meta_base_tags_map.values()))

    return (
        _launch_base_producing_tasks(
            task,
            spec_cs_input_spec_resource_id,
            spec_cs_settings_archive_resource_id,
            spec_mysql_archive_resource_id,
            spec_stat_bs_release_yt_resource_id,
            spec_stat_bs_release_tar_resource_id,
            spec_stat_setup_ya_make_task_id,
            spec_drop_other_experiments,
            base_gen_ab_experiment_settings,
            use_ab_experiment_db=use_ab_experiment_db,
            download_latest_bases=download_latest_bases,
            bases_to_download=stat_bases_to_download,
            bsfrontend_dir='//home/yabs-transport/transport/bsfrontend_dir',
        ),
        _launch_base_producing_tasks(
            task,
            spec_cs_input_spec_resource_id,
            spec_cs_settings_archive_resource_id,
            spec_mysql_archive_resource_id,
            spec_meta_bs_release_yt_resource_id,
            spec_meta_bs_release_tar_resource_id,
            spec_meta_setup_ya_make_task_id,
            spec_drop_other_experiments,
            base_gen_ab_experiment_settings,
            use_ab_experiment_db=use_ab_experiment_db,
            download_latest_bases=download_latest_bases,
            bases_to_download=meta_bases_to_download,
            bsfrontend_dir='//home/yabs-transport/transport/bsfrontend_meta_dir',
        ),
    )


def _get_binary_bases(
        task,
        binary_base_resource_id_by_tag,
        base_producing_task_ids,
):
    baseline_binary_base_resource_id_by_tag = deepcopy(binary_base_resource_id_by_tag)
    test_binary_base_resource_id_by_tag = deepcopy(binary_base_resource_id_by_tag)

    if base_producing_task_ids:
        check_tasks(task, base_producing_task_ids)
        for task_id in base_producing_task_ids:
            task = sdk2.Task[task_id]
            if isinstance(task, YabsServerGenerateConfigWithNewAbExperiment):
                logger.info('Got YabsServerGenerateConfigWithNewAbExperiment, will update only test base map')
                test_binary_base_resource_id_by_tag['ab_experiment'] = task.Context.ab_experiment_db
            elif isinstance(task, YabsServerGetProdBases):
                logger.info('Got YabsServerGetProdBases, will update both test and baseline base maps')
                test_binary_base_resource_id_by_tag.update(task.Parameters.base_resources)
                baseline_binary_base_resource_id_by_tag.update(task.Parameters.base_resources)

    return baseline_binary_base_resource_id_by_tag, test_binary_base_resource_id_by_tag


@stage(provides=(
    'baseline_stat_binary_base_resource_id_by_tag',
    'test_stat_binary_base_resource_id_by_tag',
    'baseline_meta_binary_base_resource_id_by_tag',
    'test_meta_binary_base_resource_id_by_tag',
))
def get_binary_bases(
        task,
        spec_stat_binary_base_resource_id_by_tag,
        spec_meta_binary_base_resource_id_by_tag,
        stat_base_producing_task_ids,
        meta_base_producing_task_ids,
):
    return tuple(chain(
        _get_binary_bases(task, spec_stat_binary_base_resource_id_by_tag, stat_base_producing_task_ids),
        _get_binary_bases(task, spec_meta_binary_base_resource_id_by_tag, meta_base_producing_task_ids),
    ))


@stage(provides=('base_gen_ab_experiment_settings', 'shoot_ab_experiment_settings'))
def get_ab_experiment_settings(task):
    base_gen_ab_experiment_settings = task.Parameters.ab_experiment_settings
    for data in base_gen_ab_experiment_settings:
        if data.get('HANDLER') in BS_AB_HANDLERS:
            data.pop('RESTRICTIONS', None)  # BSSERVER-14125 - shoot without RESTRICTIONS
    shoot_ab_experiment_settings = {} if task.Parameters.use_ab_experiment_db else base_gen_ab_experiment_settings
    return base_gen_ab_experiment_settings, shoot_ab_experiment_settings


@stage(provides=('baseline_ft_shoot_tasks', 'test_ft_shoot_tasks'))
def launch_ft_shoot_tasks(
        task,
        ft_shard_keys,
        shard_map,
        spec_ft_shoot_tasks,
        spec_stat_base_tags_map,
        spec_meta_base_tags_map,
        test_stat_binary_base_resource_id_by_tag,
        baseline_stat_binary_base_resource_id_by_tag,
        test_meta_binary_base_resource_id_by_tag,
        baseline_meta_binary_base_resource_id_by_tag,
        shoot_ab_experiment_settings,
        shoot_baseline_tasks=False,
        meta_modes=META_MODES,
        test_description='',
):
    ft_shoot_tasks = {'test': {}, 'baseline': {}}
    run_type_parameters = (
        (
            'test',
            test_stat_binary_base_resource_id_by_tag,
            test_meta_binary_base_resource_id_by_tag,
            {'ab_experiment_settings': shoot_ab_experiment_settings} if shoot_ab_experiment_settings else {},
        ),
        (
            'baseline',
            baseline_stat_binary_base_resource_id_by_tag,
            baseline_meta_binary_base_resource_id_by_tag,
            {},
        ),
    )
    for run_type, stat_binary_base_resource_id_by_tag, meta_binary_base_resource_id_by_tag, additional_parameters in run_type_parameters:
        if run_type == 'baseline' and not shoot_baseline_tasks:
            ft_shoot_tasks[run_type] = spec_ft_shoot_tasks
            continue
        ft_shoot_tasks[run_type] = _launch_shoot_tasks(
            task,
            ft_shard_keys,
            shard_map,
            spec_ft_shoot_tasks,
            spec_stat_base_tags_map,
            stat_binary_base_resource_id_by_tag,
            spec_meta_base_tags_map,
            meta_binary_base_resource_id_by_tag,
            description='{} ft shoot'.format(run_type),
            parent_description=test_description,
            task_type=YabsServerB2BFuncShoot2,
            meta_modes=meta_modes,
            additional_parameters=additional_parameters,
        )

    return ft_shoot_tasks['baseline'], ft_shoot_tasks['test']


@stage(provides='ft_cmp_tasks')
def launch_ft_cmp_tasks(task, baseline_ft_shoot_tasks, test_ft_shoot_tasks, ttl_for_dump_with_diff=4):
    return _launch_ft_cmp_tasks(
        task,
        baseline_ft_shoot_tasks,
        test_ft_shoot_tasks,
        cmp_task_update_parameters={
            'ttl_for_dump_with_diff': ttl_for_dump_with_diff,
        },
    )


@stage(provides='ft_cmp_painted_tasks')
def launch_ft_cmp_painted_tasks(task, baseline_ft_shoot_tasks, test_ft_shoot_tasks, ttl_for_dump_with_diff=4):
    return _launch_ft_cmp_tasks(
        task,
        baseline_ft_shoot_tasks,
        test_ft_shoot_tasks,
        cmp_task_update_parameters={
            'user_substitutes_new': AB_RESPONSE_TRANSFORMATION,
            'log_ignores_new': AB_LOG_TRANSFORMATION,
            'ext_substitutes': AB_EXT_TRANSFORMATION,
            'ext_ignores': AB_EXT_IGNORES,
            'ttl_for_dump_with_diff': ttl_for_dump_with_diff,
        },
        description='with additional AB-experiment paints',
    )


@memoize_stage()
def set_version_report_data(
        task,
        spec_stat_bs_release_tar_resource_id, spec_stat_bs_release_yt_resource_id,
        spec_meta_bs_release_tar_resource_id, spec_meta_bs_release_yt_resource_id,
):
    from sandbox.projects.yabs.release.version.sandbox_helpers import SandboxHelper
    sandbox_helper = SandboxHelper()

    report_data = {
        "stat_server": sandbox_helper.get_full_version_from_resource(spec_stat_bs_release_tar_resource_id),
        "meta_server": sandbox_helper.get_full_version_from_resource(spec_meta_bs_release_tar_resource_id),
        "stat_cs": sandbox_helper.get_full_version_from_resource(spec_stat_bs_release_yt_resource_id),
        "meta_cs": sandbox_helper.get_full_version_from_resource(spec_meta_bs_release_yt_resource_id),
    }

    return report_data


@stage(provides=('failed_ft_cmp_tasks', 'ft_report_data'))
def set_ft_report_data(task, spec_ft_request_log_resource_id_map, ft_cmp_tasks, ft_cmp_painted_tasks):
    task_statuses = check_tasks(
        task,
        list(chain(
            *([
                _cmp_tasks.values() for _cmp_tasks in ft_cmp_tasks.values()
            ] + [
                _painted_tasks.values() for _painted_tasks in ft_cmp_painted_tasks.values()
            ])
        )),
        raise_on_fail=False,
    )
    report_data = [
        _create_report_data(ft_cmp_tasks, spec_ft_request_log_resource_id_map, 'Functional'),
        _create_report_data(ft_cmp_painted_tasks, spec_ft_request_log_resource_id_map, 'Functional with AB-experiment paints'),
    ]

    return [t.id for t, status in task_statuses if status != TaskStatus.SUCCESS], report_data


@stage(provides=('failed_stat_load_cmp_tasks', 'stat_load_report_data'))
def set_stat_load_report_data(task, spec_stat_load_request_log_resource_id_map, stat_load_cmp_tasks, stat_load=False):
    if not stat_load:
        return [], []
    task_statuses = check_tasks(
        task,
        list(chain(
            *([
                _cmp_tasks.values()
                for _cmp_tasks in stat_load_cmp_tasks.values()
            ])
        )),
        raise_on_fail=False,
    )
    report_data = [
        _create_report_data(stat_load_cmp_tasks, spec_stat_load_request_log_resource_id_map, 'Stat load'),
    ]

    return [t.id for t, status in task_statuses if status != TaskStatus.SUCCESS], report_data


@stage(provides=('failed_meta_load_cmp_tasks', 'meta_load_report_data'))
def set_meta_load_report_data(task, spec_meta_load_request_log_resource_id_map, meta_load_cmp_tasks, meta_load_shard_nums, meta_load=False):
    if not meta_load:
        return [], []
    task_statuses = check_tasks(
        task,
        meta_load_cmp_tasks.values(),
        raise_on_fail=False,
    )
    report_data = [
        _create_meta_load_report_data(meta_load_cmp_tasks, spec_meta_load_request_log_resource_id_map, meta_load_shard_nums)
    ]

    return [t.id for t, status in task_statuses if status != TaskStatus.SUCCESS], report_data
