import json
import logging
import re
from copy import deepcopy
from itertools import chain

from jinja2 import Environment

from sandbox import sdk2

from sandbox.common.utils import get_task_link
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.yabs.server.db.utils import get_yabscs, get_cs_settings
from sandbox.projects.common.yabs.server.db.yt_bases import run_cs_tool
from sandbox.projects.common.yabs.server.util import fetch_http_data
from sandbox.projects.common.yabs.server.util.general import check_tasks, find_last_ready_resource
from sandbox.projects.yabs.qa.pipeline.stage import stage, memoize_stage
from sandbox.projects.yabs.qa.pipeline_test_framework.helpers import _launch_task
from sandbox.projects.yabs.qa.spec.misc import (
    get_base_tags_by_meta_mode,
    get_stat_base_tags_by_meta_mode,
    get_meta_base_tags_by_meta_mode,
    filter_base_tags_by_meta_mode,
)
from sandbox.projects.yabs.qa.tasks.YabsServerCreateOneShotSpec.spec import OneShotSpec, YABS_SERVER_HOUSE_SPEC
from sandbox.projects.yabs.qa.tasks.ExecuteYTOneshot import ExecuteYTOneshot
from sandbox.projects.yabs.qa.tasks.YabsServerBaseSizeAggregate import YabsServerBaseSizeAggregate
from sandbox.projects.yabs.qa.tasks.YabsServerBaseSizeCmp import YabsServerBaseSizeCmp
from sandbox.projects.yabs.qa.tasks.YabsServerGetYTRequestData2 import YabsServerGetYtRequestData2
from sandbox.projects.yabs.qa.tasks.YabsServerCSImportTestStability import YabsServerCSImportTestStability
from sandbox.projects.yabs.qa.tasks.YabsServerGenerateLinearModelsServiceData import YabsServerGenerateLinearModelsServiceData
from sandbox.projects.yabs.qa.tasks.YabsServerTestOneShot.report import STARTREK_REPORT_TEMPLATE, HTML_REPORT_TEMPLATE, COLORS_BY_STATUS, SB_STYLE, Status


logger = logging.getLogger(__name__)


SPEC_CONTEXT_FIELDS = ['spec_{}'.format(spec_field) for spec_field in OneShotSpec.__slots__]


@stage(provides=SPEC_CONTEXT_FIELDS, result_is_dict=True)
def get_spec_data(task, spec_resource_id=None, stat_load=False, meta_load=False):
    if not spec_resource_id:
        attributes_to_search = {
            'released_spec': True,
            'good': True,
        }
        if stat_load or meta_load:
            attributes_to_search['good_with_load'] = True
        spec_resource = find_last_ready_resource(YABS_SERVER_HOUSE_SPEC, attributes_to_search)
    else:
        spec_resource = YABS_SERVER_HOUSE_SPEC[spec_resource_id]
    one_shot_spec = sdk2.ResourceData(spec_resource)
    with open(str(one_shot_spec.path), 'r') as spec_file:
        one_shot_spec_data = json.load(spec_file)
    spec_context_data = {
        'spec_{}'.format(key): one_shot_spec_data.get(key)
        for key in OneShotSpec.__slots__
    }

    logger.info('Got spec context data:\n%s', json.dumps(spec_context_data, indent=2))

    return spec_context_data


@stage(provides='ammo_generation_tasks')
def launch_ammo_generation_tasks(
        task,
        ammo_generation_config=None,
        test_description='',
        cache_daemon_resource=None,
        days_interval=0,
        logs_interval=0,
        custom_request_spec=None,
):
    ammo_generation_tasks = {}
    for meta_mode, custom_request in ammo_generation_config.items():
        if not custom_request:
            continue
        ammo_generation_tasks[meta_mode] = _launch_task(
            task,
            YabsServerGetYtRequestData2,
            description='get_yt_request_data, {}'.format(test_description),
            cache_daemon_resource=cache_daemon_resource,
            yql_query=custom_request,
            yql_query_role=meta_mode,
            spec=(custom_request_spec or {}),
            days_interval=days_interval,
            logs_interval=logs_interval,
        )
    return ammo_generation_tasks


@stage(provides='ammo_and_stubs_map')
def get_ammo_overrides(task, ammo_generation_tasks):
    if not ammo_generation_tasks:
        return {}

    check_tasks(task, ammo_generation_tasks.values())
    ammo_and_stubs_map = {}
    for meta_mode, task_id in ammo_generation_tasks.items():
        ammo_and_stubs_map[meta_mode] = sdk2.Task[task_id].Parameters.result_spec['{}_func'.format(meta_mode)]

    return ammo_and_stubs_map


@stage(provides=('test_cs_settings', ))
def init_cs_settings(self, cs_settings=''):
    if not cs_settings:
        return cs_settings
    if not cs_settings.startswith("{") or cs_settings.startswith("["):
        return fetch_http_data(cs_settings, raw=True)
    return cs_settings


@stage(provides=('print_oneshot_tables_task_id'))
def launch_print_oneshot_tables_task(task, oneshot_path='', is_yt_oneshot=False):
    if is_yt_oneshot:
        oneshot_path = oneshot_path
        return _launch_task(
            task,
            ExecuteYTOneshot,
            description='Get tables from YT oneshot "{}"'.format(oneshot_path),
            oneshot_path=oneshot_path,
            print_tables_only=True,
        )
    return -1


@stage(provides=('test_stat_oneshot_bases', 'test_stat_oneshot_tables', 'test_meta_oneshot_bases', 'test_meta_oneshot_tables'))
def get_oneshot_bases_and_tables(
        task,
        spec_stat_bs_release_yt_resource_id,
        spec_meta_bs_release_yt_resource_id,
        spec_cs_settings_archive_resource_id,
        print_oneshot_tables_task_id,
        test_cs_settings,
        is_yt_oneshot=False,
        oneshot_bases='',
        oneshot_tables='',
        cs_settings_patch_resource_id=None,
):

    def _get_yt_oneshot_bases_and_tables(bs_release_yt_resource_id):
        # Get oneshot_tables, check it is not empty, fill `oneshot_tables` context field
        oneshot_tables = ExecuteYTOneshot[print_oneshot_tables_task_id].Parameters.oneshot_tables
        logging.info('Oneshot tables: %s', oneshot_tables)
        if not oneshot_tables:
            raise TaskFailure('Empty oneshot tables list')

        # Sync `bs_release_yt` resource
        bs_release_yt_dir = get_yabscs(task, bs_release_yt_resource_id)

        # Run `cs import --get-table-to-shards-map` to get oneshot bases
        cs_settings_json = get_cs_settings(
            task=task,
            cs_settings_archive_res_id=spec_cs_settings_archive_resource_id,
            cs_settings_patch_res_id=cs_settings_patch_resource_id,
            settings_spec=test_cs_settings,
        )
        out_path = run_cs_tool(
            yt_token='',
            proxy='',
            cs_cycle_dir=bs_release_yt_dir,
            tool='import',
            args=['--get-table-to-shards-map'],
            settings_spec=cs_settings_json
        )
        with open(out_path) as f:
            table_to_shards_map = json.load(f)

        # Get bases affected by oneshot
        oneshot_bases = set()
        missing_tables = []
        for table in oneshot_tables:
            try:
                bases = []
                for base in table_to_shards_map[table]:
                    if re.match(r'st\d+', base):
                        bases += [cluster + base for cluster in ['bs_', 'yabs_']]
                    else:
                        bases.append(base)
            except KeyError:
                missing_tables.append(table)
            else:
                oneshot_bases.update(bases)

        if missing_tables:
            msg = 'Tables {} are not found in output of "cs import --get-table-to-shards-map" command. Check if related importers are enabled.'.format(missing_tables)
            raise TaskFailure(msg)

        if not oneshot_bases:
            raise TaskFailure('Oneshot doesn\'t affect any base')

        logging.info('Oneshot bases: %s', oneshot_bases)
        return list(sorted(oneshot_bases)), list(sorted(oneshot_tables))

    if is_yt_oneshot:
        check_tasks(task, print_oneshot_tables_task_id)
        test_stat_oneshot_bases, test_stat_oneshot_tables = _get_yt_oneshot_bases_and_tables(spec_stat_bs_release_yt_resource_id)
        test_meta_oneshot_bases, test_meta_oneshot_tables = _get_yt_oneshot_bases_and_tables(spec_meta_bs_release_yt_resource_id)
    else:
        test_stat_oneshot_tables = test_meta_oneshot_tables = list(sorted(oneshot_tables.split(','))) if oneshot_tables else []
        test_stat_oneshot_bases = test_meta_oneshot_bases = list(sorted(oneshot_bases.split(','))) if oneshot_bases else []

    return (
        test_stat_oneshot_bases, test_stat_oneshot_tables,
        test_meta_oneshot_bases, test_meta_oneshot_tables,
    )


@stage(provides=('stat_cs_import_test_stability_task_id', 'meta_cs_import_test_stability_task_id'))
def launch_cs_import_test_stability_task(
        task,
        spec_stat_setup_ya_make_task_id,
        spec_meta_setup_ya_make_task_id,
        spec_cs_input_spec_resource_id,
        spec_cs_settings_archive_resource_id,
        spec_mysql_archive_resource_id,
        spec_stat_bs_release_yt_resource_id,
        spec_stat_bs_release_tar_resource_id,
        spec_meta_bs_release_yt_resource_id,
        spec_meta_bs_release_tar_resource_id,
        spec_stat_base_tags_map,
        spec_meta_base_tags_map,
        ft_shard_keys,
        test_cs_settings,
        test_stat_oneshot_bases,
        test_meta_oneshot_bases,
        cs_settings_patch_resource_id='',
        run_import_test_stability=False,
):
    def _launch_cs_import_test_stability_task(bs_release_yt_resource_id, bs_release_tar_resource_id, base_tags_map, test_oneshot_bases):
        raw_base_tags_by_meta_mode = get_base_tags_by_meta_mode(
            base_tags_map,
            set(list(ft_shard_keys) + ['COMMON'])
        )
        logger.info('Raw base tags by meta mode:\n%s', json.dumps(raw_base_tags_by_meta_mode, indent=2))

        if test_oneshot_bases:
            base_tags_by_meta_mode = filter_base_tags_by_meta_mode(raw_base_tags_by_meta_mode, test_oneshot_bases)
        else:
            base_tags_by_meta_mode = raw_base_tags_by_meta_mode

        logger.info('Base tags by meta mode:\n%s', json.dumps(base_tags_by_meta_mode, indent=2))
        return _launch_task(
            task,
            YabsServerCSImportTestStability,
            description='Import test stability',
            input_spec=spec_cs_input_spec_resource_id,
            settings_archive=spec_cs_settings_archive_resource_id,
            cs_settings_patch=cs_settings_patch_resource_id,
            settings_spec=test_cs_settings,
            mysql_archive_contents=spec_mysql_archive_resource_id,
            bs_release_yt_resource=bs_release_yt_resource_id,
            server_resource=bs_release_tar_resource_id,
            use_cs_cycle=True,
            fail_on_diff=False,
            number_of_checks=3,
            bin_db_list=' '.join(set(chain(*base_tags_by_meta_mode.values()))),
        )
    if run_import_test_stability:
        return (
            _launch_cs_import_test_stability_task(spec_stat_bs_release_yt_resource_id, spec_stat_bs_release_tar_resource_id, spec_stat_base_tags_map, test_stat_oneshot_bases),
            _launch_cs_import_test_stability_task(spec_meta_bs_release_yt_resource_id, spec_meta_bs_release_tar_resource_id, spec_meta_base_tags_map, test_meta_oneshot_bases),
        )

    return (-1, -1)


@stage(provides=(
    'stat_make_bin_bases_task_id',
    'meta_make_bin_bases_task_id',
    'generate_linear_models_service_data_task_id',
))
def launch_base_producing_tasks(
        task,
        spec_cs_input_spec_resource_id,
        spec_cs_settings_archive_resource_id,
        spec_mysql_archive_resource_id,
        spec_stat_bs_release_yt_resource_id,
        spec_stat_bs_release_tar_resource_id,
        spec_meta_bs_release_yt_resource_id,
        spec_meta_bs_release_tar_resource_id,
        spec_stat_setup_ya_make_task_id,
        spec_meta_setup_ya_make_task_id,
        spec_shard_map_resource_id,
        spec_gen_bin_bases_flags_resource_id,
        spec_stat_base_tags_map,
        spec_meta_base_tags_map,
        test_stat_oneshot_bases,
        test_stat_oneshot_tables,
        test_meta_oneshot_bases,
        test_meta_oneshot_tables,
        test_cs_settings,
        ft_shard_keys,
        stat_load_shard_keys,
        meta_load_shard_keys,
        cs_settings_patch_resource_id=None,
        is_content_system_settings_change_test=False,
        is_yt_oneshot=False,
        oneshot_path='',
        oneshot_args='',
        oneshot_hosts='',
):
    generate_linear_models_service_data_task_id = _launch_task(
        task,
        YabsServerGenerateLinearModelsServiceData,
        description='Generate data for linear models service',
        input_spec=spec_cs_input_spec_resource_id,
        settings_archive=spec_cs_settings_archive_resource_id,
        cs_settings_patch=cs_settings_patch_resource_id,
        settings_spec=test_cs_settings,
        mysql_archive_contents=spec_mysql_archive_resource_id,
        bs_release_yt_resource=spec_meta_bs_release_yt_resource_id,
    )

    def get_make_bin_bases_parameters(
            base_tags_map,
            setup_ya_make_task_id,
            bs_release_yt_resource_id,
            bs_release_tar_resource_id,
            test_oneshot_bases,
            test_oneshot_tables,
            server_role=None,
    ):
        shard_keys = set(list(ft_shard_keys) + list(stat_load_shard_keys) + list(meta_load_shard_keys) + ['COMMON'])
        if server_role == 'stat':
            raw_base_tags_by_meta_mode = get_stat_base_tags_by_meta_mode(base_tags_map, shard_keys)
        elif server_role == 'meta':
            raw_base_tags_by_meta_mode = get_meta_base_tags_by_meta_mode(base_tags_map)
        else:
            raise RuntimeError('Incorrect server_role: {}'.format(server_role))

        logger.info('Raw base tags by meta mode for %s:\n%s', server_role, json.dumps(raw_base_tags_by_meta_mode, indent=2))
        setup_ya_make_context = sdk2.Task[setup_ya_make_task_id].Context
        make_bin_bases_parameters = {
            'input_spec': spec_cs_input_spec_resource_id,
            'settings_archive': spec_cs_settings_archive_resource_id,
            'cs_settings_patch': cs_settings_patch_resource_id,
            'settings_spec': test_cs_settings,
            'mysql_archive_contents': spec_mysql_archive_resource_id,
            'bs_release_yt_resource': bs_release_yt_resource_id,
            'server_resource': bs_release_tar_resource_id,
            'use_cs_cycle': True,
            'do_not_restart': True,
            'cs_import_ver': setup_ya_make_context.cs_import_ver,
            'glue_reduce': True,
            'common_oneshots_md5': setup_ya_make_context.common_oneshots_md5,
            'common_oneshots_bases': setup_ya_make_context.common_oneshots_bases,
            'is_yt_oneshot': is_yt_oneshot,
            'oneshot_path': oneshot_path,
            'oneshot_args': oneshot_args,
            'delete_fetch_results': not is_yt_oneshot,
            'options_list': getattr(sdk2.Resource[spec_gen_bin_bases_flags_resource_id], 'add_options', '').strip(),
        }
        if test_oneshot_bases:
            base_tags_by_meta_mode = filter_base_tags_by_meta_mode(raw_base_tags_by_meta_mode, test_oneshot_bases)
            make_bin_bases_parameters['reuse_existing_bases'] = False
        elif is_content_system_settings_change_test:
            base_tags_by_meta_mode = raw_base_tags_by_meta_mode
            make_bin_bases_parameters['reuse_existing_bases'] = True
        else:
            raise ValueError('Did not find any oneshot bases to generate, please check input_parameters')
        logger.info('Base tags by meta mode for %s:\n%s', server_role, json.dumps(base_tags_by_meta_mode, indent=2))

        if not is_yt_oneshot:
            make_bin_bases_parameters.update({
                'oneshot_bases': ','.join(test_oneshot_bases),
                'oneshot_tables': ','.join(test_oneshot_tables),
                'oneshot_hosts': oneshot_hosts,
            })

        make_bin_bases_parameters.update(base_tags_by_meta_mode)
        make_bin_bases_parameters['bin_db_list'] = ' '.join(set(chain(*base_tags_by_meta_mode.values())))

        return make_bin_bases_parameters

    stat_make_bin_bases_parameters = get_make_bin_bases_parameters(
        spec_stat_base_tags_map,
        spec_stat_setup_ya_make_task_id,
        spec_stat_bs_release_yt_resource_id,
        spec_stat_bs_release_tar_resource_id,
        test_stat_oneshot_bases,
        test_stat_oneshot_tables,
        server_role='stat',
    )
    meta_make_bin_bases_parameters = get_make_bin_bases_parameters(
        spec_meta_base_tags_map,
        spec_meta_setup_ya_make_task_id,
        spec_meta_bs_release_yt_resource_id,
        spec_meta_bs_release_tar_resource_id,
        test_meta_oneshot_bases,
        test_meta_oneshot_tables,
        server_role='meta',
    )
    return (
        _launch_task(
            task,
            sdk2.Task['YABS_SERVER_MAKE_BIN_BASES'],
            description='meta make bin bases | oneshot',
            **stat_make_bin_bases_parameters
        ) if stat_make_bin_bases_parameters['bin_db_list'] != '' else -1,
        _launch_task(
            task,
            sdk2.Task['YABS_SERVER_MAKE_BIN_BASES'],
            description='stat make bin bases | oneshot',
            **meta_make_bin_bases_parameters
        ) if meta_make_bin_bases_parameters['bin_db_list'] != '' else -1,
        generate_linear_models_service_data_task_id,
    )


def _get_binary_bases(binary_base_resource_id_by_tag, make_bin_bases_task_id):
    baseline_binary_base_resource_id_by_tag = deepcopy(binary_base_resource_id_by_tag)
    test_binary_base_resource_id_by_tag = deepcopy(binary_base_resource_id_by_tag)

    if make_bin_bases_task_id > 0:
        make_bin_bases_task_context = sdk2.Task[make_bin_bases_task_id].Context
        generated_base_resource_id_by_tag = {
            sdk2.Resource[resource_id].tag: resource_id
            for resource_id in make_bin_bases_task_context.bin_base_res_ids
        }
        logger.info('Got test bases:\n%s', json.dumps(generated_base_resource_id_by_tag, indent=2))
        test_binary_base_resource_id_by_tag.update(**generated_base_resource_id_by_tag)

    return baseline_binary_base_resource_id_by_tag, test_binary_base_resource_id_by_tag


@stage(provides=(
    'baseline_stat_binary_base_resource_id_by_tag',
    'test_stat_binary_base_resource_id_by_tag',
    'baseline_meta_binary_base_resource_id_by_tag',
    'test_meta_binary_base_resource_id_by_tag',

    'test_linear_models_service_data_resource',
))
def get_binary_bases(
        task,
        generate_linear_models_service_data_task_id,
        spec_stat_binary_base_resource_id_by_tag,
        stat_make_bin_bases_task_id,
        spec_meta_binary_base_resource_id_by_tag,
        meta_make_bin_bases_task_id,
):
    check_tasks(
        task,
        [
            stat_make_bin_bases_task_id,
            meta_make_bin_bases_task_id,
            generate_linear_models_service_data_task_id,
        ]
    )

    generate_linear_models_service_data_task = YabsServerGenerateLinearModelsServiceData[generate_linear_models_service_data_task_id]
    linear_models_service_data_resource = generate_linear_models_service_data_task.Parameters.linear_models_service_data_resource

    return tuple(chain(
        _get_binary_bases(spec_stat_binary_base_resource_id_by_tag, stat_make_bin_bases_task_id),
        _get_binary_bases(spec_meta_binary_base_resource_id_by_tag, meta_make_bin_bases_task_id),
        (linear_models_service_data_resource.id, )
    ))


@stage(provides=('test_stat_chkdb_task_id', 'test_meta_chkdb_task_id'))
def launch_chkdb_task(
        task,
        ft_shard_keys,
        spec_stat_base_tags_map,
        spec_meta_base_tags_map,
        test_stat_binary_base_resource_id_by_tag,
        test_meta_binary_base_resource_id_by_tag,
        test_description='',
):
    meta_bases = {
        'base_resources_meta_{meta_mode}'.format(meta_mode=meta_mode): [
            test_meta_binary_base_resource_id_by_tag[tag]
            for tag in spec_meta_base_tags_map['base_tags_meta_{meta_mode}'.format(meta_mode=meta_mode)]
        ]
        for meta_mode in ('bs', 'bsrank', 'yabs')
    }
    stat_bases = {
        'base_resources_stat_{meta_mode}'.format(meta_mode=meta_mode): [
            test_stat_binary_base_resource_id_by_tag[tag]
            for shard in list(ft_shard_keys) + ['COMMON']
            for tag in spec_stat_base_tags_map.get('base_tags_stat_{meta_mode}_{shard}'.format(meta_mode=meta_mode, shard=shard), [])
        ]
        for meta_mode in ('bs', 'bsrank', 'yabs')
    }

    return (
        _launch_task(
            task,
            YabsServerBaseSizeAggregate,
            description='stat chkdb | {}'.format(test_description),
            **stat_bases
        ),
        _launch_task(
            task,
            YabsServerBaseSizeAggregate,
            description='meta chkdb | {}'.format(test_description),
            **meta_bases
        ),
    )


@stage(provides=('stat_chkdb_cmp_task_id', 'meta_chkdb_cmp_task_id'))
def launch_chkdb_cmp_task(
        task,
        test_stat_chkdb_task_id,
        spec_stat_chkdb_task_id,
        test_meta_chkdb_task_id,
        spec_meta_chkdb_task_id,
        test_description=''
):
    check_tasks(
        task,
        [
            test_stat_chkdb_task_id, spec_stat_chkdb_task_id,
            test_meta_chkdb_task_id, spec_meta_chkdb_task_id,
        ]
    )
    return (
        _launch_task(
            task,
            YabsServerBaseSizeCmp,
            description='stat chkdb_cmp | {}'.format(test_description),
            pre_task=spec_stat_chkdb_task_id,
            test_task=test_stat_chkdb_task_id,
            compare_same_bases=True,
            use_chkdb_without_hash=False,
        ),
        _launch_task(
            task,
            YabsServerBaseSizeCmp,
            description='meta chkdb_cmp | {}'.format(test_description),
            pre_task=spec_meta_chkdb_task_id,
            test_task=test_meta_chkdb_task_id,
            compare_same_bases=True,
            use_chkdb_without_hash=False,
        ),
    )


@stage(provides='bases_were_changed')
def check_base_changes_after_oneshot_application(
        task,
        stat_chkdb_cmp_task_id,
        meta_chkdb_cmp_task_id,
        check_binary_bases_changes=False,
):
    check_tasks(task, [_chkdb_cmp_task_id for _chkdb_cmp_task_id in (stat_chkdb_cmp_task_id, meta_chkdb_cmp_task_id) if _chkdb_cmp_task_id > 0])
    if not any([sdk2.Task[stat_chkdb_cmp_task_id].Context.has_diff, sdk2.Task[meta_chkdb_cmp_task_id].Context.has_diff]) and check_binary_bases_changes:
        raise TaskFailure("Oneshot has been applied but nothing was changed in binary bases")
    return sdk2.Task[stat_chkdb_cmp_task_id].Context.has_diff or sdk2.Task[meta_chkdb_cmp_task_id].Context.has_diff


def get_versions_report(
        spec_stat_bs_release_tar_resource_id, spec_stat_bs_release_yt_resource_id,
        spec_meta_bs_release_tar_resource_id, spec_meta_bs_release_yt_resource_id,
):
    from sandbox.projects.yabs.release.version.sandbox_helpers import SandboxHelper
    sandbox_helper = SandboxHelper()

    report_data = {
        "stat_server": sandbox_helper.get_full_version_from_resource(spec_stat_bs_release_tar_resource_id),
        "meta_server": sandbox_helper.get_full_version_from_resource(spec_meta_bs_release_tar_resource_id),
        "stat_cs": sandbox_helper.get_full_version_from_resource(spec_stat_bs_release_yt_resource_id),
        "meta_cs": sandbox_helper.get_full_version_from_resource(spec_meta_bs_release_yt_resource_id),
    }

    return report_data


def get_data_resources_report(spec_mysql_archive_resource_id, spec_cs_input_spec_resource_id):
    return {
        "mysql_archive": str(sdk2.Resource[spec_mysql_archive_resource_id].created),
        "cs_input_spec": str(sdk2.Resource[spec_cs_input_spec_resource_id].created),
    }


def get_ft_report(spec_ft_request_log_resource_id_map, ft_cmp_tasks, ft_validation_cmp_tasks):
    ammo_created = {
        role: str(sdk2.Resource[resource_id].created)
        for role, resource_id in spec_ft_request_log_resource_id_map.items()
    }
    report_data = []
    for meta_mode, ft_cmp_tasks_by_shard in ft_cmp_tasks.items():
        for shard, ft_cmp_task_id in ft_cmp_tasks_by_shard.items():
            ft_cmp_ctx = sdk2.Task[ft_cmp_task_id].Context
            ft_cmp_task_url = get_task_link(ft_cmp_task_id)
            short_report_text = getattr(ft_cmp_ctx, 'short_report_text')
            short_report_link = getattr(ft_cmp_ctx, 'short_report_link')

            validation_cmp_task_id = ft_validation_cmp_tasks[meta_mode][shard]
            validation_cmp_task = sdk2.Task[validation_cmp_task_id]
            validation_task_url = get_task_link(validation_cmp_task_id)

            report_data.append({
                "role": meta_mode,
                "shard": shard,
                "cmp_task_link": ft_cmp_task_url,
                "cmp_task_id": ft_cmp_task_id,
                "cmp_report_link": short_report_link,
                "cmp_report_text": short_report_text,
                "cmp_status": Status.FAIL if getattr(ft_cmp_ctx, "has_diff", True) else Status.OK,
                "validation_task_link": validation_task_url,
                "validation_task_id": validation_cmp_task_id,
                "validation_status": Status.FAIL if validation_cmp_task.Parameters.has_diff else Status.OK,
                "ammo_created": ammo_created[meta_mode],
            })

    return report_data


def get_stat_load_report(stat_load_cmp_tasks, spec_stat_load_request_log_resource_id_map):
    report_data = []
    for meta_mode, stat_load_cmp_tasks_by_shards in stat_load_cmp_tasks.items():
        request_log_resource_id = spec_stat_load_request_log_resource_id_map[meta_mode]
        ammo_creation_time = str(sdk2.Resource[request_log_resource_id].created)
        for shard, cmp_task_id in stat_load_cmp_tasks_by_shards.items():
            cmp_task_ctx = sdk2.Task[cmp_task_id].Context
            try:
                report_data.append({
                    "role": meta_mode,
                    "shard": shard,
                    "cmp_task_link": get_task_link(cmp_task_id),
                    "cmp_task_id": cmp_task_id,
                    "cmp_report_link": cmp_task_ctx.short_report_link,
                    "cmp_report_text": cmp_task_ctx.short_report_text,
                    "cmp_status": Status.FAIL if cmp_task_ctx.has_diff else Status.OK,
                    "ammo_created": ammo_creation_time,
                })
            except Exception as exc:
                logging.error(
                    "Got exception while trying to build report for {} role at {} shard: {}".format(meta_mode, shard, exc),
                    exc_info=True,
                )

    return report_data


def get_meta_load_report(meta_load_cmp_tasks, spec_meta_load_request_log_resource_id_map):
    ammo_created = {
        role: str(sdk2.Resource[resource_id].created)
        for role, resource_id in spec_meta_load_request_log_resource_id_map.items()
    }
    report_data = []
    for meta_mode, meta_load_cmp_task_id in meta_load_cmp_tasks.items():
        meta_load_cmp_task = sdk2.Task[meta_load_cmp_task_id]
        meta_load_cmp_task_url = get_task_link(meta_load_cmp_task_id)

        report_data.append({
            "role": meta_mode,
            "cmp_task_link": meta_load_cmp_task_url,
            "cmp_task_id": meta_load_cmp_task_id,
            "cmp_status": Status.FAIL if meta_load_cmp_task.Parameters.has_diff else Status.OK,
            "st_report": meta_load_cmp_task.Parameters.st_report,
            "html_report": meta_load_cmp_task.Parameters.st_report,  # TODO: Create beautiful html report BSSERVER-15395
            "ammo_created": ammo_created[meta_mode],
        })

    return report_data


def get_chkdb_report(chkdb_cmp_task_id, report_size_threshold=10000):
    if not chkdb_cmp_task_id or chkdb_cmp_task_id < 0:
        return None

    chkdb_task = sdk2.Task[chkdb_cmp_task_id]
    chkdb_task_link = get_task_link(chkdb_cmp_task_id)
    chkdb_task_report = chkdb_task.Context.report

    return {
        "report": chkdb_task_report[:report_size_threshold],
        "report_is_too_long": len(chkdb_task_report) >= report_size_threshold,
        "task_link": chkdb_task_link,
    }


def get_stability_report(test_ft_stability_tasks):
    if not test_ft_stability_tasks:
        return None

    report_data = []
    for meta_mode, ft_stability_tasks_by_shard in test_ft_stability_tasks.items():
        for shard, ft_stability_task_id in ft_stability_tasks_by_shard.items():
            ft_stability_task = sdk2.Task[ft_stability_task_id]
            ft_stability_task_link = get_task_link(ft_stability_task_id)

            ft_stability_status = Status.OK if ft_stability_task.status == 'SUCCESS' else Status.FAIL

            report_data.append({
                "role": meta_mode,
                "shard": shard,
                "task_link": ft_stability_task_link,
                "task_id": ft_stability_task_id,
                "status": ft_stability_status,
            })

    return report_data


def get_import_stability_report(import_stability_task_id):
    if import_stability_task_id < 0:
        return None
    import_stability_task = sdk2.Task[import_stability_task_id]
    import_stability_link = get_task_link(import_stability_task_id)
    import_stability_report = import_stability_task.Context.report
    return {
        'report': import_stability_report,
        'task_link': import_stability_link,
    }


@memoize_stage()
def create_report(
        task,
        spec_stat_bs_release_tar_resource_id,
        spec_stat_bs_release_yt_resource_id,
        spec_meta_bs_release_tar_resource_id,
        spec_meta_bs_release_yt_resource_id,
        spec_mysql_archive_resource_id,
        spec_cs_input_spec_resource_id,
        spec_ft_request_log_resource_id_map,
        ft_cmp_tasks,
        ft_validation_cmp_tasks,
        stat_load_cmp_tasks,
        spec_stat_load_request_log_resource_id_map,
        meta_load_cmp_tasks,
        stat_chkdb_cmp_task_id,
        meta_chkdb_cmp_task_id,
        test_ft_stability_tasks,
        stat_cs_import_test_stability_task_id,
        meta_cs_import_test_stability_task_id,
):
    check_tasks(task, list(chain(*(
        [
            tasks_by_shard.values() for tasks_by_shard in ft_cmp_tasks.values()
        ] + [
            tasks_by_shard.values() for tasks_by_shard in ft_validation_cmp_tasks.values()
        ] + [
            tasks_by_shard.values() for tasks_by_shard in stat_load_cmp_tasks.values()
        ] + [
            meta_load_cmp_tasks.values()
        ] + [
            [_chkdb_cmp_task_id for _chkdb_cmp_task_id in (stat_chkdb_cmp_task_id, meta_chkdb_cmp_task_id)]
        ]
    ))))
    check_tasks(task, list(chain(*(
        [
            tasks_by_shard.values() for tasks_by_shard in test_ft_stability_tasks.values()
        ] + [
            [
                _cs_import_test_stability_task_id
                for _cs_import_test_stability_task_id in (stat_cs_import_test_stability_task_id, meta_cs_import_test_stability_task_id)
                if _cs_import_test_stability_task_id > 0
            ]
        ]
    ))), raise_on_fail=False)
    report_data = {
        "versions": get_versions_report(
            spec_stat_bs_release_tar_resource_id, spec_stat_bs_release_yt_resource_id,
            spec_meta_bs_release_tar_resource_id, spec_meta_bs_release_yt_resource_id,
        ),
        "data_resources": get_data_resources_report(spec_mysql_archive_resource_id, spec_cs_input_spec_resource_id),
        "func": get_ft_report(spec_ft_request_log_resource_id_map, ft_cmp_tasks, ft_validation_cmp_tasks),
        "load": get_stat_load_report(stat_load_cmp_tasks, spec_stat_load_request_log_resource_id_map),
        "perf_meta": get_meta_load_report(meta_load_cmp_tasks, spec_ft_request_log_resource_id_map),
        "stat_chkdb": get_chkdb_report(stat_chkdb_cmp_task_id),
        "meta_chkdb": get_chkdb_report(meta_chkdb_cmp_task_id),
        "stability": get_stability_report(test_ft_stability_tasks),
        'stat_import_stability': get_import_stability_report(stat_cs_import_test_stability_task_id),
        'meta_import_stability': get_import_stability_report(meta_cs_import_test_stability_task_id),
    }
    env = Environment()
    html_report_template = env.from_string(HTML_REPORT_TEMPLATE)
    html_report = html_report_template.render(report_data=report_data, style=SB_STYLE, color_map=COLORS_BY_STATUS)

    startrek_report_template = env.from_string(STARTREK_REPORT_TEMPLATE)
    startrek_report = startrek_report_template.render(report_data=report_data, style=SB_STYLE, color_map=COLORS_BY_STATUS)

    return html_report, startrek_report
