import logging
import os
from datetime import timedelta, datetime
from multiprocessing.dummy import Pool as ThreadPool

from sandbox import sdk2

from sandbox.sandboxsdk.environments import PipEnvironment
import sandbox.common.types.resource as ctr
from sandbox.common.types.task import ReleaseStatus

from sandbox.projects.common.yabs.server.util.general import try_get_from_vault
from sandbox.projects.yabs.qa.utils.yt_utils import create_node

from sandbox.projects.yabs.qa.response_tools.unpacker import get_unpacker_executable_path, uc_decompress
from sandbox.projects.yabs.qa.resource_types import (
    YabsResponseDumpUnpacker,
    YabsServerResponseDump,
    UCTool,
)
from sandbox.projects.yabs.qa.utils.general import html_hyperlink

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers import subprocess
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import new_resource_data


CONTENT_TYPE_MAPPING = {'reshoot': 'primary_ammo'}


logger = logging.getLogger()


def get_shoot_task_dumps(task_id):
    return list(
        YabsServerResponseDump.find(task_id=task_id, state=ctr.State.READY).order(sdk2.Resource.id).limit(100)
    )


def get_uc_executable_path():
    resource = sdk2.Resource.find(UCTool, attrs={'released': ReleaseStatus.STABLE}).order(-sdk2.Resource.id).first()
    uc_archive_path = str(new_resource_data(resource).path)
    subprocess.check_call(['tar', '-xf', uc_archive_path])
    return os.path.join(os.getcwd(), 'uc')


class YTUploadParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group('Uploading to YT') as yt_upload_group:
        upload_to_yt_prefix = sdk2.parameters.String(
            'YT prefix for result upload',
            default_value='//home/yabs-cs-sandbox/reports',
        )
        write_ext_responses = sdk2.parameters.List(
            'Write response to YT report for external services',
            default=['linear_models_service', 'linear_models_service_wide'],
        )
        write_ext_requests = sdk2.parameters.List(
            'Write request to YT report for external services',
            default=[],
        )
        with sdk2.parameters.CheckGroup(
            'Write response headers to YT report for selected shoot types'
        ) as yt_write_response_headers_content_types:
            yt_write_response_headers_content_types.values.primary_ammo =\
                yt_write_response_headers_content_types.Value('Primary ammo', checked=True)
            yt_write_response_headers_content_types.values.secondary_count_links = 'Secondary count links'
        yt_node_ttl = sdk2.parameters.Integer(
            'YT node ttl in days',
            default=2,
        )


class YabsServerUploadShootResultToYt(sdk2.Task):
    class Parameters(sdk2.Parameters):
        shoot_task = sdk2.parameters.Integer(
            'Shoot task id to upload results',
            required=True,
        )
        dump_parser_resource = sdk2.parameters.Resource(
            'Dump parser resource',
            resource_type=YabsResponseDumpUnpacker,
        )
        yt_upload_parameters = YTUploadParameters()
        upload_workers = sdk2.parameters.Integer(
            'The number of upload workers',
            default=4,
        )
        with sdk2.parameters.Output:
            uploaded_logs_to_yt_prefix = sdk2.parameters.String('YT path with uploaded shoot result')

    class Requirements(sdk2.Task.Requirements):
        disk_space = 150 * 1024  # 150 GiB
        environments = [
            PipEnvironment('yandex-yt', use_wheel=True),
        ]

    @property
    def uc_executable_path(self):
        if not hasattr(self, '_uc_executable_path'):
            self._uc_executable_path = get_uc_executable_path()
        return self._uc_executable_path

    @property
    def yt_token(self):
        return try_get_from_vault(self, 'yt_token_for_report_upload')

    @trace_calls(save_arguments=(1, 'cmd'))
    def upload_job(self, args):
        cmd, path = args
        env = os.environ.copy()
        env['YT_TOKEN'] = self.yt_token
        env['YT_LOG_LEVEL'] = 'WARNING'
        logger.debug('Run %s', cmd)
        with sdk2.helpers.ProcessLog(self, logger='upload_{}'.format(path)) as pl:
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout, env=env)

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        from yt.wrapper import ypath_join, YtClient

        base_yt_path = ypath_join(self.Parameters.upload_to_yt_prefix, str(self.id))
        yt_client = YtClient(proxy='hahn', token=self.yt_token)
        yt_node_ttl = None
        if self.Parameters.yt_node_ttl:
            yt_node_ttl = get_ttl_with_respect_to_weekends(datetime.now(), self.Parameters.yt_node_ttl)
        create_node(base_yt_path, yt_client, ttl=yt_node_ttl)

        commands = []
        decompressed_prefix = 'decompressed'
        os.mkdir(decompressed_prefix)

        dumps = get_shoot_task_dumps(self.Parameters.shoot_task)
        if self.Parameters.dump_parser_resource:
            dump_parser = self.Parameters.dump_parser_resource
        elif dumps:
            dump_parser = YabsResponseDumpUnpacker[dumps[0].__getattr__('dump_parser_id')]
        else:
            raise ValueError('Task {} has no READY response dumps'.format(self.Parameters.shoot_task))
        dump_parser_path = get_unpacker_executable_path(dump_parser)

        for i, resource in enumerate(dumps):
            path = str(new_resource_data(resource).path)
            decompressed_file_name = 'dump_{}_{}_{}'.format(resource.content_type, resource.shoot_index, i)
            decompressed_path = os.path.join(decompressed_prefix, decompressed_file_name)
            uc_decompress(self, path, decompressed_path)
            content_type = str(resource.content_type)
            yt_path = ypath_join(base_yt_path, content_type, str(resource.shoot_index))
            cmd = [
                dump_parser_path,
                '--write-yt', decompressed_path,
                '-r', yt_path,
            ]
            mapped_content_type = CONTENT_TYPE_MAPPING.get(content_type, content_type)
            if mapped_content_type in self.Parameters.yt_write_response_headers_content_types:
                cmd.append('--write-response-headers')

            for ext_tag in self.Parameters.write_ext_responses:
                cmd += ['--write-ext-response', ext_tag]

            for ext_tag in self.Parameters.write_ext_requests:
                cmd += ['--write-ext-request', ext_tag]

            commands.append((cmd, decompressed_file_name))

        pool = ThreadPool(self.Parameters.upload_workers)
        pool.map(self.upload_job, commands)

        self.set_info(
            html_hyperlink(
                link='https://yt.yandex-team.ru/hahn/?page=navigation&path={}'.format(base_yt_path),
                text='Shoot results uploaded to Hahn',
            ),
            do_escape=False,
        )
        self.Parameters.uploaded_logs_to_yt_prefix = base_yt_path


def get_ttl_with_respect_to_weekends(now_datetime, ttl_days):
    _ttl_days = 0
    while ttl_days:
        now_datetime += timedelta(days=1)
        if now_datetime.weekday() < 5:
            # is a workday
            ttl_days -= 1
        _ttl_days += 1
    return timedelta(days=_ttl_days).total_seconds()
