import json
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr
from sandbox.common.errors import TaskFailure
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_AB_EXPERIMENT_SETTINGS, BaseBackupSdk2Resource


HANDLER_BS = 'BS'
HANDLER_BIGB = 'BIGB_MAIN'
BS_AB_HANDLERS = (HANDLER_BS, HANDLER_BIGB)

INFO_TEMPL = '''
<h2>AB experiment config validation</h2>
{}
<span>{}</span>
'''
FAIL_INFO = '<h3 style="color: red">FAIL</h3>'
OK_INFO = '<h3 style="color: green">OK</h3>'
FAILURE_MSG = "Validation failed"


def handler_bigb_and_has_bs_parameters(flag):
    return (
        flag.get('HANDLER') == HANDLER_BIGB and
        'BSParameters' in flag.get('CONTEXT', {}).get('MAIN', {}).get('BIGB_MAIN', {})
    )


def get_ab_experiment(ab_experiment_resource, ab_experiment_input):
    ab_experiment = {}
    if ab_experiment_resource:
        data_path = str(sdk2.ResourceData(ab_experiment_resource).path)
        with open(data_path) as f:
            ab_experiment = json.load(f)
        for data in ab_experiment:
            if data.get('HANDLER') in BS_AB_HANDLERS:
                data.pop('RESTRICTIONS', None)
    else:
        ab_experiment = ab_experiment_input
    return ab_experiment


class YabsServerValidateAbExperimentBin(BaseBackupSdk2Resource):
    pass


class YabsServerValidateAbExperiment(sdk2.Task):
    """ task for validating {"HANDLER": "BS"} AB experiment config """

    class Parameters(sdk2.Parameters):
        ab_experiment = sdk2.parameters.JSON('Json with AB experiments')
        ab_experiment_resource = sdk2.parameters.Resource('Resource with json AB experiment', resource_type=YABS_SERVER_AB_EXPERIMENT_SETTINGS)
        validate_ab_experiment_bin_id = sdk2.parameters.Integer('Validate AB Experiment binary_id', default=0)
        kill_timeout = 10 * 60

    def save_result_info(self, validation_info):
        self.Context.validation_info = validation_info
        self.set_info(validation_info, do_escape=False)

    def check_exps(self, tool_path, exps, handler, expected_testids):
        file_with_exps_filename = 'experimets.json'
        with open(file_with_exps_filename, 'w') as file_with_exps:
            json.dump(exps, file_with_exps)
        cmd = [tool_path, '--handler', handler, '--from-file', file_with_exps_filename]
        self.Context.parsing_result[handler] = json.loads(sp.check_output(cmd))

        logs = self.Context.parsing_result[handler]['logs']
        if logs:
            validation_info = INFO_TEMPL.format(
                FAIL_INFO,
                'Errors occurred during parsing HANDLER {}:\n{}'.format(handler, logs)
            )
            self.save_result_info(validation_info)
            raise TaskFailure(FAILURE_MSG)

        size = self.Context.parsing_result[handler]['size']
        if size < 1:
            validation_info = INFO_TEMPL.format(
                FAIL_INFO,
                'No HANDLER {} experiments found'.format(handler),
            )
            self.save_result_info(validation_info)
            raise TaskFailure(FAILURE_MSG)

        parsed_testids = self.Context.parsing_result[handler]['test_ids']
        if sorted(expected_testids) != sorted(parsed_testids):
            validation_info = INFO_TEMPL.format(
                FAIL_INFO,
                'Expected HANDLER {} TESIDs: {}\nGot TESIDs: {}'.format(handler, expected_testids, parsed_testids),
            )
            self.save_result_info(validation_info)
            raise TaskFailure(FAILURE_MSG)

    def on_execute(self):
        if self.Parameters.validate_ab_experiment_bin_id:
            binary_id = sdk2.Resource.find(
                resource_type=YabsServerValidateAbExperimentBin,
                task=sdk2.Task[self.Parameters.validate_ab_experiment_bin_id]
            ).limit(1).first()
        else:
            binary_id = sdk2.Resource.find(
                resource_type=YabsServerValidateAbExperimentBin,
                state=ctr.State.READY
            ).limit(1).first()
        validation_tool_path = str(sdk2.ResourceData(binary_id).path)
        self.Context.parsing_result = dict()
        exps = get_ab_experiment(self.Parameters.ab_experiment_resource, self.Parameters.ab_experiment)
        logging.info('AB experiment data: %s\n', json.dumps(exps, indent=2))
        self.Context.bs_testids = [int(e['TESTID'][0]) for e in exps if e.get('HANDLER') == HANDLER_BS]
        if self.Context.bs_testids:
            self.check_exps(
                validation_tool_path,
                exps=exps,
                handler=HANDLER_BS,
                expected_testids=self.Context.bs_testids,
            )

        self.Context.bigb_testids = [int(e['TESTID'][0]) for e in exps if e.get('HANDLER') == HANDLER_BIGB]
        self.Context.bigb_testids_with_bs_params = [int(e['TESTID'][0]) for e in exps if handler_bigb_and_has_bs_parameters(e)]
        if self.Context.bigb_testids_with_bs_params:
            self.check_exps(
                validation_tool_path,
                exps=exps,
                handler=HANDLER_BIGB,
                expected_testids=self.Context.bigb_testids,
            )

        validation_info = INFO_TEMPL.format(OK_INFO, '')
        self.save_result_info(validation_info)
