import logging
import posixpath
from datetime import timedelta

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import execution_dir_link
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShoot2 import (
    YabsServerB2BFuncShoot2,
    YabsServerResponseDump,
    YabsBadRequestsID,
)
from sandbox.projects.yabs.qa.response_tools.parse_json import JSONP_BEHAVIORS
from sandbox.projects.yabs.qa.response_tools.unpacker import unpack_responses_dump_resource, find_unpacker_resource
from sandbox.projects.yabs.qa.resource_types import (
    BaseBackupSdk2Resource,
    UCTool,
    YabsResponseDumpUnpacker,
)

from sandbox.projects.yabs.qa.tasks.YabsServerValidateResponses.utils.validate import (
    validate_responses,
    ValidationStatus,
)
from sandbox.projects.yabs.qa.tasks.YabsServerValidateResponses.utils.report import (
    count_statuses,
    create_summary_report,
    create_skipped_report,
    create_yql_quieries,
    create_report_links,
    write_meta_report,
    write_full_report,
    REPORTABLE_STATUSES
)

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace, trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import new_resource
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import new_resource_data


SHOOT_REPORTS_YT_NODE_KEY = 'shoot_reports_yt_node'
RESPONSES_DIR = 'responses'
REPORTS_DIR = 'reports'
META_REPORT_FILENAME = 'meta_report.json'
YQL_TOKEN_VAULT_NAME = 'yabs-cs-sb-yql-token'
SKIPPED_PAGE_ID_ISSUES = {}


logger = logging.getLogger(__name__)


class YabsResponseValidationReport(BaseBackupSdk2Resource):
    auto_backup = True


class YabsResponseValidationMetaReport(BaseBackupSdk2Resource):
    auto_backup = True


class YabsServerValidateResponses(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = int(timedelta(hours=1).total_seconds())

        shoot_task = sdk2.parameters.Task(
            'Shoot task',
            task_type=YabsServerB2BFuncShoot2.type)
        response_dumps = sdk2.parameters.Resource(
            'Responses dump resource',
            resource_type=YabsServerResponseDump,
            multiple=True)

        with sdk2.parameters.Group("Validation options") as validation:
            with sdk2.parameters.CheckGroup("Allow JSONP validation options") as jsonp_validation_options:
                for key, desc, default in JSONP_BEHAVIORS:
                    jsonp_validation_options.values[key] = jsonp_validation_options.Value(desc, checked=default)

            with sdk2.parameters.CheckGroup("Skip validation for following page ids") as skipped_page_ids:
                for page_id, issue in SKIPPED_PAGE_ID_ISSUES.items():
                    skipped_page_ids.values[page_id] = skipped_page_ids.Value("{}: {}".format(page_id, issue), checked=True)

        with sdk2.parameters.Group("Report options") as reporting:
            report_max_line_width = sdk2.parameters.Integer("Max line width in report", default=500)
            fail_on_invalid_responses = sdk2.parameters.Bool("Fail task in case of invalid responses", default=False)
            with fail_on_invalid_responses.value[False]:
                max_invalid_responses_ratio = sdk2.parameters.Float(
                    "Max bad responses ratio",
                    default=0.1,
                    description="Fail task if bad responses ratio exceeds this threshold"
                )

        with sdk2.parameters.Group("Tools") as tools_group:
            unpacker_resource = sdk2.parameters.Resource(
                'Unpacker (dolbilo2json) resource',
                resource_type=YabsResponseDumpUnpacker)
            uc_resource = sdk2.parameters.Resource(
                'uc tool resource',
                resource_type=UCTool)

        with sdk2.parameters.Output:
            valid_responses = sdk2.parameters.Integer("Valid JSON/JSONP responses")
            invalid_responses = sdk2.parameters.Integer("Invalid JSON/JSONP responses")
            unsupported_responses = sdk2.parameters.Integer("Responses with unsupported Content-Type")
            broken_responses = sdk2.parameters.Integer("Responses that caused exception during parsing")
            skipped_responses = sdk2.parameters.Integer("Skipped responses")
            empty_responses = sdk2.parameters.Integer("Empty responses")
            not_parsed_responses = sdk2.parameters.Integer("Bad responses in general")

            validation_report_link = sdk2.parameters.String("Link to validation report")

    class Requirements(sdk2.Task.Requirements):
        disk_space = 125 * 1024
        environments = (
            environments.PipEnvironment('jsbeautifier', '1.10.1'),
            environments.PipEnvironment('demjson'),
            environments.PipEnvironment('yandex-yt', use_wheel=True),  # temporary fix for yql dependencies. https://ml.yandex-team.ru/thread/yt/171981210770270850/#message171981210770270855
            environments.PipEnvironment('yql', version='1.2.91'),
        )

    @trace_calls
    def write_meta_report(self, validation_results):
        meta_report_resource = new_resource(YabsResponseValidationMetaReport, self, 'Responses validation metareport', META_REPORT_FILENAME)
        meta_report_resource_data = new_resource_data(meta_report_resource)

        write_meta_report(
            str(meta_report_resource_data.path),
            validation_results
        )

        meta_report_resource_data.ready()

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        if not (self.Parameters.response_dumps or self.Parameters.shoot_task):
            raise TaskFailure('Either shoot task or responses dump resource required')

        shoot_reports_yt_path = None
        with self.memoize_stage.get_yt_path(commit_on_entrance=False, commit_on_wait=False), trace('get_yt_path'):
            if self.Parameters.shoot_task and self.Parameters.shoot_task.Parameters.upload_to_yt_task_id:
                upload_task_results = check_tasks(self, self.Parameters.shoot_task.Parameters.upload_to_yt_task_id, raise_on_fail=False)
                if all([status == ctt.Status.SUCCESS for _, status in upload_task_results]):
                    shoot_reports_yt_path = self.Parameters.shoot_task.Parameters.uploaded_logs_to_yt_prefix
            else:
                shoot_reports_yt_path = getattr(self.response_dumps[0], SHOOT_REPORTS_YT_NODE_KEY)

        from yql.api.v1.client import YqlClient

        yql_token = sdk2.Vault.data(YQL_TOKEN_VAULT_NAME)
        yql_client = YqlClient(token=yql_token)

        response_dumps = get_response_dumps(self.Parameters.shoot_task, self.Parameters.response_dumps)
        if self.Parameters.shoot_task:
            shoot_task = self.Parameters.shoot_task
        else:
            shoot_task = self.Parameters.response_dumps[0].task.id

        try:
            bad_requests_resource = YabsBadRequestsID.find(task_id=shoot_task.id, limit=1).first()
            logger.info("Found resource with bad requests id for shoot_task %s: %s", shoot_task, bad_requests_resource.id)
            with open(str(new_resource_data(bad_requests_resource).path), 'r') as bad_requests_resource_file:
                bad_request_ids = [str(int(line)) for line in bad_requests_resource_file if line]
            logger.info("Bad request ids are^ %s", bad_request_ids)
        except Exception:
            logger.warning("Can't find resource with bad requests id for shoot_task %s", shoot_task, exc_info=True)
            bad_request_ids = ()

        if self.Parameters.unpacker_resource:
            unpacker_resource = self.Parameters.unpacker_resource
        else:
            unpacker_resource = find_unpacker_resource(
                global_key=self.Parameters.shoot_task.Parameters.meta_server_resource.global_key,
                global_key_type=self.Parameters.shoot_task.Parameters.meta_server_resource.global_key_type,
                build_mode="release",
            )
        unpack_responses_dump_resource(
            self,
            dump_resources=response_dumps,
            destination_path=RESPONSES_DIR,
            uc_resource=self.Parameters.uc_resource,
            unpacker_resource=unpacker_resource,
        )

        jsonp_validation_options_dict = {"allow_{}".format(setting): True for setting in self.Parameters.jsonp_validation_options}
        validation_results = validate_responses(
            RESPONSES_DIR,
            jsonp_validation_options=jsonp_validation_options_dict,
            skipped_page_ids=self.Parameters.skipped_page_ids,
            bad_request_ids=bad_request_ids,
        )

        if self.Parameters.skipped_responses > 0:
            self.set_info(
                create_skipped_report(validation_results, SKIPPED_PAGE_ID_ISSUES),
                do_escape=False
            )

        status_counter = count_statuses(validation_results)

        self.Parameters.valid_responses = status_counter[ValidationStatus.OK.name].count
        self.Parameters.invalid_responses = status_counter[ValidationStatus.FAILED.name].count
        self.Parameters.unsupported_responses = status_counter[ValidationStatus.UNSUPPORTED.name].count
        self.Parameters.broken_responses = status_counter[ValidationStatus.EXCEPTION.name].count
        self.Parameters.skipped_responses = status_counter[ValidationStatus.SKIPPED.name].count
        self.Parameters.empty_responses = status_counter[ValidationStatus.EMPTY_RESPONSE.name].count
        self.Parameters.not_parsed_responses = status_counter[ValidationStatus.BROKEN_FILE.name].count

        report_dir_link = posixpath.join(execution_dir_link(self.id), REPORTS_DIR)
        self.Parameters.validation_report_link = report_dir_link

        yql_operation_links = create_yql_quieries(yql_client, shoot_reports_yt_path, validation_results) if shoot_reports_yt_path else {}

        report_dir_links = create_report_links(status_counter, report_dir_link)

        self.write_meta_report(validation_results)
        write_validation_report(self, validation_results, status_counter, self.Parameters.report_max_line_width, yql_operation_links=yql_operation_links)

        self.set_info(
            create_summary_report(status_counter, report_dir_links, yql_operation_links=yql_operation_links),
            do_escape=False
        )

        logger.info("Job is done")
        invalid_responses_count = sum((
            status_counter[ValidationStatus.FAILED.name].count,
            status_counter[ValidationStatus.EXCEPTION.name].count
        ))
        total_responses_count = sum((counter.count for counter in status_counter.values()))
        invalid_responses_ratio = invalid_responses_count / float(total_responses_count)

        if invalid_responses_count > 0 and self.Parameters.fail_on_invalid_responses:
            raise TaskFailure('Found {} invalid response(s)'.format(invalid_responses_count))
        elif invalid_responses_ratio > self.Parameters.max_invalid_responses_ratio:
            raise TaskFailure(
                'Too many invalid responses: {} while threshold was {}'.format(
                    invalid_responses_count, int(self.Parameters.max_invalid_responses_ratio * total_responses_count)
                )
            )


@trace_calls
def write_validation_report(task, validation_results, status_counter, max_line_width, yql_operation_links=None):
    reports_resource = new_resource(YabsResponseValidationReport, task, 'Responses validation reports', REPORTS_DIR)
    reports_resource_data = new_resource_data(reports_resource)

    write_full_report(
        str(reports_resource_data.path),
        validation_results,
        status_counter,
        max_line_width,
        [s.name for s in REPORTABLE_STATUSES],
        yql_operation_links=yql_operation_links,
    )

    reports_resource_data.ready()
    return reports_resource


def get_response_dumps(shoot_task, response_dumps):
    if response_dumps:
        return response_dumps

    resources_list = list(
        YabsServerResponseDump
        .find(task=shoot_task, state=ctr.State.READY)
        .order(sdk2.Resource.id)
        .limit(100)
    )
    return filter(lambda x: x.content_type != "secondary_count_links", resources_list)
