import json
import logging
import os
import posixpath
from collections import defaultdict, Counter, OrderedDict, namedtuple
from functools import partial
from multiprocessing import Pool as ProcessPool

from jinja2 import Environment, FunctionLoader

from sandbox.common.types.misc import NotExists
from sandbox.projects.common.startrek_utils import ST
from sandbox.projects.yabs.qa.utils.general import truncate_string

from .validate import ValidationStatus
from .yql_helpers import get_yql_operation_share_url, run_get_full_request_and_response_yt_operation

from sandbox.projects.yabs.sandbox_task_tracing import flush_trace


logger = logging.getLogger(__name__)

REPORTABLE_STATUSES = filter(lambda status: status != ValidationStatus.OK, ValidationStatus)
STATUS_DESCRIPTION_MAP = {
    ValidationStatus.OK.name: "Responses are valid",
    ValidationStatus.FAILED.name: "Responses are invalid",
    ValidationStatus.UNSUPPORTED.name: "Responses have unsupported Content-Type header",
    ValidationStatus.EXCEPTION.name: "Validation broken due to an exception",
    ValidationStatus.EMPTY_RESPONSE.name: "Responses with 200 HTTP status code and empty body",
    ValidationStatus.SKIPPED.name: "Validation skipped because of known issue",
    ValidationStatus.BROKEN_FILE.name: "File with response is broken",
}
STATUS_COLOR_MAP = {
    ValidationStatus.OK.name: "green",
    ValidationStatus.FAILED.name: "red",
    ValidationStatus.UNSUPPORTED.name: "grey",
    ValidationStatus.EXCEPTION.name: "brown",
    ValidationStatus.EMPTY_RESPONSE.name: "purple",
    ValidationStatus.SKIPPED.name: "blue",
    ValidationStatus.BROKEN_FILE.name: "black",
}


StatusCounter = namedtuple("StatusCounter", ("count", "increase", "decrease"))
StatusCounter.__new__.__defaults__ = (
    0,  # increase
    0,  # decrease
)


def load_template(template_name):
    template_path = os.path.normpath(
        os.path.join(
            os.path.dirname(__file__),
            '..',
            'templates',
            template_name,
        )
    )
    try:
        import library.python.resource as lpr
    except ImportError:
        with open(template_path, 'r') as f:
            template_text = f.read()
    else:
        template_text = lpr.find(template_path)

    return template_text


def get_template(template_name):
    env = Environment(loader=FunctionLoader(load_template))
    return env.get_template(template_name)


def count_statuses(validation_results):
    status_counter = defaultdict(int)
    for validation_result in validation_results:
        status_counter[validation_result["status"]] += 1

    return OrderedDict([
        (status.name, StatusCounter(status_counter.get(status.name, 0)))
        for status in ValidationStatus
    ])


def create_meta_report(validation_results):
    meta_report = {}
    for validation_result in validation_results:
        request_id = validation_result["request_id"]
        meta_report[request_id] = validation_result

    return meta_report


def create_yql_quieries(yql_client, shoot_reports_yt_path, validation_results):
    operation_links = {}

    has_shoot_reports_in_yt = shoot_reports_yt_path not in (None, NotExists, '')
    if has_shoot_reports_in_yt:
        full_request_and_response_yt_operation_url = None
        for status in REPORTABLE_STATUSES:
            failed_request_ids = list([r["request_id"] for r in validation_results if r["status"] == status.name])
            if len(failed_request_ids) > 0:
                operation_id = \
                    run_get_full_request_and_response_yt_operation(yql_client, failed_request_ids, shoot_reports_yt_path)
                full_request_and_response_yt_operation_url = get_yql_operation_share_url(operation_id)
            operation_links.update({status.name: "" if len(failed_request_ids) == 0 else full_request_and_response_yt_operation_url})
    else:
        operation_links = {
            status.name: ""
            for status in REPORTABLE_STATUSES
        }

    return operation_links


def create_report_links(status_counter, report_dir_link):
    return {
        status.name: "" if status_counter[status.name].count == 0 else posixpath.join(report_dir_link, status.name)
        for status in REPORTABLE_STATUSES
    }


def create_skipped_report(validation_results, skipped_page_id_issues):
    skipped_page_ids = Counter(
        validation_result["request_metainfo"]["page_id"]
        for validation_result in validation_results
        if validation_result["status"] == ValidationStatus.SKIPPED.name
    )

    issue_links = {
        page_id: ST.html_issue_link(issue)
        for page_id, issue in skipped_page_id_issues.items()
    }

    template = get_template("skipped.html")
    return template.render(
        skipped_page_ids=skipped_page_ids,
        issue_links=issue_links,
    )


def create_summary_report(status_counter, full_report_links, yql_operation_links=None, template_name='summary.html'):
    yql_operation_links = yql_operation_links or {}
    text_report_links = {
        status.name: full_report_links[status.name]
        for status in REPORTABLE_STATUSES
        if status_counter[status.name].count > 0
    }
    yql_report_links = {
        status.name: yql_operation_links[status.name]
        for status in REPORTABLE_STATUSES
        if status.name in yql_operation_links and status_counter[status.name].count > 0
    }

    template = get_template(template_name)
    return template.render(
        total_count=sum([c.count for c in status_counter.values()]),
        status_color_map=STATUS_COLOR_MAP,
        status_description_map=STATUS_DESCRIPTION_MAP,
        status_counter=status_counter,
        text_report_links=text_report_links,
        yql_report_links=yql_report_links,
    )


def make_validation_result_report(errors, max_line_width=200):
    validation_report = ''

    for error_info, error_lines in errors:
        truncated_error_lines = map(partial(truncate_string, limit=max_line_width, placeholder="...<truncated>"), error_lines.split("\n"))
        validation_report += (
            '{error_message}. At {position}.\n'
            '{error_lines}\n\n\n'
            .format(
                error_message=error_info["message"],
                position=error_info["position"],
                error_lines="\n".join(truncated_error_lines),
            )
        )

    return validation_report


def create_response_validation_report_html(validation_result, max_line_width, yql_operation_links=None):
    yql_operation_links = yql_operation_links or {}

    if validation_result["exception"]:
        full_validation_report = validation_result["exception"]
    else:
        full_validation_report = make_validation_result_report(validation_result["errors"], max_line_width=max_line_width)

    template = get_template("single_response_validation_report.html")
    return template.render(
        status_style_class=validation_result["status"].lower(),
        validation_result=validation_result,
        full_report=full_validation_report,
        yql_report_link=yql_operation_links.get(validation_result["status"]),
    )


def write_meta_report(meta_report_path, validation_results):
    meta_report = create_meta_report(validation_results)
    with open(meta_report_path, "w") as meta_report_file:
        json.dump(meta_report, meta_report_file)


def safe_write_single_full_report(validation_result, reports_base_path=".", max_line_width=200, reportable_status_names=(), yql_operation_links=None):
    request_id = validation_result["request_id"]
    try:
        write_single_full_report(validation_result, reports_base_path, max_line_width, reportable_status_names, yql_operation_links=yql_operation_links)
    except Exception:
        logger.error("Cannot write %s report due to an exception", request_id, exc_info=True)


def write_single_full_report(validation_result, reports_base_path, max_line_width, reportable_status_names, yql_operation_links=None):
    if validation_result["status"] not in reportable_status_names:
        return

    validation_result_html_report = create_response_validation_report_html(validation_result, max_line_width, yql_operation_links=yql_operation_links)
    request_id = validation_result["request_id"]

    report_folder_path = os.path.join(reports_base_path, validation_result["status"])
    report_file_path = os.path.join(report_folder_path, request_id + os.extsep + "html")
    with open(report_file_path, "w") as report_file:
        report_file.write(validation_result_html_report.encode('utf-8'))

    logger.debug('Validation report for response #%s saved to %s', request_id, report_file_path)


def write_full_report(reports_base_path, validation_results, status_counter, max_line_width, reportable_status_names, yql_operation_links=None):
    for status in REPORTABLE_STATUSES:
        report_folder_path = os.path.join(reports_base_path, status.name)
        if not os.path.isdir(report_folder_path):
            logger.info("%s does not exists, creating", report_folder_path)
            os.makedirs(report_folder_path)

    flush_trace()

    pool = ProcessPool()
    validation_results = pool.map(
        partial(
            safe_write_single_full_report,
            reports_base_path=reports_base_path,
            max_line_width=max_line_width,
            reportable_status_names=reportable_status_names,
            yql_operation_links=yql_operation_links,
        ),
        validation_results
    )
    pool.close()
    pool.join()
