import json
import logging
import os
import re
import requests
import sys
import traceback
from enum import Enum
from functools import partial
from multiprocessing import Pool as ProcessPool

from sandbox.projects.yabs.qa.utils.general import decode_text, truncate_string
from sandbox.projects.yabs.qa.response_tools.preparse import (
    extract_json_from_jsonp,
    JSON_RESPONSE_SPLITTER,
    UnexpectedEmptyString,
)

from .validate_json import validate_json

from sandbox.projects.yabs.sandbox_task_tracing import flush_trace, trace_calls


logger = logging.getLogger(__name__)
SKIPPED_CODES = (
    requests.codes.NO_CONTENT,
)


def get_content_type(response):
    try:
        return filter(lambda item: item['key'].lower() == 'Content-Type'.lower(), response['http']['headers'])[0]['value']
    except IndexError:
        return None


def get_code(response):
    return response['http']['code']


def get_request_snippet(response, limit=20):
    try:
        return truncate_string(decode_text(response['logs']['request'][0]['Request']), limit=limit)
    except (IndexError, ValueError, KeyError) as exc:
        logger.error("Cannot get request snippet because of %s", exc)
        return ""


def get_response_body(response):
    return decode_text(response['http']['entity'])


class UnsupportedContentType(Exception):
    pass


class ValidationStatus(Enum):
    OK = 0
    FAILED = 1
    EXCEPTION = 2
    UNSUPPORTED = 3
    BROKEN_FILE = 4
    SKIPPED = 5
    EMPTY_RESPONSE = 6


class Metainfo(object):
    def __str__(self):
        return "\n".join(("{}: {}".format(k, v) for k, v in self.__dict__.items()))


class ResponseMetainfo(Metainfo):
    def __init__(self, response):
        self.content_type = get_content_type(response)
        self.code = get_code(response)


class RequestMetainfo(Metainfo):
    def __init__(self, response):
        self.request_snippet = get_request_snippet(response, limit=40)
        metainfo_pattern = re.compile(r"(?P<method>[^\s]+)\s+/(?P<handler>(code|page|meta|metadsp))/(?P<page_id>[^\?^\.]+)")
        metainfo_match = metainfo_pattern.match(self.request_snippet)
        if metainfo_match is not None:
            for k, v in metainfo_match.groupdict().items():
                setattr(self, k, v)


class ValidationResult(object):
    def __init__(self, request_id, status, request_metainfo=None, response_metainfo=None, exception=None, errors=None):
        self.request_id = request_id
        self.status = status
        self.exception = exception
        self.errors = errors

        self.request_metainfo = request_metainfo
        self.response_metainfo = response_metainfo

    def as_dict(self):
        errors = [
            (
                {
                    "description": err.pretty_description(show_positions=True),
                    "message": err.message,
                    "position": str(err.position),
                },
                error_lines
            ) for err, error_lines in self.errors or []
        ]
        return {
            "request_id": self.request_id,
            "status": self.status.name,
            "exception": self.exception,
            "errors": errors,
            "request_metainfo": {} if self.request_metainfo is None else self.request_metainfo.__dict__,
            "response_metainfo": {} if self.response_metainfo is None else self.response_metainfo.__dict__,
        }


@trace_calls
def validate_response(response, content_type, jsonp_validation_options=None):
    jsonp_validation_options = jsonp_validation_options if jsonp_validation_options else {}
    if content_type is not None and content_type.lower().startswith('application/json'):
        response_body = get_response_body(response)
        raw_response_parts = response_body.split(JSON_RESPONSE_SPLITTER)
        return sum(map(validate_json, raw_response_parts), [])
    elif content_type is not None and content_type.lower().startswith('application/x-javascript'):
        response_body = get_response_body(response)
        return validate_json(extract_json_from_jsonp(response_body), **jsonp_validation_options)
    else:
        raise UnsupportedContentType('Unsupported content type: {}'.format(content_type))


# `trace_calls` decorator cannot be put here due to multiprocessing in `validate_responses` below
def validate_response_file(path, jsonp_validation_options=None, skipped_page_ids=(), bad_request_ids=()):
    response_filename = os.path.split(path)[1]

    with open(path) as response_file:
        try:
            response = json.load(response_file)
        except ValueError as exc:
            logger.warning('Failed to load file %s: %s', path, exc.message)
            return ValidationResult(
                response_filename,
                ValidationStatus.BROKEN_FILE,
                exception=str(exc),
            ).as_dict()

    request_metainfo = RequestMetainfo(response)
    response_metainfo = ResponseMetainfo(response)
    validation_errors = None
    exception = None
    if response_metainfo.code in SKIPPED_CODES:
        return ValidationResult(
            response_filename,
            ValidationStatus.OK,
            request_metainfo=request_metainfo,
            response_metainfo=response_metainfo,
            exception=exception,
            errors=validation_errors,
        ).as_dict()
    try:
        validation_errors = validate_response(response, response_metainfo.content_type, jsonp_validation_options=jsonp_validation_options)
    except UnsupportedContentType:
        status = ValidationStatus.UNSUPPORTED
        logger.warning('Response %s has unsupported content type: %s', response_filename, response_metainfo.content_type)
    except UnexpectedEmptyString:
        status = ValidationStatus.EMPTY_RESPONSE
    except Exception as exc:
        if getattr(request_metainfo, "page_id", None) in skipped_page_ids:
            logger.warning("Request %s was skipped because page_id %s is skipped", response_filename, getattr(request_metainfo, "page_id", None))
            status = ValidationStatus.SKIPPED
        elif response_filename in bad_request_ids:
            logger.warning("Request %s was skipped because response is in bad responses list", response_filename)
            status = ValidationStatus.SKIPPED
        else:
            status = ValidationStatus.EXCEPTION
        exc_type, exc_obj, tb = sys.exc_info()
        exception = "".join(traceback.format_exception(exc_type, exc_obj, tb))
        logger.warning('Response %s cannot be validated: %s', response_filename, exc_info=True)
    else:
        if not validation_errors and response_filename not in bad_request_ids:
            status = ValidationStatus.OK
        elif getattr(request_metainfo, "page_id", None) in skipped_page_ids:
            status = ValidationStatus.SKIPPED
        elif response_filename in bad_request_ids:
            logger.warning("Request %s was skipped because response is in bad responses list", response_filename)
            status = ValidationStatus.SKIPPED
        else:
            status = ValidationStatus.FAILED

    return ValidationResult(
        response_filename,
        status,
        request_metainfo=request_metainfo,
        response_metainfo=response_metainfo,
        exception=exception,
        errors=validation_errors,
    ).as_dict()


@trace_calls
def validate_responses(responses_dir, jsonp_validation_options=None, limit=None, skipped_page_ids=(), bad_request_ids=()):
    responses_paths = [os.path.join(responses_dir, filename) for filename in os.listdir(responses_dir)]
    if limit is not None:
        responses_paths = responses_paths[:limit]

    flush_trace()

    pool = ProcessPool()
    validation_results = pool.map(
        partial(
            validate_response_file,
            jsonp_validation_options=jsonp_validation_options,
            skipped_page_ids=skipped_page_ids,
            bad_request_ids=bad_request_ids,
        ),
        responses_paths
    )
    pool.close()
    pool.join()

    return validation_results
