import posixpath
import textwrap
import logging


logger = logging.getLogger(__name__)

YT_CLUSTER = 'hahn'


def get_yql_operation_url(operation_id):
    return posixpath.join('https://yql.yandex-team.ru/Operations/', operation_id)


def generate_query(request_ids_filename, reports_yt_node):
    return textwrap.dedent("""\
        PRAGMA yt.InferSchema;
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;

        USE {cluster};

        $requestIds = ParseFile("String", "{request_ids_filename}");

        $response = (
            SELECT RequestID, HttpCode, Response, RequestIDBinary FROM (
                SELECT MAX_BY(row, path) FROM (
                    SELECT TableRow() AS row, TablePath() AS path FROM RANGE(
                        '{primary_ammo_node}',
                        ``, -- from
                        ``, -- to
                        'response'
                    )
                    WHERE RequestID in $requestIds
                ) GROUP BY row.RequestID AS RequestID
            ) FLATTEN COLUMNS
        );

        SELECT
            response.RequestID as RequestID,
            request.Request as Request,
            response.Response as Response,
            response.HttpCode as HttpCode
        FROM
            `{primary_ammo_node}/0/request` AS request
        JOIN
            $response as response
        ON request.RequestID == response.RequestIDBinary
        WHERE response.RequestID in $requestIds;
    """.format(
        cluster=YT_CLUSTER,
        request_ids_filename=request_ids_filename,
        primary_ammo_node=posixpath.join(reports_yt_node, 'primary_ammo')
    ))


def run_get_full_request_and_response_yt_operation(yql_client, request_ids, reports_yt_node):
    request_ids_filename = "invalid_request_ids.txt"
    with open(request_ids_filename, "w") as request_ids_file:
        request_ids_file.write("\n".join([str(request_id) for request_id in request_ids]))

    query = generate_query(request_ids_filename=request_ids_filename, reports_yt_node=reports_yt_node)

    request = yql_client.query(query, syntax_version=1)
    request.attach_file(request_ids_filename, request_ids_filename)
    result = request.explain()
    return result.operation_id


def get_yql_operation_share_id(operation_id):
    try:
        from yql.client.operation import YqlOperationShareIdRequest
    except ImportError:
        logger.error('Failed to import YqlOperationShareIdRequest')
        return None

    share_id_request = YqlOperationShareIdRequest(operation_id)
    share_id_request.run()

    if share_id_request.is_ok:
        return share_id_request.json

    return None


def get_yql_operation_share_url(operation_id):
    share_id = get_yql_operation_share_id(operation_id)
    if share_id is None:
        return None

    return get_yql_operation_url(share_id)
