import json
import logging
import posixpath
from collections import defaultdict, OrderedDict

from sandbox import sdk2
from sandbox.common.types.resource import State as ResourceState
from sandbox.projects.yabs.qa.tasks.base_compare_task import parameters as base_compare_task_parameters
from sandbox.projects.yabs.qa.tasks.base_compare_task.task import BaseCompareTask
from sandbox.projects.yabs.qa.utils import resource
from sandbox.projects.yabs.qa.tasks.YabsServerValidateResponses import (
    YabsResponseValidationMetaReport,
    write_validation_report,
)
from sandbox.projects.yabs.qa.tasks.YabsServerValidateResponses.utils import report, validate
from sandbox.projects.yabs.qa.resource_types import BaseBackupSdk2Resource

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import new_resource
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import new_resource_data


def compare_metareports(pre, test):
    results = {
        "changed": defaultdict(list),
        "unchanged": defaultdict(list),
        "added": defaultdict(list),
        "removed": defaultdict(list),
    }

    pre_test_ids = set(pre.keys())
    test_test_ids = set(test.keys())
    skipped_test_ids = \
        set(request_id for request_id, test_result in pre.items() if test_result["status"] == validate.ValidationStatus.SKIPPED.name) | \
        set(request_id for request_id, test_result in test.items() if test_result["status"] == validate.ValidationStatus.SKIPPED.name)
    test_ids_to_compare = pre_test_ids & test_test_ids - skipped_test_ids
    logging.info("Comparing validation results for %s", test_ids_to_compare)
    extra_pre_test_ids = pre_test_ids - test_test_ids
    extra_test_test_ids = test_test_ids - pre_test_ids

    for test_id in test_ids_to_compare:
        pre_result = pre[test_id]
        test_result = test[test_id]

        result_class = "unchanged" if pre_result["status"] == test_result["status"] else "changed"
        results[result_class][(pre_result["status"], test_result["status"])].append(test_result)

    for test_id in extra_pre_test_ids:
        pre_result = pre[test_id]
        results["removed"][(pre_result["status"], None)].append(pre_result)

    for test_id in extra_test_test_ids:
        test_result = test[test_id]
        results["added"][(None, test_result["status"])].append(test_result)

    return results


class YabsServerValidationDiffReport(BaseBackupSdk2Resource):
    auto_backup = True


class YabsServerValidateResponsesCmp(BaseCompareTask):

    class Parameters(base_compare_task_parameters.BaseCompareTaskParameters):

        with sdk2.parameters.Group("Report options") as reporting:
            report_max_line_width = sdk2.parameters.Integer("Max line width in report", default=500)

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less
        disk_space = 50 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_metareport(self, run_type):
        metareport_resource = YabsResponseValidationMetaReport.find(
            task_id=getattr(self, '{}_task'.format(run_type)).id,
            limit=1,
            status=ResourceState.READY,
        ).first()

        metareport_file_path = resource.sync_resource(resource=metareport_resource)
        with open(metareport_file_path, "r") as f:
            metareport_data = json.load(f)

        return metareport_data

    def write_diff_report(self, report_html):
        report_resource = new_resource(YabsServerValidationDiffReport, self, 'Responses validation diff report', 'index.html')
        report_resource_data = new_resource_data(report_resource)

        with open(str(report_resource_data.path), 'w') as report_file:
            report_file.write(report_html)

        report_resource_data.ready()

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        pre_metareport_data = self.get_metareport('pre')
        test_metareport_data = self.get_metareport('test')

        comparison_result = compare_metareports(pre_metareport_data, test_metareport_data)
        status_change_counter = {
            status.name: {
                "count": 0,
                "increase": 0,
                "decrease": 0
            }
            for status in validate.ValidationStatus
        }
        changed_validation_results = []
        total_count = 0
        changed_count = sum([
            len(validation_results)
            for (old_status, new_status), validation_results in comparison_result['changed'].items()
            if report.ValidationStatus[new_status] in report.REPORTABLE_STATUSES
        ])

        for result_class, result_data in comparison_result.items():
            for (old_status, new_status), validation_results in result_data.items():
                total_count += len(validation_results)

                if new_status is not None:
                    status_change_counter[new_status]['count'] += len(validation_results)

                if new_status == old_status:
                    continue

                if old_status is not None:
                    status_change_counter[old_status]['decrease'] += len(validation_results)

                if new_status is not None:
                    logging.debug("Got new %s response", new_status)
                    status_change_counter[new_status]['increase'] += len(validation_results)
                    changed_validation_results.extend(validation_results)

        status_counter = OrderedDict([
            (status, report.StatusCounter(counter["count"], counter["increase"], counter["decrease"]))
            for status, counter in sorted(status_change_counter.items(), key=lambda (s, c): validate.ValidationStatus[s].value)
        ])

        report_resource = None
        if changed_count > 0:
            report_resource = write_validation_report(self, changed_validation_results, status_counter, self.Parameters.report_max_line_width)

        full_report_links = {
            status.name: "" if report_resource is None else posixpath.join(report_resource.http_proxy, status.name)
            for status in report.REPORTABLE_STATUSES
        }
        diff_report_summary_st = 'No validation status change'
        if changed_count > 0:
            diff_report_summary = report.create_summary_report(status_counter, full_report_links, template_name='diff_report.html')
            diff_report_summary_st = report.create_summary_report(status_counter, full_report_links, template_name='diff_report.md')
            self.write_diff_report(diff_report_summary)

        task_info_summary = report.create_summary_report(status_counter, full_report_links)
        self.set_info(task_info_summary, do_escape=False)

        self.Parameters.has_diff = self.Context.has_diff = changed_count > 0
        self.Parameters.st_report = diff_report_summary_st
