import logging
import os
import re
import textwrap
import time

from sandbox import sdk2
import sandbox.common.errors as sandbox_errors
import sandbox.projects.yabs.qa.utils.arcadia as arcadia_utils
import sandbox.projects.yabs.qa.utils.general as yabs_qa_utils
from sandbox.sandboxsdk import ssh, environments
from sandbox.sdk2.vcs import svn
from sandbox.projects.yabs.qa.resource_types import BaseBackupAbstractResource


logger = logging.getLogger(__name__)


ARCADIA_ONESHOTS_ROOT_DIR = 'yabs/qa/oneshots'
COMMON_ONESHOTS_DIR = 'common'
LOCAL_ARCADIA_DIR = 'arcadia'
ONESHOT_FILENAME = 'oneshot.py'
YA_MAKE_FILENAME = 'ya.make'
COMMIT_USER = 'robot-yabs-oneshot'
SSH_KEY_VAULT_KEY = 'robot-yabs-oneshot-rsa'
BSINT_URL = 'http://bsint.yandex-team.ru/devapi/mysqlquery.cgi'
SVN_ARCADIA_PREFIX = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia'
MAX_DEPTH_FOR_GRAPH_CONNECTIVITY = 3  # Used to connect new oneshots to Arcadia root

RECURSE_TEMPLATE = '''\
RECURSE(
    {relative_target}
)
'''


def is_under_version_control(path):
    try:
        svn.Arcadia.svn('info', url=path)
    except svn.SvnError:
        return False

    return True


def add_to_version_control(path):
    if is_under_version_control(path):
        logger.info('Path is under version control: %s', path)
    else:
        logger.info('Add path to version control: %s', path)
        svn.Arcadia.add(path, parents=True)


def prepare_oneshot_dir(local_oneshot_dir):
    logger.info('svn update %s', local_oneshot_dir)
    svn.Arcadia.update(local_oneshot_dir, parents=True)

    logger.info('Create directory: %s', local_oneshot_dir)
    try:
        os.makedirs(local_oneshot_dir)
    except OSError as e:
        if e.errno == 17:
            # File exists
            pass
        else:
            raise e


class TextYtOneshot(BaseBackupAbstractResource):
    auto_backup = True
    ttl = 7


class BaseYtOneshot(sdk2.Task):
    """Base class for YT oneshot tasks"""

    BSINT_REVIEW_GROUP = 'server-tables'

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        environments = (
            environments.PipEnvironment('startrek_client', use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 0

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('SVN url for arcadia', required=True)
        ticket = sdk2.parameters.StrictString(
            'Startrek ticket',
            required=True,
            regexp=r'^[A-Z]+-\d+$')
        oneshot_directory_name = sdk2.parameters.StrictString(
            'Oneshot directory name',
            description='Must match regexp "^[a-z0-9_-]*$"',
            required=True,
            regexp=r'^[a-z0-9_-]*$')
        create_bsint_oneshot = sdk2.parameters.Bool('Create bsint oneshot', default=True)
        task_mode = sdk2.parameters.RadioGroup(
            'Task mode',
            choices=(
                    ('create_bsint_oneshot', 'create_bsint_oneshot'),
                    ('create_oneshot_locally', 'create_oneshot_locally'),
            ),
            default='create_bsint_oneshot'
        )
        with task_mode.value["create_bsint_oneshot"]:
            oneshot_description = sdk2.parameters.String('Oneshot description for ticket', required=True)
            oneshot_assignee = sdk2.parameters.String(
                'Oneshot assignee for ticket',
                description="If empty the author of the task will be used"
            )

    def generate_oneshot_file_content(self):
        raise NotImplementedError()

    def generate_ya_make_file_content(self):
        raise NotImplementedError()

    def bsint_create_oneshot_request(self, arcadia_oneshot_path):
        import requests
        logger.info("Create bsint yt oneshot")
        username = self.Parameters.oneshot_assignee
        if not username:
            username = self.author
        data = {
            'username': username,
            'act': 'create',
            'scripttype': 'yt',
            'code_arcadia': arcadia_oneshot_path,
            'startdate': '2020-01-01',
            'starthour': '-1',
            'testrun': '1',
            'ccusers': '',
            'worktask': self.Parameters.ticket,
            'review_group': self.BSINT_REVIEW_GROUP,
            'timeout': '3600',
            'comments': self.Parameters.oneshot_description,
        }
        logger.debug("url: %s, data: %s", BSINT_URL, data)
        raw_response = requests.post(BSINT_URL, data)
        logger.debug("bsint response: %d, %s", raw_response.status_code, raw_response.text)
        response = raw_response.json()
        if response.get("success"):
            link = yabs_qa_utils.html_hyperlink(link="https://st.yandex-team.ru/{ticket}".format(ticket=response["taskid"]), text=response["taskid"])
            self.set_info("Oneshot ticket: {link}".format(link=link), do_escape=False)
        else:
            self.set_info("Something wrong. Do not restart task, create oneshot manually from bsint")
            raise sandbox_errors.TaskError("bsint response: {message}".format(message=str(response)))
        return response

    def commit_changes(self, path):
        logger.info('Commit changes')
        commit_message = '{ticket} create yt oneshot {description}'.format(
            ticket=self.Parameters.ticket,
            description=self.Parameters.oneshot_directory_name
        )
        try:
            with ssh.Key(self, key_owner=COMMIT_USER, key_name=SSH_KEY_VAULT_KEY):
                svn.Arcadia.commit(path=path, message=commit_message, user=COMMIT_USER)
        except svn.SvnError as e:
            logger.debug(e)
            match = re.search(r'^https:\/\/a\.yandex-team\.ru\/review\/\d+$', str(e), re.MULTILINE)
            try:
                arcanum_review_url = match.group(0)
            except IndexError:
                return None
            return arcanum_review_url
        return None

    def ensure_user_follows_issue(self, username, ticket):
        from startrek_client import Startrek
        import startrek_client.exceptions

        startrek_token = sdk2.Vault.data('robot-yabs-oneshot-startrek-token')
        st_client = Startrek(useragent='sandbox task #' + str(self.id), token=startrek_token)

        try:
            issue = st_client.issues[ticket]
            # self.ensure_user_follows_issue(COMMIT_USER, self.Parameters.ticket)
        except startrek_client.exceptions.Forbidden:
            error_message = (
                'user "{username}" has no access to startrek issue {ticket}'
                .format(username=COMMIT_USER, ticket=self.Parameters.ticket))
            raise sandbox_errors.TaskError(error_message)
        except startrek_client.exceptions.NotFound:
            error_message = (
                'Issue {ticket} not found'.format(ticket=self.Parameters.ticket))
            raise sandbox_errors.TaskFailure(error_message)

        issue.update(followers=issue.followers + [username])

    def create_oneshot_file(self, path):
        oneshot_file_content = self.generate_oneshot_file_content()
        logger.debug('Oneshot file content: %s', oneshot_file_content)
        with open(path, 'w') as oneshot_file:
            oneshot_file.write(oneshot_file_content)

    def create_ya_make_file(self, path):
        ya_make_file_content = self.generate_ya_make_file_content()
        with open(path, 'w') as oneshot_file:
            oneshot_file.write(ya_make_file_content)

    def create_resource(self, ya_make_file_path, oneshot_file_path):
        resource = TextYtOneshot(self, "Yt oneshot resource", self.Parameters.oneshot_directory_name)
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        with open(oneshot_file_path) as f:
            resource_data.path.joinpath(ONESHOT_FILENAME).write_bytes(f.read())
        with open(ya_make_file_path) as f:
            resource_data.path.joinpath(YA_MAKE_FILENAME).write_bytes(f.read())
        resource_data.ready()
        return resource

    def create_command_for_local_merge(self, resource):
        arcadia_common_oneshots_dir = os.path.join(ARCADIA_ONESHOTS_ROOT_DIR, COMMON_ONESHOTS_DIR)
        arcadia_oneshots_dir = os.path.join(arcadia_common_oneshots_dir, self.author, self.Parameters.ticket)
        arcadia_oneshots_name_dir = os.path.join(arcadia_oneshots_dir, self.Parameters.oneshot_directory_name)
        self.set_info(
            """Execute next command in your arcadia root path:
            sky get -d {oneshots_dir} -w {skynetid}; ya svn add {common_dir} --force; cd {common_dir}; ya make -rE --dist; ./create_package_json
            Oneshot will be located at {name_dir}""".format(
            oneshots_dir=arcadia_oneshots_dir, skynetid=resource.skynet_id, common_dir=arcadia_common_oneshots_dir, name_dir=arcadia_oneshots_name_dir)
        )

    def on_execute(self):
        self.ensure_user_follows_issue(COMMIT_USER, self.Parameters.ticket)

        logger.info('Checkout empty arcadia to %s', LOCAL_ARCADIA_DIR)
        svn.Arcadia.checkout(url=self.Parameters.checkout_arcadia_from_url, path=LOCAL_ARCADIA_DIR, depth='empty')

        arcadia_oneshot_dir = os.path.join(
            ARCADIA_ONESHOTS_ROOT_DIR, self.author, self.Parameters.ticket, self.Parameters.oneshot_directory_name)
        local_oneshot_dir = os.path.join(LOCAL_ARCADIA_DIR, arcadia_oneshot_dir)
        prepare_oneshot_dir(local_oneshot_dir)

        oneshot_file_path = os.path.join(local_oneshot_dir, ONESHOT_FILENAME)
        logger.info('Create oneshot file: %s', oneshot_file_path)
        self.create_oneshot_file(oneshot_file_path)

        ya_make_file_path = os.path.join(local_oneshot_dir, YA_MAKE_FILENAME)
        logger.info('Create ya.make file: %s', ya_make_file_path)
        self.create_ya_make_file(ya_make_file_path)

        ya_make_graph_paths = _connect_ya_make_to_graph(local_oneshot_dir, MAX_DEPTH_FOR_GRAPH_CONNECTIVITY, logger)

        if self.Parameters.task_mode == "create_bsint_oneshot":
            logger.info('Commit oneshot file')
            add_to_version_control(oneshot_file_path)
            logger.info('Commit ya.make file for the oneshot')
            add_to_version_control(ya_make_file_path)
            logger.info('Commit ya.make files for ya make graph connectivity')
            for path in ya_make_graph_paths:
                add_to_version_control(path)

        elif self.Parameters.task_mode == "create_oneshot_locally":
            logger.info("Save text of oneshot into resource")
            resource = self.create_resource(ya_make_file_path, oneshot_file_path)
            self.create_command_for_local_merge(resource)
            return

        logger.debug('svn status:\n%s', svn.Arcadia.status(LOCAL_ARCADIA_DIR))

        arcanum_review_url = self.commit_changes(LOCAL_ARCADIA_DIR)
        if arcanum_review_url is not None:
            arcanum_review_hyperlink = yabs_qa_utils.html_hyperlink(link=arcanum_review_url, text=arcanum_review_url)
            self.set_info('Arcanum review created: {}'.format(arcanum_review_hyperlink), do_escape=False)

        oneshot_arcadia_url = os.path.join(
            self.Parameters.checkout_arcadia_from_url, arcadia_oneshot_dir, ONESHOT_FILENAME)
        oneshot_arcanum_url = arcadia_utils.get_arcanum_url(oneshot_arcadia_url)
        oneshot_hyperlink = yabs_qa_utils.html_hyperlink(link=oneshot_arcanum_url, text=oneshot_arcanum_url)
        self.set_info('Oneshot will be located at {}'.format(oneshot_hyperlink), do_escape=False)

        if self.Parameters.create_bsint_oneshot:
            logger.info("Wait until commit will be merged")
            arcadia_oneshot_path = os.path.join(arcadia_oneshot_dir, ONESHOT_FILENAME)
            svn_oneshot_file_path = os.path.join(SVN_ARCADIA_PREFIX, arcadia_oneshot_path)
            while not is_under_version_control(svn_oneshot_file_path):
                logger.info("Waiting for PR {link} to be merged to trunk".format(link=arcanum_review_url))
                time.sleep(60)
            logger.info("Commit {link} is successfully merged".format(link=arcanum_review_url))
            self.bsint_create_oneshot_request(arcadia_oneshot_path)


def _generate_ya_make_file_content(owner, peerdirs):
    peerdir = '\n    '.join(peerdirs)
    ya_make_template = textwrap.dedent('''\
        PY3_PROGRAM(oneshot)

        OWNER(
            {owner}
            g:yabs-qa
        )

        PEERDIR(
            {peerdir}
        )

        PY_SRCS(MAIN oneshot.py)

        END()
    ''')
    return ya_make_template.format(owner=owner, peerdir=peerdir)


def _connect_ya_make_to_graph(target_path, max_depth_left, logger=None):
    if max_depth_left <= 0:
        return []
    current_directory, current_child_to_recurse = os.path.normpath(target_path).rsplit(os.path.sep, 1)
    current_ya_make_path = os.path.join(current_directory, YA_MAKE_FILENAME)
    ya_make_content = ''
    try:
        svn.Arcadia.update(current_ya_make_path)
        with open(current_ya_make_path, 'r') as ya_make:
            ya_make_content = ya_make.read()
    except svn.SvnError:  # File does not exist in the remote repo
        pass
    if current_child_to_recurse in ya_make_content:  # parent ya.make already contains current_child_to_recurse
        logger.info("ya make graph already connected, found {}".format(current_ya_make_path))
        return []
    with open(current_ya_make_path, 'w') as ya_make:
        if logger:
            logger.info('Wrote ya.make to {} to ensure ya make graph connectivity'.format(current_ya_make_path))
        ya_make.write(_generate_new_ya_make_for_graph(ya_make_content, current_child_to_recurse))
    return _connect_ya_make_to_graph(current_directory, max_depth_left - 1, logger) + [current_ya_make_path]


def _generate_new_ya_make_for_graph(old_ya_make_content, current_target):
    if not old_ya_make_content:
        return RECURSE_TEMPLATE.format(relative_target=current_target)
    splitter = r'([\t ]*)(RECURSE\s*\()'
    ya_make_content_parts = re.split(splitter, old_ya_make_content, maxsplit=1)
    if len(ya_make_content_parts) < 4:
        return old_ya_make_content.rstrip('\n') + '\n' + RECURSE_TEMPLATE.format(relative_target=current_target)
    prefix, indent, recurse, postfix = ya_make_content_parts
    added_line = '\n{indent}    {relative_target}'.format(indent=indent, relative_target=current_target)
    return "".join((prefix, indent, recurse, added_line, postfix))
