from abc import abstractproperty
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.yabs.qa.tasks.base_compare_task import parameters
from sandbox.projects.yabs.qa.tasks.YabsBinarySearchHelper import YabsBinarySearchHelper
from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink
import logging

from sandbox.projects.yabs.sandbox_task_tracing import trace
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import enqueue_task


def create_input_parameters(default_max_diff_percent=0.0):
    class CmpInputParameters(sdk2.Parameters):
        with sdk2.parameters.Group('Checking parameters') as checking_params:
            max_diff_percent = sdk2.parameters.Float('Max diff (%)', default=default_max_diff_percent)
            compare_same_bases = sdk2.parameters.Bool('Compare the same bases from pre and test', default=False)
            check_input_param_for_equality = sdk2.parameters.String('Name of the input parameter of the check task. Verify that this parameter value is the same for pre and test', default='')

    return CmpInputParameters


class BaseCompareTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 1GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(parameters.BaseCompareTaskParameters):
        pass

    @property
    def pre_task(self):
        return self.Parameters.pre_task

    @property
    def test_task(self):
        return self.Parameters.test_task

    @abstractproperty
    def ignore_compare_input_parameters(self):
        """Input parameters of the compared tasks that will be ignored if differ"""
        return []

    @abstractproperty
    def critical_compare_input_parameters(self):
        """Input parameters of the compared tasks that must be equal"""
        return []

    @abstractproperty
    def diff_resource_type(self):
        pass

    @abstractproperty
    def diff_resource_search_attributes(self):
        pass

    def _get_compared_task_context(self, key):
        task = getattr(self.Parameters, key)
        if task.status != 'SUCCESS':
            raise TaskFailure('Task %s is not in SUCCESS status' % task)
        return task.Context

    def compare_tasks_parameters(self):
        pre_task_id = self.pre_task.id
        test_task_id = self.test_task.id

        pre_task_input_parameters = self.server.task[pre_task_id].read()['input_parameters']
        test_task_input_parameters = self.server.task[test_task_id].read()['input_parameters']

        different_parameters = {}
        for key in (set(pre_task_input_parameters.keys()) & set(test_task_input_parameters.keys())) - set(self.ignore_compare_input_parameters):
            if pre_task_input_parameters[key] != test_task_input_parameters[key]:
                different_parameters[key] = (pre_task_input_parameters[key], test_task_input_parameters[key])

        if different_parameters:
            report_lines = []
            for key, (pre_value, test_value) in different_parameters.items():
                report_lines.append('Parameter {}:\n-{}\n+{}'.format(key, str(pre_value), str(test_value)))

            self.set_info('Different input parameters:\n{}'.format('\n'.join(report_lines)))

        different_critical_keys = set(different_parameters.keys()) & set(self.critical_compare_input_parameters)
        return different_critical_keys

    def check_tasks_parameters(self):
        terminate_task = False
        with self.memoize_stage.compare_input_parameters:
            if not self.Parameters.compare_input_parameters:
                return False

            different_critical_keys = self.compare_tasks_parameters()
            if not different_critical_keys:
                return False

            if self.Parameters.critical_parameters_mode == parameters.CriticalParametersDiffHandleMode.fail_task:
                raise TaskFailure('Critical input parameters [{}] differ, can not continue'.format(', '.join(different_critical_keys)))
            elif self.Parameters.critical_parameters_mode == parameters.CriticalParametersDiffHandleMode.mark_as_diff:
                self.set_info('Critical input parameters [{}] differ, consider this as a diff'.format(', '.join(different_critical_keys)))
                self.Parameters.has_diff = True
                self.Context.has_diff = True
                terminate_task = True
            elif self.Parameters.critical_parameters_mode == parameters.CriticalParametersDiffHandleMode.do_nothing:
                self.set_info('Critical input parameters [{}] differ, but I\'ve been told to ignore it'.format(', '.join(different_critical_keys)))
            else:
                self.set_info('Critical input parameters [{}] differ, but I don\'t know what to do with it'.format(', '.join(different_critical_keys)))

        return terminate_task

    def on_success(self, prev_status):
        if 'TESTENV-COMMIT-CHECK' in self.Parameters.tags and self.Parameters.binary_search_helper and self.Context.has_diff:
            logging.debug('current_diff_resource {}'.format(self.diff_resource_type))
            try:
                self.run_binary_search_helper()
                logging.debug('Binary_search_helper start')
            except Exception as e:
                logging.error('Got exception %s', e, exc_info=True)

    def run_binary_search_helper(self):
        try:
            with self.memoize_stage.run_helper(commit_on_entrance=False), trace('run_helper'):
                binary_search_helper_task = YabsBinarySearchHelper(
                    self,
                    description="Help for task {}".format(get_task_html_hyperlink(self.id)),
                    owner=self.owner,
                    pre_revision=self.Parameters.pre_revision,
                    test_revision=self.Parameters.test_revision,
                    find_key=self.diff_resource_search_attributes,
                    testenv_project=self.Context.testenv_database,
                    test_name=self.Parameters.test_name,
                    diff_resource_type=self.diff_resource_type,
                )
                enqueue_task(binary_search_helper_task)
                self.Context.binary_search_helper_task_id = binary_search_helper_task.id
                self.set_info("Run binary search helper. Task {}".format(get_task_html_hyperlink(binary_search_helper_task.id)), do_escape=False)
        except Exception as e:
            logging.error('Got exception %s', e, exc_info=True)
            self.set_info("Cannot run binary search helper. {}".format(e.message))
