# -*- coding: utf-8 -*-

import json
import logging

from sandbox import sdk2
from sandbox.common.types.misc import NotExists
import sandbox.common.types.task as ctt

from sandbox.projects.common.yabs.server.db import yt_bases
from sandbox.projects.common.yabs.server.db.utils import (
    get_cs_settings,
    get_yabscs,
    get_cs_import_ver,
    get_cs_advmachine_export_phrase_info_ver,
    get_cs_advmachine_export_banners_ver,
    get_importer_code_ver,
    get_full_mkdb_info,
)
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink
from sandbox.projects.yabs.qa.utils.resource import json_from_resource
from sandbox.projects.yabs.qa.tasks.YabsServerCalculateCSImportDigest import YabsServerCalculateCSImportDigest

from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import enqueue_task
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import new_resource_data


logger = logging.getLogger(__name__)


class CSHelper(object):

    @property
    def cs_import_ver(self):
        if not bool(self.Context.cs_import_ver):
            bs_release_yt_dir = self._get_yabscs()
            try:
                cs_import_ver = get_cs_import_ver(bs_release_yt_dir)
            except Exception as e:
                logger.exception(e)
                cs_import_ver = None
            self.Context.cs_import_ver = cs_import_ver
        return self.Context.cs_import_ver

    @property
    def get_cs_advmachine_export_phrase_info_ver(self):
        if self.Context.cs_advmachine_export_phrase_info_ver is NotExists:
            bs_release_yt_dir = self._get_yabscs()
            try:
                cs_advmachine_export_phrase_info_ver = get_cs_advmachine_export_phrase_info_ver(bs_release_yt_dir)
            except Exception as e:
                logger.exception(e)
                cs_advmachine_export_phrase_info_ver = None
            self.Context.cs_advmachine_export_phrase_info_ver = cs_advmachine_export_phrase_info_ver
        return self.Context.cs_advmachine_export_phrase_info_ver

    @property
    def get_cs_advmachine_export_banners_ver(self):
        if self.Context.cs_advmachine_export_banners_ver is NotExists:
            bs_release_yt_dir = self._get_yabscs()
            try:
                cs_advmachine_export_banners_ver = get_cs_advmachine_export_banners_ver(bs_release_yt_dir)
            except Exception as e:
                logger.exception(e)
                cs_advmachine_export_banners_ver = None
            self.Context.cs_advmachine_export_banners_ver = cs_advmachine_export_banners_ver
        return self.Context.cs_advmachine_export_banners_ver

    @property
    def base_tags(self):
        if self.Context.base_tags is NotExists:
            self.Context.base_tags = self.get_base_tags()

        return self.Context.base_tags

    @property
    def archive_root_path(self):
        if self.Context.archive_root_path is NotExists:
            input_spec = self.get_input_spec(self.Parameters.input_spec)
            validate_input_spec(input_spec)
            self.Context.archive_root_path = input_spec[yt_bases.AUXILLARY_DATA_IN_SPEC_KEY][yt_bases.ARCHIVE_ROOT_KEY]

        return self.Context.archive_root_path

    @property
    def mkdb_info(self):
        return get_full_mkdb_info(self._get_yabscs())

    @property
    def cs_settings(self):
        if self.Context.cs_settings is NotExists:
            self.Context.cs_settings = get_cs_settings(self, self.cs_settings_archive_res_id, self.cs_settings_patch_res_id, self.Parameters.settings_spec)

        return self.Context.cs_settings

    @property
    def cs_settings_version(self):
        if self.Context.cs_settings_version is NotExists:
            self.Context.cs_settings_version = yt_bases.get_cs_settings_version(self._get_yabscs(), self.cs_settings)

        return self.Context.cs_settings_version

    @property
    def cs_settings_archive_res_id(self):
        if not self.Parameters.settings_archive:
            return None
        return self.Parameters.settings_archive.id

    @property
    def cs_settings_patch_res_id(self):
        if not self.Parameters.cs_settings_patch:
            return None
        return self.Parameters.cs_settings_patch.id

    @property
    def importer_code_version(self):
        if self.Context.importer_code_version is NotExists:
            self.Context.importer_code_version = get_importer_code_ver(self._get_yabscs())

        return self.Context.importer_code_version

    @property
    def node_to_reuse(self):
        return self.Context.node_to_reuse

    @node_to_reuse.setter
    def node_to_reuse(self, value):
        self.Context.node_to_reuse = value

    def _get_yabscs(self):
        """Sync and unpack BS_RELEASE_YT"""
        try:
            return self._yabscs_path
        except AttributeError:
            yabscs_res_id = self.Parameters.bs_release_yt_resource
            self._yabscs_path = get_yabscs(self, yabscs_res_id)
            return self._yabscs_path

    def sync_resource(self, resource):
        if isinstance(resource, int):
            resource = sdk2.Resource[resource]
        resource_data = new_resource_data(resource)
        return unicode(resource_data.path)

    def get_base_tags(self):
        """Should be callable from on_enqueue"""
        tags = self.Parameters.bin_db_list.split()

        return list(set(tags))

    def get_input_spec(self, resource_id):
        input_spec_path = self.sync_resource(resource_id)
        with open(input_spec_path) as spec_file:
            return json.load(spec_file)

    def calculate_digest(self, import_destination_path, wait_task, hashes_history_dir="", skip_list=None):
        """
        Runs YABS_SERVER_CALCULATE_CS_IMPORT_DIGEST to get digest
        :param import_destination_path: path to cs_import resulting node
        :return: calculated digest
        """
        with self.memoize_stage.calculate_digest(commit_on_entrance=False):
            description = 'Calculating digest for node {0} (subtask of {1})'.format(import_destination_path, self.id)
            digest_task = YabsServerCalculateCSImportDigest(self, description=description, owner=self.owner)
            digest_task.Parameters.import_destination_path = import_destination_path
            digest_task.Parameters.hashes_history_dir = hashes_history_dir
            if skip_list is not None:
                digest_task.Parameters.skip_list = skip_list
            if not wait_task:
                # Separate semaphore and pool for async digests
                digest_task.Parameters.yt_pool = 'yabs-cs-sandbox-infra'
                digest_task.Requirements.semaphores = ctt.Semaphores(
                    acquires=[
                        ctt.Semaphores.Acquire(name="yabscs/pools/yabs-cs-sandbox-infra")
                    ],
                )
            digest_task.Requirements.tasks_resource = self.Requirements.tasks_resource  # for testing purposes
            enqueue_task(digest_task.save())
            logger.info('Run digest calculation task #%d', digest_task.id)
            self.set_info('Run digest calculation task {}'.format(get_task_html_hyperlink(digest_task.id)),
                          do_escape=False)
        if not wait_task:
            return
        # XXX: looks racy
        digest_task = self.find(YabsServerCalculateCSImportDigest).order(-sdk2.Task.id).first()
        check_tasks(self, digest_task)
        return digest_task.Context.digest


def validate_input_spec(spec):
    try:
        spec['proxy']
        spec[yt_bases.AUXILLARY_DATA_IN_SPEC_KEY][yt_bases.ARCHIVE_ROOT_KEY]
        spec[yt_bases.AUXILLARY_DATA_IN_SPEC_KEY][yt_bases.ARCHIVE_ROOT_TTL_KEY]
    except KeyError as e:
        raise InvalidInputSpec('Missing required key: {}'.format(e))


def get_spec_archive_root_path(input_spec_resource):
    input_spec = json_from_resource(input_spec_resource)
    validate_input_spec(input_spec)
    return input_spec[yt_bases.AUXILLARY_DATA_IN_SPEC_KEY][yt_bases.ARCHIVE_ROOT_KEY]


class InvalidInputSpec(Exception):
    pass
