import logging
import multiprocessing
import os
from collections import namedtuple

from sandbox import sdk2
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_BASE_PACK_EXECUTABLE

from sandbox.projects.yabs.sandbox_task_tracing.wrappers import subprocess
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import new_resource_data


logger = logging.getLogger(__name__)

B_IN_MB = 1 << 20
MAX_UNPACKING_WORKERS = 4

BaseIdentityKey = namedtuple('BaseIdentityKey', ('resource_id', 'use_packed'))


def truncate_base_prefix(base):
    if base.startswith('bs_'):
        return base.replace('bs_', '', 1)

    if base.startswith('yabs_'):
        return base.replace('yabs_', '', 1)

    return base


def get_transport_resource_path():
    unpacker_resource = sdk2.Resource.find(YABS_SERVER_BASE_PACK_EXECUTABLE, attrs={'released': ReleaseStatus.STABLE}).order(-sdk2.Resource.id).first()
    return os.path.join(str(new_resource_data(unpacker_resource).path), 'transport')


def unpack_base(src, dst, threads_count=None):
    transport = get_transport_resource_path()

    if threads_count is None:
        threads_count = multiprocessing.cpu_count()

    cmd = [transport, "decompress", src, dst, str(threads_count)]
    logger.debug("Run: \"%s\"", " ".join(cmd))
    logger_name = "transport_decompress_{}".format(os.path.basename(src))
    with sdk2.helpers.ProcessLog(sdk2.Task.current, logger=logger_name) as pl:
        subprocess.check_call(
            cmd,
            stdout=pl.stdout,
            stderr=pl.stderr,
        )


def generate_basenos(stats, stats_in_clusters):
    res = set()
    for stat_num in stats:
        for num in stats_in_clusters:
            res |= {no + 1 for no in range(stat_num - 1, 360, num)}
    return res


def get_bin_bases_unpacked_size_fast(rest_client, bases):
    bases = list(set(bases))
    if bases:
        resources = rest_client.resource.read(id=bases, limit=len(bases))['items']
        return sorted([
            int(resource['attributes']['unpacked_size'])
            for resource in resources
        ])
    return []


def get_bin_bases_unpacked_size(bases):
    bases = list(set(bases))
    if bases:
        resources = sdk2.Resource.find(id=bases).limit(len(bases))
        return sorted([
            int(resource.unpacked_size) // B_IN_MB
            for resource in resources
        ])
    return []


def get_bin_bases_packed_size(rest_client, bases):
    bases = list(set(bases))
    if bases:
        resources = rest_client.resource.read(id=bases, limit=len(bases))['items']
        return sorted([
            int(resource['size'])
            for resource in resources
        ])
    return []


def get_max_unpacking_workers(bin_bases_unpacked_size, ram, ramdrive):
    result = 1
    for unpacking_workers in range(len(bin_bases_unpacked_size)):
        if sum(bin_bases_unpacked_size[:unpacking_workers + 1]) >= ram - ramdrive:
            break
        result = unpacking_workers + 1
    return max(result, MAX_UNPACKING_WORKERS)
