from collections import namedtuple
from datetime import datetime, timedelta
from sandbox.projects.common.time_utils import datetime_to_timestamp
import dateutil.parser
import pytz


class InvalidDatetimeInterval(Exception):
    pass


class DatetimeInterval(namedtuple('DatetimeInterval', ('start', 'end'))):
    def __repr__(self):
        return u'{start} / {end}'.format(start=self.start.isoformat(), end=self.end.isoformat())

    def duration(self):
        return self.end - self.start

    @classmethod
    def from_str(cls, interval_str):
        """Construct DatetimeInterval from string

        :param interval_str: String in "{start}/{end}" format,
                             where `start` and `end` are datetimes in ISO-8601 standard
        :type interval_str: str
        """
        try:
            start_str, end_str = interval_str.split('/')
        except ValueError:
            raise InvalidDatetimeInterval('Invalid format: {}'.format(interval_str))

        start_datetime = dateutil.parser.parse(start_str)
        end_datetime = dateutil.parser.parse(end_str)

        if end_datetime <= start_datetime:
            raise InvalidDatetimeInterval(
                'The start of an interval is greater then the end: {}'.format(interval_str)
            )

        return cls(start_datetime, end_datetime)


def split_datetime_interval(start, duration, day_start_time, day_end_time):
    if not all([start.tzinfo, day_start_time.tzinfo, day_end_time.tzinfo]):
        raise ValueError(
            '"start", "day_start_time" and "day_end_time" objects must be timezone aware',
            start, day_start_time, day_end_time
        )
    if start.tzinfo != day_start_time.tzinfo or start.tzinfo != day_end_time.tzinfo:
        raise ValueError(
            '"start", "day_start_time" and "day_end_time" objects must have same timezone',
            start, day_start_time, day_end_time
        )

    if duration < timedelta():
        raise ValueError('"duration" must be non-negative')

    if day_start_time >= day_end_time:
        raise ValueError('"day_start_time" must be less or equal to "day_end_time"')

    day_start = datetime.combine(start, day_start_time)
    day_end = datetime.combine(start, day_end_time)
    if start.timetz() > day_end_time:
        day_start += timedelta(days=1)
        day_end += timedelta(days=1)

    interval_start = max(start, day_start)
    interval_end = interval_start + duration

    if interval_end <= day_end + timedelta(minutes=10):
        return [DatetimeInterval(start=interval_start, end=interval_end)]

    current_interval = DatetimeInterval(start=interval_start, end=day_end)
    next_intervals = split_datetime_interval(
        start=day_start + timedelta(days=1),
        duration=duration - current_interval.duration(),
        day_start_time=day_start_time,
        day_end_time=day_end_time
    )
    return [current_interval] + next_intervals


def round_ab_time_interval(analysis_interval):
    '''Rounding inside time interval to 30 minutes to create AB links'''
    start_timestamp = int(datetime_to_timestamp(analysis_interval.start))
    end_timestamp = int(datetime_to_timestamp(analysis_interval.end))
    delta = 30 * 60
    # If start interval like 12:30:01 round to 12:30:00
    if start_timestamp % delta > 60:
        rounded_start = start_timestamp + delta - (start_timestamp % delta)
    else:
        rounded_start = start_timestamp - start_timestamp % delta
    # If end interval like 12:29:59 round to 12:30:00
    if end_timestamp % delta < 29 * 60:
        rounded_end = end_timestamp - end_timestamp % delta
    else:
        rounded_end = end_timestamp + delta - (end_timestamp % delta)
    return datetime.fromtimestamp(rounded_start, tz=pytz.utc), datetime.fromtimestamp(rounded_end, tz=pytz.utc)
