import errno
import hashlib
import json
import logging
import os
import random
import shutil
import sys
import tarfile
from base64 import b64decode, b64encode
from contextlib import contextmanager
try:
    # python3
    from builtins import str as unicode
except ImportError:
    pass

import requests
import six

from sandbox import sdk2
from sandbox.common.rest import Client as SandboxRestClient
from sandbox.common.errors import ResourceNotFound, SandboxException
from sandbox.common.utils import get_task_link, get_resource_link


logger = logging.getLogger(__name__)


def truncate_string(line, limit=200, placeholder="..."):
    return line if len(line) <= limit else line[:limit] + placeholder


def copyanything(src, dst):
    try:
        shutil.copytree(src, dst)
    except OSError as exc:
        if exc.errno == errno.ENOTDIR:
            shutil.copy(src, dst)
        else:
            raise


def makedirs_exist(path):
    if os.path.isdir(path):
        return
    os.makedirs(path)


def makedirs_except(path):
    try:
        makedirs_exist(path)
    except Exception:
        logging.error('Failed to create directory {path}'.format(path=path))
        _, value, tb = sys.exc_info()
        e = SandboxException(value)
        e.__traceback__ = tb
        raise e


def is_precommit_check(task):
    return 'TESTENV-PRECOMMIT-CHECK' in task.Parameters.tags


def decode_text(encoded_object):
    result = encoded_object
    if '$encoding' in encoded_object:
        # Remove '$base64' after all yabs-server in testing move past r3772977
        if encoded_object['$encoding'] in ('$base64', 'base64'):
            result = b64decode(encoded_object['$data'])
        elif encoded_object['$encoding'] == 'plain':
            result = encoded_object['$data']

    if isinstance(result, unicode):
        return result

    return try_decode(result)


def encode_text(text, encoding='base64'):
    if encoding == 'base64':
        return {
            "$encoding": encoding,
            "$data": b64encode(text)
        }
    else:
        raise NotImplementedError("Unknown encoding: {}".format(encoding))


def try_decode(text):
    for encoding in ['utf-8', 'cp1251', 'latin-1', 'koi8-r']:
        try:
            return text.decode(encoding)
        except UnicodeDecodeError:
            pass

    return text.decode('string-escape')


def wrap_diff(key, diff):
    if diff.strip():
        return u'@@ {KEY} DIFF @@\n{diff}\n'.format(KEY=key.upper(), diff=diff)
    else:
        return u''


def html_hyperlink(link, text, target='_blank'):
    return '<a href="{link}" target="{target}">{text}</a>'.format(link=link, text=text, target=target)


def startrek_hyperlink(link, text):
    return '(({link} {text}))'.format(link=link, text=text)


def markdown_link(url, text):
    return '[{text}]({url})'.format(url=url, text=text)


def _get_task(task_id):
    sandbox_client = SandboxRestClient()
    try:
        return sandbox_client.task[task_id].read()
    except sandbox_client.HTTPError as e:
        if e.status == requests.codes.not_found:
            logger.error("Task not found: %s", task_id)
            return None
        raise e


def get_task_html_hyperlink(task_id):
    task = _get_task(task_id)
    if not task:
        return None
    return html_hyperlink(link=get_task_link(task_id), text='{task[type]} #{task[id]}'.format(task=task))


def get_task_markdown_hyperlink(task_id):
    task = _get_task(task_id)
    if not task:
        return None
    return markdown_link(url=get_task_link(task_id), text='{task[type]} #{task[id]}'.format(task=task))


def get_resource_html_hyperlink(resource_id):
    try:
        resource = sdk2.Resource[resource_id]
    except ResourceNotFound as e:
        logger.warn(e.message)
        return None
    return html_hyperlink(link=get_resource_link(resource_id),
                          text='{resource.type} #{resource.id}'.format(resource=resource))


def get_yt_path_html_hyperlink(proxy, path):
    url = 'https://yt.yandex-team.ru/{proxy}/navigation?path={path}'.format(proxy=proxy, path=path)
    return html_hyperlink(link=url, text=path)


@contextmanager
def random_seed(seed):
    current_random_state = random.getstate()
    random.seed(seed)
    yield
    random.setstate(current_random_state)


def is_bad_http_code(code):
    try:
        intcode = int(code)
    except ValueError:
        return True
    return (intcode < 100) or (intcode >= 400)


def is_http_client_error(code):
    try:
        intcode = int(code)
    except ValueError:
        return False
    return (intcode >= 400) and (intcode < 500)


def unpack_targz_package(package_path, output_dir):
    if not os.path.exists(output_dir) or os.listdir(output_dir):
        raise SandboxException("Can't unpack package {}. Output directory {} doesn't exist or isn't empty".format(package_path, output_dir))
    with tarfile.open(package_path, mode='r|gz') as tf:
        tf.extractall(path=output_dir)
    cur_dir = os.getcwd()

    def full_path(f):
        return os.path.join(cur_dir, output_dir, f)

    return map(full_path, os.listdir(output_dir))


def get_files_md5(file_list):
    if not file_list:
        return ''
    digest = hashlib.md5()
    for file_name in file_list:
        with open(file_name, 'rb') as f:
            data = f.read()
        digest.update(data)
    return digest.hexdigest()


def issubset_dict(dict_, other_):
    """Test whether every item in the dict is in other.
    It doesn't work with nested dictionaries.

    :type dict_: dict
    :type other_: dict
    :rtype: bool
    """
    return six.viewitems(dict_) <= six.viewitems(other_)


def get_json_md5(data):
    sorted_json_string = json.dumps(data, sort_keys=True)
    return hashlib.md5(sorted_json_string).hexdigest()


def calc_md5_from_json(json_string):
    if not json_string:
        return None
    data = json.loads(json_string)
    return get_json_md5(data)


def diff_dict(a, b):
    diff_string = ""
    for key in sorted(a.keys()):
        if key not in b:
            diff_string += "- {tag}: {resource_id}\n".format(tag=key, resource_id=a[key])
            continue
        if a[key] != b[key]:
            diff_string += "- {tag}: {resource_id}\n".format(tag=key, resource_id=a[key])
            diff_string += "+ {tag}: {resource_id}\n".format(tag=key, resource_id=b[key])
    for key in sorted(set(b.keys()) - set(a.keys())):
        diff_string += "+ {tag}: {resource_id}\n".format(tag=key, resource_id=b[key])
    return diff_string
