import json
import os
import requests

from sandbox import sdk2
from sandbox.common import rest
from sandbox.common.errors import ResourceNotFound, TaskError
from sandbox.common.types.task import ReleaseStatus
from sandbox.sdk2.paths import copy_path

from sandbox.projects.yabs.sandbox_task_tracing import trace_calls
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import new_resource_data


def get_last_released_resource(resource_type, release_status=ReleaseStatus.STABLE):
    client = rest.Client()

    response = client.release.read(resource_type=resource_type, type=release_status, limit=1)
    try:
        task_id = response['items'][0]['task_id']
    except (IndexError, KeyError):
        raise ResourceNotFound(
            'Failed to get last released resource of type {resource_type}. Sandbox\'s response: {response}'
            .format(resource_type=resource_type, response=response)
        )

    resource = sdk2.Resource[resource_type].find(task_id=task_id).first()
    if not resource:
        raise ResourceNotFound()

    return resource


@trace_calls(save_arguments=(0, 1, 'resource', 'resource_type'))
def sync_resource(resource=None, resource_type=None, destination_path=None, release_status=ReleaseStatus.STABLE):
    if not (resource or resource_type):
        raise TaskError('Need either resource_id or resource_type')

    if isinstance(resource, int):
        resource = sdk2.Resource[resource]
    resource = resource or get_last_released_resource(resource_type, release_status=release_status)

    if resource is None:
        raise TaskError('Failed to get resource %s', resource_type)

    path = str(new_resource_data(resource).path)
    if destination_path:
        copy_path(path, destination_path)
        return destination_path

    return path


def sync_resource_via_rest(resource_id):
    client = rest.Client()

    try:
        resource = client.resource.read(id=resource_id, limit=1)['items'][0]
    except IndexError:
        raise ResourceNotFound()

    r = requests.get(resource["http"]["proxy"])
    r.raise_for_status()

    resource_path = os.path.abspath(resource["file_name"])
    with open(resource_path, "w") as f:
        f.write(r.text)

    return resource_path


def get_resource_attributes(resource):
    if isinstance(resource, sdk2.Resource):
        resource = resource.id

    if sdk2.Task.current:
        return sdk2.Resource[resource].__attrs__

    client = rest.Client()
    return client.resource[resource].read()['attributes']


def json_from_resource(resource_id):
    sync_resource_func = sync_resource if sdk2.Task.current else sync_resource_via_rest
    resource_path = sync_resource_func(resource_id)
    with open(resource_path) as resource_file:
        return json.load(resource_file)


def resource_from_json(task, resource_type, file_name, data, description=""):
    resource = resource_type(task, description=description, path=file_name + ".json")
    resource.path.write_bytes(json.dumps(data))
    sdk2.ResourceData(resource).ready()
    return resource.id


def get_resource_id(resource):
    if isinstance(resource, sdk2.Resource):
        return int(resource.id)

    return resource
