import logging
import re

from sandbox.common import enum


logger = logging.getLogger(__name__)


SANDBOX_TASK_TAGS = {
    'oneshot_test': 'ONESHOT-TEST',
    'content_system_settings_change_test': 'CONTENT-SYSTEM-SETTINGS-CHANGE-TEST',
    'create_oneshot_spec': 'CREATE-ONESHOT-SPEC',
    'precommit_check': 'TESTENV-PRECOMMIT-CHECK',
    'commit_check': 'TESTENV-COMMIT-CHECK',
}


class RunType(enum.Enum):
    ONESHOT_TEST = 'oneshot_test'
    CONTENT_SYSTEM_SETTINGS_CHANGE_TEST = 'content_system_settings_change_test'
    CREATE_ONESHOT_SPEC = 'create_oneshot_spec'
    PRECOMMIT_CHECK = 'precommit_check'
    COMMIT_CHECK = 'commit_check'


class RunTypeGroup(enum.Enum):
    ONESHOT = None
    TRUNK = None


def get_task_run_type(tags):
    run_types = set()

    if SANDBOX_TASK_TAGS['oneshot_test'] in tags:
        run_types.add(RunType.ONESHOT_TEST)

    if SANDBOX_TASK_TAGS['content_system_settings_change_test'] in tags:
        run_types.add(RunType.CONTENT_SYSTEM_SETTINGS_CHANGE_TEST)

    if SANDBOX_TASK_TAGS['create_oneshot_spec'] in tags:
        run_types.add(RunType.CREATE_ONESHOT_SPEC)

    if SANDBOX_TASK_TAGS['precommit_check'] in tags:
        run_types.add(RunType.PRECOMMIT_CHECK)

    if SANDBOX_TASK_TAGS['commit_check'] in tags:
        run_types.add(RunType.COMMIT_CHECK)

    if len(run_types) > 1:
        logger.debug('Ambiguous run type: tags "%s" found', run_types)
        return None

    try:
        return run_types.pop()
    except KeyError:
        return None


def get_task_run_type_group(tags):
    run_type = get_task_run_type(tags)
    if run_type is None:
        return None

    oneshot_run_types = {
        RunType.ONESHOT_TEST,
        RunType.CONTENT_SYSTEM_SETTINGS_CHANGE_TEST,
        RunType.CREATE_ONESHOT_SPEC
    }

    trunk_run_types = {
        RunType.PRECOMMIT_CHECK,
        RunType.COMMIT_CHECK
    }

    if run_type in oneshot_run_types:
        return RunTypeGroup.ONESHOT

    if run_type in trunk_run_types:
        return RunTypeGroup.TRUNK

    return None


def get_task_testenv_database(tags):
    for tag in tags:
        match = re.match(r'TESTENV-DATABASE-(.*)', tag)
        if match:
            return match.group(1).lower()

    return None
