import datetime
import logging
import posixpath
import time
import urllib
from requests.compat import urlunparse


TABLET_CHECK_INTERVAL = 0.1


logger = logging.getLogger(__name__)


def _check_paths_arg_type(paths):
    if not isinstance(paths, (list, tuple, set)):
        raise TypeError('"paths" should be one of types: list, tuple, set, got {}'.format(type(paths)))


def mount_tables_batched(paths, client):
    _check_paths_arg_type(paths)
    if not paths:
        return

    batch_client = client.create_batch_client(raise_errors=True)

    map(batch_client.mount_table, paths)
    batch_client.commit_batch()
    wait_for_tablet_state(paths, 'mounted', client)


def freeze_tables_batched(paths, client):
    _check_paths_arg_type(paths)
    if not paths:
        return

    batch_client = client.create_batch_client(raise_errors=True)

    map(batch_client.freeze_table, paths)
    batch_client.commit_batch()
    wait_for_tablet_state(paths, 'frozen', client)


def wait_for_tablet_state(paths, state, client, timeout=60):
    from yt.wrapper import YtError

    _check_paths_arg_type(paths)

    batch_client = client.create_batch_client(raise_errors=True)

    start_time = time.time()
    while True:
        if start_time - time.time() > timeout:
            raise Exception('Timed out while waiting for tablets')

        responses = [
            batch_client.get(path + '/@tablet_state')
            for path in paths
        ]

        batch_client.commit_batch()

        try:
            tablets_state = set([response.get_result() for response in responses])
        except YtError:
            pass
        else:
            if tablets_state == {state}:
                return

        logger.debug('Not all tablets are in state "%s". Wait for %f seconds', state, TABLET_CHECK_INTERVAL)
        time.sleep(TABLET_CHECK_INTERVAL)


def get_dynamic_child_tables(path, client):
    return client.search(
        path,
        node_type=['table'],
        object_filter=lambda obj: obj.attributes.get('dynamic'),
        attributes=['dynamic']
    )


def set_yt_node_ttl(path, ttl, client):
    """
    Set TTL for YT node

    :param path: path of yt node
    :param ttl: TTL in seconds
    :param client: YT client instance
    """
    from yt.wrapper import YtHttpResponseError

    expiration_time = datetime.datetime.utcnow() + datetime.timedelta(seconds=ttl)
    try:
        client.set(path + '/@expiration_time', expiration_time.isoformat())
    except YtHttpResponseError as e:
        logger.warn('Failed to set TTL: %s', e.error['message'])


def set_yt_node_timeout(path, ttl, client):
    """
    Set TTL for YT node

    :param path: path of yt node
    :param ttl: TTL in seconds
    :param client: YT client instance
    """
    from yt.wrapper import YtHttpResponseError

    try:
        client.set(path + '/@expiration_timeout', ttl * 1000)
    except YtHttpResponseError as e:
        logger.warn('Failed to set TTL: %s', e.error['message'])


def set_node_attributes(path, attributes, yt_client):
    with yt_client.Transaction():
        for attr_name, attr_value in attributes.iteritems():
            yt_client.lock(
                path,
                mode='shared',
                attribute_key=attr_name,
                waitable=True,
                wait_for=10 * 1000,
            )
            yt_client.set_attribute(path, attr_name, attr_value)


def create_node(path, yt_client, attributes=None, ttl=None, ignore_existing=True, use_expiration_timeout=False):
    """Create YT map_node, set attributes and TTL

    :param path: str, path to the node
    :param yt_client: yt.wrapper.YtClient, YT client
    :param attributes: dict, attributes to be set for node
    :param ttl: int or datetime.timedelta, TTL for node in seconds
    :return: str, path of created node
    """

    attributes = attributes or {}
    logger.debug('Create node %s with attributes %s', path, attributes)
    yt_client.create('map_node', path, attributes=attributes, recursive=True, ignore_existing=ignore_existing)

    if ttl is not None:
        if isinstance(ttl, datetime.timedelta):
            ttl = ttl.total_seconds()
        logger.debug('Set TTL for node %s: %s', path, ttl)
        if use_expiration_timeout:
            set_yt_node_timeout(path=path, ttl=ttl, client=yt_client)
        else:
            set_yt_node_ttl(path=path, ttl=ttl, client=yt_client)

    return path


def create_tmp_node(client, parent_path, attributes=None, ttl=None, use_expiration_timeout=False):
    """
    Create temporary YT node

    :param client: yt.wrapper.YtClient, YT client
    :param parent_path: str, parent path for node
    :param attributes: dict, attributes to be set for node
    :param ttl: int, TTL for node in seconds, default 12 hours
    :return: str, path of created node
    """
    from yt.wrapper import ypath_join
    path_prefix = ypath_join(parent_path, 'tmp')
    tmp_path = client.find_free_subpath(path_prefix)
    if ttl is None:
        ttl = datetime.timedelta(hours=12).total_seconds()
    logging.info('Create temporary node: %s', tmp_path)
    return create_node(tmp_path, client, attributes=attributes, ttl=ttl, use_expiration_timeout=use_expiration_timeout)


def yt_copy(yt_client, src_path, dst_path):
    # Freeze dynamic tables in src_path
    for table_path in yt_client.search(
            src_path,
            node_type=["table"],
            object_filter=lambda obj: obj.attributes.get("dynamic"),
            attributes=["dynamic"]
    ):
        logger.info('Freezing table "%s"', table_path)
        yt_client.freeze_table(table_path, sync=True)

    logger.info('Copy from "%s" to "%s"', src_path, dst_path)
    yt_client.copy(src_path, dst_path, force=True, recursive=True)

    # Mount dynamic tables in dst_path
    for table_path in yt_client.search(
            dst_path,
            node_type=["table"],
            object_filter=lambda obj: obj.attributes.get("dynamic"),
            attributes=["dynamic"]
    ):
        logger.info('Mounting table "%s"', table_path)
        yt_client.mount_table(table_path, sync=True)


def yt_shallow_copy(yt_client, src, dst, exclude=None):
    paths = yt_client.search(src, map_node_order=None, enable_batch_mode=True)
    exclude = exclude or []
    for path in paths:
        if path in exclude:
            logger.info('Exclude %s', path)
            continue
        if yt_client.get_type(path) == 'map_node':
            logger.info('%s is map_node, skip', path)
            continue
        dst_path = path.replace(src, dst)
        logger.info('Create link %s -> %s', dst_path, path)
        yt_client.link(path, dst_path, recursive=True)


def get_operations_filter_link(yt_proxy_name, filter_by, start_time, finish_time, include_archive=True):
    query_data = {
        'from': start_time,
        'to': finish_time,
        'filter': filter_by,
    }
    if include_archive:
        query_data['dataMode'] = 'archive'
    return urlunparse(('https', 'yt.yandex-team.ru', posixpath.join(yt_proxy_name, 'operations'), None, urllib.urlencode(query_data), None))
