# coding: utf-8

import datetime
import json
import logging
import requests
import six


if six.PY3:
    from urllib.parse import urlunparse
    from urllib.parse import urlencode
else:
    from urlparse import urlunparse
    from urllib import urlencode

from jinja2 import Template

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.utils2 import resource_redirect_url
from sandbox.projects.runtime_models.BuildHitModelsFlameGraph import HitModelsBuildFlameGraph
from sandbox.projects.runtime_models.BuildLinearModelsFlameGraph import LinearModelsBuildFlameGraph
from sandbox.projects.yabs.YabsBuildFlameGraph import YabsBuildFlameGraph
from sandbox.projects.yabs.qa.pipeline.stage import stage
from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs import DATETIME_FORMAT
from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs import YabsServerRankingGraphs
from sandbox.projects.yabs.yt_logfeller_sampler import YtLogFellerHourSampler

REPORT_FILENAME = 'report.html'
TIME_REGEXP = r'^(\d\d:\d\d)$'
JNS_SENDER_PROJECT = 'VTeamNewYabs'
JNS_SENDER_TEMPLATE = 'ranking_digest'
JNS_URL = 'https://jns.yandex-team.ru/api/messages/send_to_channel_json'

SEPARATION_CLUSTERS = {
    'bs': ['101', '111', '1', '2', '11'],
    'yabs': ['103', '113', '3', '4', '13'],
    'bsrank': ['105', '118', '5', '6', '18'],
}
CLUSTER_TO_ROLE = {
    'bs': 'metapartner',
    'yabs': 'metasearch',
    'bsrank': 'metarank',
}

REPORT_TEMPLATE = '''<!DOCTYPE html>
<html>
    <head>
        <title>{{ title }}</title>
    </head>
    <body>
        <div>
            <ul>{% for name in message.params.names %}
                <li><a href="{{ message.params.graph_reports_urls[loop.index0] }}">{{ name }}</a> from {{
                    message.params.periods_start[loop.index0]
                }} to {{
                    message.params.periods_finish[loop.index0]
                }}</li>{% endfor %}
            </ul>
            By <a href="{{ message.params.task_url }}">task</a>
        </div>
    </body>
</html>'''


class PeriodsNames(object):
    Daily = 'daily'
    Weekly = 'weekly'


def _iter_tasks(
    task=None,
    short_start_period=None,
    short_finish_period=None,
    long_start_period=None,
    long_finish_period=None,
    iter_only_names=False,
):
    for start_period, finish_period, period_name in zip(
        (short_start_period, long_start_period),
        (short_finish_period, long_finish_period),
        (PeriodsNames.Daily, PeriodsNames.Weekly),
    ):
        for cluster in SEPARATION_CLUSTERS:
            res = dict(
                Name='_'.join((period_name, 'graph_task', cluster)),
                HumanName='{} graph for {}'.format(period_name, cluster),
                Start=start_period,
                Finish=finish_period,
            )
            if not iter_only_names:
                basic_task_params = dict(
                    balancer_key_enable=True,
                    balancer_key_whitelist=SEPARATION_CLUSTERS[cluster],
                    begin_time=start_period,
                    binary_executor_release_type=ctt.ReleaseStatus.STABLE,
                    cluster=task.Parameters.yt_cluster,
                    description='{}. Child of task {}'.format(res['HumanName'], task.id),
                    enable_grouping_stages_stats_graph=period_name == PeriodsNames.Weekly,
                    enable_pmatch_fails_by_exps=period_name == PeriodsNames.Daily,
                    enable_stage_timings_graph=period_name == PeriodsNames.Daily,
                    enable_stages_stats_graph=period_name == PeriodsNames.Daily,
                    end_time=finish_period,
                    notifications=task.Parameters.notifications,
                    owner=task.Parameters.owner,
                    report_ttl=task.Parameters.report_ttl,
                    user_definitions=task.Parameters.user_yql_definitions,
                    yql_token=task.Parameters.yql_token,
                )
            else:
                basic_task_params = None

            if iter_only_names:
                yield res
            elif getattr(task.Parameters, 'run_' + res['Name']):
                res.update(dict(
                    Task=lambda: YabsServerRankingGraphs(
                        task,
                        **basic_task_params
                    )
                ))
                yield res

            res = dict(
                Name='_'.join((period_name, 'stage_task', cluster, 'bad_pmatch')),
                HumanName='{} stage graph for {} for bad pmatch'.format(period_name, cluster),
                Start=start_period,
                Finish=finish_period,
            )
            if iter_only_names:
                yield res
            elif getattr(task.Parameters, 'run_' + res['Name']):
                basic_task_params.update(dict(
                    description='{}. Child of task {}'.format(res['HumanName'], task.id),
                ))
                res.update(dict(
                    Task=lambda: YabsServerRankingGraphs(
                        task,
                        pmatch_fails_bits_stat_groups_count_enable=True,
                        pmatch_fails_bits_stat_groups_count_param=6,
                        **basic_task_params
                    )
                ))
                yield res
            for pmatch_fails in (False, True):
                for program in ('stat', 'meta'):
                    res = dict(
                        Name='_'.join((period_name, 'flame_graph', cluster, program) + (('bad_pmatch',) if pmatch_fails else tuple())),
                        HumanName='{} {} flame graph for {}{}'.format(period_name, cluster, program, ' for bad pmatch' if pmatch_fails else ''),
                        Start=start_period,
                        Finish=finish_period,
                    )
                    if iter_only_names:
                        yield res
                    elif getattr(task.Parameters, 'run_' + res['Name']):
                        res.update(dict(
                            Task=lambda: YabsBuildFlameGraph(
                                task,
                                binary_executor_release_type=ctt.ReleaseStatus.STABLE,
                                mode='base',
                                base_parameters=dict(
                                    start_time=(datetime.datetime.strptime(start_period, DATETIME_FORMAT) - datetime.datetime(1970, 1, 1)).total_seconds(),
                                    end_time=(datetime.datetime.strptime(finish_period, DATETIME_FORMAT) - datetime.datetime(1970, 1, 1)).total_seconds(),
                                    pmatch_fails_bits_count_min=6 if pmatch_fails else 0,
                                    program=program,
                                    role=CLUSTER_TO_ROLE[cluster],
                                ),
                                description='{}. Child of task {}'.format(res['HumanName'], task.id),
                                yt_cluster=task.Parameters.yt_cluster,
                            )
                        ))
                        yield res

    res = dict(
        Name='_'.join((PeriodsNames.Daily, 'yabs_hit_models')),
        HumanName='{} flame graph for {}'.format(PeriodsNames.Daily, 'yabs_hit_models'),
        Start=short_start_period,
        Finish=short_finish_period,
    )
    if iter_only_names:
        yield res
    elif getattr(task.Parameters, 'run_' + res['Name']):
        res.update(dict(
            Task=lambda: HitModelsBuildFlameGraph(
                task,
                base_begin_time=short_start_period,
                base_end_time=short_finish_period,
                binary_executor_release_type=ctt.ReleaseStatus.STABLE,
                cluster=task.Parameters.yt_cluster,
                description='{}. Child of task {}'.format(res['HumanName'], task.id),
                diff_report=False,
                notifications=task.Parameters.notifications,
                owner=task.Parameters.owner,
                sampler=YtLogFellerHourSampler.name,
                user_definitions=task.Parameters.user_yql_definitions,
                yql_token=task.Parameters.yql_token,
            ),
        ))
        yield res

    res = dict(
        Name='_'.join((PeriodsNames.Daily, 'yabs_linear_models')),
        HumanName='{} flame graph for {}'.format(PeriodsNames.Daily, 'yabs_linear_models'),
        Start=short_start_period,
        Finish=short_finish_period,
    )
    if iter_only_names:
        yield res
    elif getattr(task.Parameters, 'run_' + res['Name']):
        res.update(dict(
            Task=lambda: LinearModelsBuildFlameGraph(
                task,
                base_begin_time=short_start_period,
                base_end_time=short_finish_period,
                binary_executor_release_type=ctt.ReleaseStatus.STABLE,
                cluster=task.Parameters.yt_cluster,
                description='{}. Child of task {}'.format(res['HumanName'], task.id),
                diff_report=False,
                notifications=task.Parameters.notifications,
                owner=task.Parameters.owner,
                sampler=YtLogFellerHourSampler.name,
                user_definitions=task.Parameters.user_yql_definitions,
                yql_token=task.Parameters.yql_token,
            ),
        ))
        yield res


def render_report(**kwargs):
    return Template(REPORT_TEMPLATE).render(**kwargs)


class YabsServerRankingDigestReport(sdk2.Resource):
    report_type = None
    ttl = 14


def generate_last_resource_link(report_type=None):
    params = dict()
    if report_type:
        params.update(dict(
            attrs=json.dumps(dict(
                report_type=report_type,
            )),
        ))
    return urlunparse((
        'https',
        'proxy.sandbox.yandex-team.ru',
        '/'.join(('last', 'YABS_SERVER_RANKING_DIGEST_REPORT')),
        None,
        urlencode(params),
        None
    ))


class GraphsParameters(sdk2.Parameters):
    for info in _iter_tasks(iter_only_names=True):
        sdk2.helpers.set_parameter(
            'run_' + info['Name'],
            sdk2.parameters.Bool('Build ' + info['HumanName'], default=False)
        )


class DebugParameters(sdk2.Parameters):
    for info in _iter_tasks(iter_only_names=True):
        sdk2.helpers.set_parameter(
            'debug_' + info['Name'],
            sdk2.parameters.Task('Debug task_id for ' + info['HumanName'])
        )


def get_report_url(task_id):
    report = sdk2.Task[task_id].Parameters.report
    return resource_redirect_url(resource_id=report.id)


def get_datetime(now, time, shift):
    time_parsed = map(int, time.split(':'))
    return (now.replace(
        hour=time_parsed[0],
        minute=time_parsed[1],
        second=0,
        microsecond=0,
    ) - datetime.timedelta(days=shift)).strftime(DATETIME_FORMAT)


class YabsServerRankingDigest(sdk2.Task):
    """Digest for Ranking group"""

    class Parameters(sdk2.Parameters):

        start_time = sdk2.parameters.StrictString(
            'Start time analyze report period',
            regexp=TIME_REGEXP,
            default='00:00',
            required=True,
        )
        start_shift = sdk2.parameters.Integer(
            'Shift in days for start from start task',
            default=0,
        )
        start_long_shift = sdk2.parameters.Integer(
            'Shift in day for start for long tasks from start task',
            default=0,
        )
        finish_time = sdk2.parameters.StrictString(
            'Finish time analyze report period',
            regexp=TIME_REGEXP,
            default='23:59',
            required=True,
        )
        finish_shift = sdk2.parameters.Integer(
            'Shift in days for finish from start task',
            default=0,
        )
        finish_long_shift = sdk2.parameters.Integer(
            'Shift in day for start for long tasks from start task',
            default=0,
        )
        user_yql_definitions = sdk2.parameters.String(
            'User text inserted in yql. For example for pragmas',
            default='',
            multiline=True,
        )

        with sdk2.parameters.Group('Graphs') as graphs_group:
            graphs_group_parameters = GraphsParameters()

        yql_token = sdk2.parameters.YavSecret('YQL token secret', required=True)
        juggler_token = sdk2.parameters.YavSecret('Juggler token secret', required=True)
        yt_cluster = sdk2.parameters.String('YT cluster', default='hahn', required=True)

        report_title = sdk2.parameters.String('Title for report html', default='Digest Report')
        report_type = sdk2.parameters.String('Value for report_type attr for report resource')
        jns_channels = sdk2.parameters.JSON('JNS channels in format: `{"project": ["channel_name"]}')

        report_ttl = sdk2.parameters.Integer('Report ttl in days', default=14)

        with sdk2.parameters.Group('Debug', collapse=True) as debug_group:
            debug_group_parameters = DebugParameters()

        with sdk2.parameters.Output():
            report = sdk2.parameters.Resource(
                'Report page',
                resource_type=YabsServerRankingDigestReport,
            )

    @stage(provides=('start_period', 'finish_period', 'long_start_period', 'long_finish_period'), result_is_dict=True)
    def get_period(self):
        now = datetime.datetime.now()

        return {
            'start_period': get_datetime(now, self.Parameters.start_time, self.Parameters.start_shift),
            'finish_period': get_datetime(now, self.Parameters.finish_time, self.Parameters.finish_shift),
            'long_start_period': get_datetime(now, self.Parameters.start_time, self.Parameters.start_long_shift),
            'long_finish_period': get_datetime(now, self.Parameters.finish_time, self.Parameters.finish_long_shift),
        }

    def get_task_id(self, name, task_func):
        task = getattr(self.Parameters, 'debug_' + name)
        logging.debug('For %s task id: %s', name, task)
        if task:
            return task.id

        context_field = name + '_task_id'
        task_id = getattr(self.Context, context_field)
        logging.debug('ContextField: %s, Value: %s', context_field, task_id)
        if task_id:
            return task_id

        task = task_func()
        task.save()
        task.enqueue()
        task_id = task.id
        logging.debug('TaskId: %s', task_id)
        setattr(self.Context, context_field, task_id)
        return task_id

    @stage(provides='tasks_info')
    def get_tasks_info(
        self,
        start_period,
        finish_period,
        long_start_period,
        long_finish_period,
    ):
        res = []
        for info in _iter_tasks(self, start_period, finish_period, long_start_period, long_finish_period):
            res.append({
                'id': self.get_task_id(info['Name'], info['Task']),
                'start': info['Start'],
                'finish': info['Finish'],
                'name': info['HumanName'],
            })
        logging.debug('TaskInfo:\n%s', res)
        return res

    @stage(provides='notification_message')
    def make_notification_message(self, tasks_info):
        names = []
        start_periods = []
        finish_periods = []
        urls = []
        for info in tasks_info:
            names.append(info['name'])
            start_periods.append(info['start'])
            finish_periods.append(info['finish'])
            urls.append(get_report_url(info['id']))
        return {
            'project': JNS_SENDER_PROJECT,
            'template': JNS_SENDER_TEMPLATE,
            'request_id': str(self.id),
            'params': {
                'names': names,
                'periods_start': start_periods,
                'periods_finish': finish_periods,
                'graph_reports_urls': urls,
                'task_url': lb.task_link(task_id=self.id, plain=True),
            },
        }

    def send_message_in_jns(self, message):
        responses = []
        message['params'].update(dict(
            current_report_resource_url=get_report_url(self.id),
            last_report_resource_url=generate_last_resource_link(self.Parameters.report_type),
        ))
        for project, channels in self.Parameters.jns_channels.items():
            for channel in channels:
                message.update({
                    'target_project': project,
                    'channel': channel,
                })
                logging.debug('MessageJson:\n%s', json.dumps(message, indent=4))
                response = requests.post(
                    JNS_URL,
                    json=message,
                    headers={
                        'Authorization': 'OAuth {}'.format(self.Parameters.juggler_token.value()),
                    },
                )
                logging.error('Response for %s %s: status code: %s, text: %s', project, channel, response.status_code, response.text)
                responses.append(response)

        for response in responses:
            response.raise_for_status()

    def create_report_resource(self, notification_message):
        if self.Parameters.report:
            return

        attrs = dict(ttl=self.Parameters.report_ttl)
        if self.Parameters.report_type:
            attrs.update(dict(
                report_type=self.Parameters.report_type,
            ))

        report = render_report(
            title=self.Parameters.report_title,
            message=notification_message,
        )
        report_resource = YabsServerRankingDigestReport(
            task=self,
            description='Report page',
            path=REPORT_FILENAME,
            **attrs
        )
        report_data = sdk2.ResourceData(report_resource)
        with report_data.path.open('w') as f:
            f.write(report.decode('utf-8'))
        report_data.ready()
        self.Parameters.report = report_resource

    def on_execute(self):
        self.get_period()
        tasks_info = self.get_tasks_info()

        with self.memoize_stage.waiting_report_tasks:
            raise sdk2.WaitTask(
                [
                    v['id'] for v in tasks_info
                ],
                [
                    ctt.Status.Group.FINISH,
                    ctt.Status.Group.BREAK,
                ],
                wait_all=True,
            )

        message = self.make_notification_message()
        self.create_report_resource(message)
        self.send_message_in_jns(message)
