# coding: utf-8

import datetime
import json
import logging

from collections import namedtuple

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, LastBinaryReleaseParameters
from sandbox.projects.common.environments import PipEnvironment
from sandbox.projects.common.utils2 import resource_redirect_link
from sandbox.projects.yabs.qa.pipeline.stage import stage

import sandbox.projects.common.link_builder as lb

from .definitions import Definitions
from .filtrations import Filtrations
from .graphs import Graphs
from .inputs import Inputs
from .samplers import Samplers
from .templates import render_request


DATETIME_FORMAT = '%Y-%m-%d %H:%M'


GroupingDescription = namedtuple(
    'GroupingDescription',
    (
        'Name',
        'Function',
        'TimeFormat',
    )
)

GroupingVariants = (
    GroupingDescription(
        Name='by hours',
        Function='SUBSTRING({}, 0, 13)',
        TimeFormat='%Y-%m-%d %H',
    ),
)

DEFAULT_GROUPING = GroupingVariants[0].Name

GroupingVariantsInvert = {
    group.Name: group
    for group in GroupingVariants
}


class YabsServerRankingGraphTable(sdk2.Resource):
    ttl = 7
    pass


class YabsServerRankingGraphReport(sdk2.Resource):
    ttl = 7
    pass


def parse_datetime(data):
    return datetime.datetime.strptime(data, DATETIME_FORMAT)


class DateTimeParameter(sdk2.parameters.String):
    description = 'Datetime in format: `{}`'.format(DATETIME_FORMAT)

    @classmethod
    def cast(cls, value):
        parse_datetime(value)
        return value


class FiltrationsParametersGroup(sdk2.Parameters):
    for filtration in Filtrations:
        sdk2.helpers.set_parameter(
            filtration.Name + '_filtration_group',
            filtration.get_parameter_group(),
        )


class InputsParametersGroup(sdk2.Parameters):
    for input in Inputs:
        for name, parameter in input.iter_parameters():
            sdk2.helpers.set_parameter(name, parameter())


class YabsServerRankingGraphsPlainText(sdk2.Resource):
    pass


class YabsServerRankingGraphs(LastBinaryTaskRelease, sdk2.Task):
    """Plot ranking graphs"""

    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('plotly'),
        )

    class Parameters(LastBinaryReleaseParameters):
        kill_timeout = 40 * 60

        yql_token = sdk2.parameters.YavSecret('YQL token secret', required=True)
        user_definitions = sdk2.parameters.String('User text inserted in yql. For example for pragmas',
                                                  default='', multiline=True)

        with sdk2.parameters.Group('YT Parameters') as yt_paths:
            cluster = sdk2.parameters.String('YT cluster', default='hahn', required=True)

        begin_time = DateTimeParameter(
            'Begin time in format `{}`'.format(DATETIME_FORMAT),
            default=datetime.datetime.now().strftime(DATETIME_FORMAT),
            required=True,
        )
        end_time = DateTimeParameter(
            'End time in format `{}`'.format(DATETIME_FORMAT),
            default=datetime.datetime.now().strftime(DATETIME_FORMAT),
            required=True,
        )

        with sdk2.parameters.Group('Graphs') as graphs_parameters:
            for graph in Graphs:
                sdk2.helpers.set_parameter(
                    'enable_' + graph.Name,
                    sdk2.parameters.Bool('Enable graph: ' + graph.Name)
                )

        with sdk2.parameters.RadioGroup('GroupingBy (Not Implemented)') as group_by:
            for group in GroupingVariants:
                group_by.values[group.Name] = group_by.Value(group.Name, default=DEFAULT_GROUPING == group.Name)

        with sdk2.parameters.Group('Filtrations') as filtrations:
            filtrations_parameters = FiltrationsParametersGroup()

        with sdk2.parameters.Output():
            report = sdk2.parameters.Resource(
                'Report page',
                resource_type=YabsServerRankingGraphReport,
            )

        with sdk2.parameters.Group('Input parameters', collapse=True) as input_parameters_group:
            input_parameters = InputsParametersGroup()

        with sdk2.parameters.Group('Report parameters') as report_parameters:
            report_ttl = sdk2.parameters.Integer('Resource ttl in days', default=14)

        with sdk2.parameters.Group('Debug', collapse=True) as debug_parameters:
            debug_operation_id = sdk2.parameters.String('OperationID of YQL')
            enable_profiler = sdk2.parameters.Bool('Enabling profiling task', default_value=False)

    def is_filtration_enable(self, name):
        if not getattr(self.Parameters, name + '_enable'):
            return False
        if getattr(self.Parameters, name + '_whitelist') or getattr(self.Parameters, name + '_blacklist'):
            return True
        return False

    def if_graph_enabled(self, graph):
        return getattr(self.Parameters, 'enable_' + graph.Name)

    @property
    def grouping_stages_stats_grouping_by(self):
        return GroupingVariantsInvert[self.Parameters.group_by].Function.format('iso_eventtime')

    def iter_ab_config_info(self):
        resource = sdk2.Resource.find(
            type='BIGB_AB_EXPERIMENTS_PRODUCTION_CONFIG',
            attrs={"released": "stable"},
        ).first()

        if not resource:
            raise errors.TaskError('Could not found resource')

        data = sdk2.ResourceData(resource)
        with open(str(data.path)) as f:
            for conf in json.load(f)['configs']:
                for info in conf['test_id_tree_info']['Infos']:
                    yield info

    def get_full_exps_res_id(self):
        full_exps_resource = YabsServerRankingGraphsPlainText(self, "Experiments of 100%", "full_exps.txt")
        full_exps_data = sdk2.ResourceData(full_exps_resource)

        with full_exps_data.path.open(mode='w') as exps_file:
            for info in self.iter_ab_config_info():
                if info['Percent'] == 100:
                    exps_file.write('{}\n'.format(info['TestId']))

        full_exps_data.ready()
        return full_exps_resource.id

    def get_parsed_begin_time(self):
        return parse_datetime(self.Parameters.begin_time)

    def get_parsed_end_time(self):
        return parse_datetime(self.Parameters.end_time)

    @stage(provides='operation_id')
    def get_operation_id(self):
        if self.Parameters.debug_operation_id:
            return self.Parameters.debug_operation_id

        from yql.api.v1.client import YqlClient
        from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs.lib.sources import SourcesDict

        enabled_definitions_names = set()
        enabled_dictionaries = set()
        enabled_graphs = list()
        enabled_sources_names = set()

        for graph in self.iter_graphs():
            enabled_graphs.append(graph)
            enabled_sources_names.add(graph.SourceName)

        for enabled_source_name in enabled_sources_names:
            enabled_definitions_names |= set(SourcesDict[enabled_source_name].get_definitions_names(self))

        for filtration in Filtrations:
            if filtration.is_enabled(self):
                enabled_dictionaries |= set(filtration.Dictionaries)
                enabled_definitions_names |= set(filtration.DefinitionsNames)

        render_request_params = dict(
            author_url=lb.staff_link(self.author, link_type=lb.LinkType.plain),
            enabled_definitions=(Definitions[name] for name in enabled_definitions_names),
            enabled_dictionaries=enabled_dictionaries,
            filtrations=Filtrations,
            enabled_graphs=enabled_graphs,
            enabled_sources=(SourcesDict[source](self) for source in enabled_sources_names),
            task=self,
            task_url=lb.task_link(self.id, plain=True),
            user_definitions=self.Parameters.user_definitions,
        )
        logging.debug('YQL Request Params: %s', render_request_params)
        client = YqlClient(db=self.Parameters.cluster, token=self.Parameters.yql_token.value())
        request = client.query(
            render_request(
                **render_request_params
            ),
            syntax_version=1,
            title='YQL YabsServerRankingGraphs',
        )
        request.run()
        self.set_info(
            '<a href="{0}" target="_blank">{1}</a>'.format(
                request.share_url,
                'Ссылка ны Ыкль'
            ),
            do_escape=False,
        )
        return request.operation_id

    def get_request(self, operation_id):
        from yql.client.operation import YqlOperationResultsRequest
        from yql.api.v1.client import config as YqlConfig

        YqlConfig.token = self.Parameters.yql_token.value()

        request = YqlOperationResultsRequest(operation_id)
        request.run()
        if request.in_progress:
            raise sdk2.WaitTime(300)

        if not request.is_success:
            raise Exception('\n'.join(map(str, request.errors)))

        return request

    def iter_graphs(self):
        for graph in Graphs:
            if self.if_graph_enabled(graph):
                yield graph

    def iter_graphs_names(self):
        for graph in self.iter_graphs():
            yield graph.Name

    def get_exp_name(self, exp_id_str):
        if self.ab_experiments is None:
            self.ab_experiments = {}
            for info in self.iter_ab_config_info():
                self.ab_experiments[info['TestId']] = info

        exp_id = int(exp_id_str)
        if exp_id not in self.ab_experiments:
            import requests
            r = requests.get('https://ab.yandex-team.ru/api/testid/{}?form=short'.format(exp_id))
            if r.status_code == requests.codes.ok:
                title = r.json()['title']
                self.ab_experiments[exp_id] = {
                    'Title': title
                }
            else:
                self.ab_experiments[exp_id] = None

        info = self.ab_experiments[exp_id]
        if info is None:
            return exp_id_str
        res = '{}: '.format(exp_id_str)

        percentage = info.get('Percent')
        if percentage is not None:
            res += '{:.1f}%: '.format(percentage)

        res += info['Title']
        return res

    def on_execute(self):
        if self.Parameters.enable_profiler:
            import cProfile
            profiler = cProfile.Profile()
            profiler.runcall(self._on_execute, self)
            profiler.sort_stats('cumulative')
            profiler.print_stats()
        else:
            self._on_execute()

    def _on_execute(self):
        self.ab_experiments = None

        operation_id = self.get_operation_id()

        begin_time = Samplers.Hours.RoundingFunction(
            parse_datetime(self.Parameters.begin_time)
        )
        end_time = Samplers.Hours.RoundingFunction(
            parse_datetime(self.Parameters.end_time)
        )

        request = self.get_request(operation_id)

        from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs.lib.graph_render import render_graphs

        report = render_graphs(
            task=self,
            request=request,
            begin_time=begin_time,
            end_time=end_time + datetime.timedelta(hours=1),
        )
        self.Parameters.report = report
        self.set_info(
            resource_redirect_link(
                report.id,
                'Ссылка на граф',
            ),
            do_escape=False,
        )
