# coding: utf-8

from sandbox import sdk2

from .definitions import BitsToSetDefinition
from .definitions import GetExperimentsSetDefinition
from .inputs import HitLogBase

ENABLE_SUFFIX = '_enable'
WHITELIST_SUFFIX = '_whitelist'
BLACKLIST_SUFFIX = '_blacklist'
PARAM_SUFFIX = '_param'


class FiltrationSetParametersGroup(sdk2.Parameters):
    enable = sdk2.parameters.Bool('Enable {filtration} filtration', default=False)
    if enable.value[True]:
        whitelist = sdk2.parameters.List('Whitelist for {filtration} filtration', value_type=sdk2.parameters.String)
        blacklist = sdk2.parameters.List('Blacklist for {filtration} filtration', value_type=sdk2.parameters.String)


class FiltrationByPMatchBitsCountParametersGroup(sdk2.Parameters):
    enable = sdk2.parameters.Bool('Enable filtration by PMatchFailsBits count >= pmatch_fails_bits_count', default=False)
    if enable.value[True]:
        param = sdk2.parameters.Integer('pmatch_bits_count', default=0, required=True)


class FiltrationInterface(object):
    Name = None
    LogsNames = tuple()
    Dictionaries = tuple()
    DefinitionsNames = tuple()

    @classmethod
    def is_enabled(cls, task):
        raise NotImplementedError()

    @classmethod
    def get_parameter_group(cls):
        raise NotImplementedError()

    @classmethod
    def iter_filtrations(cls, task):
        raise NotImplementedError()


class FiltrationSetInterface(FiltrationInterface):
    FiltrationSet = None
    HumanName = None

    @classmethod
    def is_enabled(cls, task):
        if not getattr(task.Parameters, cls.Name + ENABLE_SUFFIX):
            return False
        return any(getattr(task.Parameters, cls.Name + suffix) for suffix in (WHITELIST_SUFFIX, BLACKLIST_SUFFIX))

    @classmethod
    def get_parameter_group(cls):
        return FiltrationSetParametersGroup(
            prefix=cls.Name + '_',
            label_substs=dict(filtration=cls.HumanName),
        )

    @classmethod
    def iter_filtrations(cls, task):
        if cls.is_enabled(task):
            whitelist = getattr(task.Parameters, cls.Name + WHITELIST_SUFFIX)
            blacklist = getattr(task.Parameters, cls.Name + BLACKLIST_SUFFIX)
            filter_fmt = 'SetIsDisjoint({filtration_set}, AsList({list}))'
            if whitelist:
                yield 'NOT ' + filter_fmt.format(
                    filtration_set=cls.FiltrationSet,
                    list=', '.join(map('"{}"'.format, whitelist)),
                )
            if blacklist:
                yield filter_fmt.format(
                    filtration_set=cls.FiltrationSet,
                    list=', '.join(map('"{}"'.format, blacklist)),
                )


class FiltrationByPageIDSet(FiltrationSetInterface):
    Name = 'page_id'
    HumanName = 'PageID'
    FiltrationSet = 'ToSet(AsList({}.`pageid`))'.format(HitLogBase.Name)
    LogsNames = (HitLogBase.Name,)


class FiltrationByPageLabelsSet(FiltrationSetInterface):
    Name = 'page_labels'
    HumanName = 'Page.Labels'
    FiltrationSet = 'ToSet(String::SplitToList({}.`pagelabels`, ","))'.format(HitLogBase.Name)
    LogsNames = (HitLogBase.Name,)


class FiltrationByPmatchFailsBitsStatGroupsSet(FiltrationSetInterface):
    Name = 'pmatch_fails_bits_stat_groups'
    HumanName = 'StatsGroups PMatchFailBits'
    FiltrationSet = '${func_name}({log_name}.`pmatchfailbits`)'.format(
        func_name=BitsToSetDefinition.Name,
        log_name=HitLogBase.Name,
    )
    LogsNames = (HitLogBase.Name,)
    DefinitionsNames = (BitsToSetDefinition.Name,)


class FiltrationByYabsBalancerSamogonKeySet(FiltrationSetInterface):
    Name = 'balancer_key'
    HumanName = 'Yabs Balancer Samogon Key'
    FiltrationSet = 'ToSet([{}.`yabsbalancersamogonkey`])'.format(HitLogBase.Name)
    LogsNames = (HitLogBase.Name,)


class FiltrationByActiveTestIdsSet(FiltrationSetInterface):
    Name = 'active_test_ids'
    HumanName = 'Active Experiments Test ids'
    FiltrationSet = '${}({}.`activetestids`)'.format(
        GetExperimentsSetDefinition.Name,
        HitLogBase.Name,
    )
    LogsNames = (HitLogBase.Name,)
    DefinitionsNames = (GetExperimentsSetDefinition.Name,)


class FiltrationByPMatchFailsBitsCount(FiltrationInterface):
    Name = 'pmatch_fails_bits_stat_groups_count'
    LogsNames = (HitLogBase.Name,)

    @classmethod
    def is_enabled(cls, task):
        return getattr(task.Parameters, cls.Name + ENABLE_SUFFIX) and getattr(task.Parameters, cls.Name + PARAM_SUFFIX) > 0

    @classmethod
    def get_parameter_group(cls):
        return FiltrationByPMatchBitsCountParametersGroup(
            prefix=cls.Name + '_',
        )

    @classmethod
    def iter_filtrations(cls, task):
        if cls.is_enabled(task):
            yield 'ClickHouse::bitCount(CAST({hit_log_name}.`pmatchfailbits` AS UINT64)) >= {param}'.format(
                hit_log_name=HitLogBase.Name,
                param=getattr(task.Parameters, cls.Name + PARAM_SUFFIX),
            )


Filtrations = (
    FiltrationByPageIDSet,
    FiltrationByPageLabelsSet,
    FiltrationByPmatchFailsBitsStatGroupsSet,
    FiltrationByYabsBalancerSamogonKeySet,
    FiltrationByPMatchFailsBitsCount,
    FiltrationByActiveTestIdsSet,
)
