# coding: utf-8

from typing import Dict
from typing import Type

from sandbox import sdk2

from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs import YabsServerRankingGraphReport
from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs import YabsServerRankingGraphTable
from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs.graphs import GraphsDefinitions
from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs.templates import render_report


from .graph_render_base import TGraphRenderBase
from .grouping_graph import GroupingGraphRender
from .pmatch_fails_graph import PMatchFailsGraphRender
from .stage_timings_graph import StageTimingsGraphRender
from .stages_graph import StageGraphsRender


GraphMap: Dict[str, Type[TGraphRenderBase]] = {
    GraphsDefinitions.StagesStatsGraph.Name: StageGraphsRender,
    GraphsDefinitions.GroupingStagesStatsGraph.Name: GroupingGraphRender,
    GraphsDefinitions.StageTimingsGraph.Name: StageTimingsGraphRender,
    GraphsDefinitions.PMatchFailsByExps.Name: PMatchFailsGraphRender,
}


def expand_columns(row):
    import yql
    import pandas as pd
    from sandbox.projects.yabs.ranking_group.YabsServerRankingGraphs.lib.stage_stats import AllTableColumns

    names = []
    values = []
    for k, val in row.iteritems():
        if type(val) != yql.util.format.YqlStruct:
            names.append(k)
            values.append(val)
        else:
            for column in AllTableColumns:
                names.append(k + column)
                values.append(val.__dict__[column])
    return pd.Series(values, index=names)


def iter_graphs(task, request):
    for table, graph_name in zip(request.get_results(), task.iter_graphs_names()):
        graph = GraphMap[graph_name]
        df = table.full_dataframe.apply(expand_columns, axis=1)
        table_resource = sdk2.ResourceData(YabsServerRankingGraphTable(
            task=task,
            description='Table for graph: {}'.format(graph.Name),
            path='{}.pkl'.format(graph.FileName),
            ttl=task.Parameters.report_ttl,
        ))
        with table_resource.path.open('wb') as f:
            df.to_pickle(f, compression=None)
        table_resource.ready()

        yield graph.render(task, df)


def render_graphs(task, request, begin_time, end_time):
    data = '\n'.join(iter_graphs(task, request))

    report = render_report(
        begin_time=begin_time,
        end_time=end_time,
        data=data,
        task=task,
    )

    report_resource = YabsServerRankingGraphReport(
        task=task,
        description='Graphs',
        path='resource.html',
        ttl=task.Parameters.report_ttl,
    )
    report_data = sdk2.ResourceData(report_resource)
    with report_data.path.open('w') as f:
        f.write(report)
    report_data.ready()

    return report_resource
