# coding: utf-8

from typing import Iterable
from typing import Tuple
from typing import TypeVar

ORDER_COLUMN = 'StageOrder'
OVERALL_COLUMN = 'Overall'


class StageStatsColumnName(object):
    def __init__(self, table_column, graph_column):
        self.table_column: str = table_column
        self.graph_column: str = graph_column

    def get_select(self, source_name: str) -> str:
        return f'{source_name}.{self.table_column} AS {self.table_column}'


StageStatsColumnNameType = TypeVar('StageStatsColumnNameType', bound=StageStatsColumnName)


class StageStatsCorrectionColumnName(StageStatsColumnName):
    def get_select(self, source_name: str) -> str:
        return f'IF({source_name}.{self.table_column} == 1 OR {source_name}.{self.table_column} == 0, NULL, {source_name}.{self.table_column}) AS {self.table_column}'


class StageStatsGroup(object):
    def __init__(self, name: str, columns: Tuple[StageStatsColumnNameType, ...]):
        self.name: str = name
        self.columns: Tuple[StageStatsColumnNameType, ...] = columns

    def iter_table_columns(self) -> Iterable[str]:
        for column in self.columns:
            yield column.table_column

    def iter_columns(self):
        return self.columns


class StagesGraphSchema(object):
    def __init__(self, title: str, groups: Tuple[StageStatsGroup, ...]):
        self.title: str = title
        self.groups: Tuple[StageStatsGroup, ...] = groups

    def iter_table_columns(self) -> Iterable[str]:
        for group in self.groups:
            yield from group.iter_table_columns()

    def iter_columns(self):
        for group in self.groups:
            yield from group.iter_columns()


class StagesSchemas(object):
    BannerCounts: StagesGraphSchema = StagesGraphSchema('Banner Count', (
        StageStatsGroup('Overall', (
            StageStatsColumnName('Overall', OVERALL_COLUMN),
        )),
        StageStatsGroup('Smart Template', (
            StageStatsColumnName('SmartTemplate', 'Smart Template'),
        )),
        StageStatsGroup('Rmp', (
            StageStatsColumnName('Rmp', 'Rmp'),
        )),
        StageStatsGroup('By Product Type', (
            StageStatsColumnName('ProductTypeDirect', 'Direct'),
            StageStatsColumnName('ProductTypeMediaCreative', 'Media Creative'),
            StageStatsColumnName('ProductTypeMediaCreativeReach', 'Media Creative Reach'),
            StageStatsColumnName('ProductTypeMediaImage', 'Media Image'),
            StageStatsColumnName('ProductTypeMediaSmart', 'Media Smart'),
            StageStatsColumnName('ProductTypeVideoCreativeReach', 'Video Creative Reach'),
        )),
        StageStatsGroup('By Context Type', (
            StageStatsColumnName('ContextTypeBroad', 'Broad'),
            StageStatsColumnName('ContextTypeCompactBroad', 'CompactBroad'),
            StageStatsColumnName('ContextTypeDistribution', 'Distribution'),
            StageStatsColumnName('ContextTypeGoal', 'Goal'),
            StageStatsColumnName('ContextTypeNormal', 'Normal'),
            StageStatsColumnName('ContextTypeOffer', 'Offer'),
            StageStatsColumnName('ContextTypeRelevanceMatch', 'RelevanceMatch'),
        )),
        StageStatsGroup('By FF Banner Source', (
            StageStatsColumnName('FFBannerSourceADVM', 'ADVM'),
            StageStatsColumnName('FFBannerSourceAdvMachineRtQuorums', 'AdvMachineRtQuorums'),
            StageStatsColumnName('FFBannerSourceBroadMatch', 'BroadMatch'),
            StageStatsColumnName('FFBannerSourceGoal', 'Goal'),
            StageStatsColumnName('FFBannerSourceNone', 'None'),
            StageStatsColumnName('FFBannerSourceSmartAutoTargeting', 'SmartAutoTargeting'),
            StageStatsColumnName('FFBannerSourceUserDirect', 'UserDirect'),
        )),
    ))

    AverageValues = StagesGraphSchema('Average Values', (
        StageStatsGroup('Source Cost', (
            StageStatsColumnName('SourceCostAvg', 'SourceCost'),
        )),
        StageStatsGroup('Real Cost', (
            StageStatsColumnName('RealCostAvg', 'RealCost'),
        )),
        StageStatsGroup('CTR', (
            StageStatsColumnName('CtrAvg', 'CTR'),
        )),
        StageStatsGroup('PCTR', (
            StageStatsColumnName('PctrAvg', 'PCTR'),
        )),
        StageStatsGroup('AB Conversion Cost Coef', (
            StageStatsCorrectionColumnName('ABConversionCostCoefAvg', 'ABConversionCostCoef'),
        )),
        StageStatsGroup('Bid Correction', (
            StageStatsCorrectionColumnName('BidCorrectionAvg', 'BidCorrection'),
        )),
        StageStatsGroup('Value', (
            StageStatsColumnName('ValueAvg', 'Value'),
        )),
    ))

    @classmethod
    def iter_table_columns(cls):
        yield from cls.BannerCounts.iter_table_columns()
        yield from cls.AverageValues.iter_table_columns()

    @classmethod
    def iter_columns(cls):
        yield from cls.BannerCounts.iter_columns()
        yield from cls.AverageValues.iter_columns()


AllTableColumns: Tuple[str, ...] = tuple(StagesSchemas.iter_table_columns())
AllColumns: Tuple[StageStatsColumnNameType, ...] = tuple(StagesSchemas.iter_columns())
