import logging
from collections import defaultdict, namedtuple


NO_CMP_TASK = "no_cmp_task"
NO_DIFF = "no_diff"
DIFF_DATA = "diff"
RESOLVED_DIFF = "resolved_diff"
BROKEN = "broken"
FIXED = "fixed"

Interval = namedtuple("Interval", ("begin", "end", "status", "task_id", "database", "owner"))
Interval.__new__.__defaults__ = (NO_CMP_TASK, None, None, '')
Interval.__str__ = lambda self: "{self.status}({self.begin}, {self.end})".format(self=self)
Interval.__repr__ = lambda self: "{self.status}({self.begin}, {self.end})".format(self=self)

logger = logging.getLogger(__name__)


def get_sequence_score(pair):
    return pair[1]


def get_best_partition(candidates):
    choices = []
    for sequence in candidates:
        score = (
            sum(i.end - i.begin for i in sequence if i.status in (DIFF_DATA, NO_CMP_TASK, BROKEN)),
            -sum(1 for i in sequence if i.status in (DIFF_DATA, NO_CMP_TASK, BROKEN)),
            sum(i.end - i.begin for i in sequence if i.status in (DIFF_DATA, )),
            -sum(i.end - i.begin for i in sequence if i.status in (NO_DIFF, )),
        )
        choices.append((sequence, score))
    logger.debug("Choices score:\n%s", "\n".join("{}: {}".format(seq, score) for seq, score in choices))
    return min(choices, key=get_sequence_score)[0]


class Partitioner(object):
    def __init__(self, interval_map):
        self.intervals = set()
        self.interval_begin_map = defaultdict(list)

        for (begin, end), interval_info in interval_map.items():
            self.intervals.add((begin, end))
            self.interval_begin_map[begin].append(
                Interval(begin, end,
                         interval_info.get("diff_type", NO_CMP_TASK),
                         interval_info.get("task_id"),
                         interval_info.get("database"),
                         interval_info.get("owner", ''))
            )

        self.interval_begin_list = sorted(list(self.interval_begin_map.keys()))
        self._result_cache = {}

        logger.debug("Intervals: %s", self.intervals)
        logger.debug("Interval begin map: %s", self.interval_begin_map)

    def partition_interval(self, interval):
        if interval not in self._result_cache:
            self._result_cache[interval] = self._partition_interval(Interval(*interval))
        return self._result_cache[interval]

    def _partition_interval(self, interval):
        logger.debug("Started partitionting %s", interval)
        candidates = []
        if interval.begin == interval.end:
            return candidates

        for prefix in sorted(self.interval_begin_map.get(interval.begin, []), key=lambda i: (i.end - i.begin)):
            logger.debug("[covered] partition %s with existing prefix %s", interval, prefix)
            candidates.append([prefix] + self.partition_interval((prefix.end, interval.end)))

        for suffix_begin in self.interval_begin_list:
            if suffix_begin <= interval.begin:
                continue
            if (interval.begin, suffix_begin) in self.intervals:
                continue
            logger.debug("[skip_prefix] trying to partition %s with skipped prefix %s", interval, (interval.begin, suffix_begin))
            candidates.append([Interval(interval.begin, suffix_begin)] + self.partition_interval((suffix_begin, interval.end)))

        candidates = candidates or [[interval]]
        logger.debug("Candidates for %s are %s", interval, candidates)
        return get_best_partition(candidates)
