import logging
import textwrap
from collections import defaultdict

import jinja2

from sandbox.projects.yabs.release.binary_search import intervals
from sandbox.projects.yabs.qa.utils import general


logger = logging.getLogger(__name__)

STATUS_NAME_MAP = {
    intervals.NO_DIFF: "No diff",
    intervals.NO_CMP_TASK: "No CMP task",
    intervals.DIFF_DATA: "Unresolved diff",
    intervals.RESOLVED_DIFF: "Resolved diff",
}
STATUS_COLOR_MAP = {
    intervals.NO_DIFF: "green",
    intervals.RESOLVED_DIFF: "green",
    intervals.NO_CMP_TASK: "red",
    intervals.DIFF_DATA: "red",
}


def get_task_link(task_id):
    return 'https://sandbox.yandex-team.ru/task/{task_id}/view'.format(task_id=task_id)


def get_testenv_database_link(database):
    return 'https://beta-testenv.yandex-team.ru/project/{database}/timeline'.format(database=database)


def get_reports(task_ids, sandbox_client):
    reports = {task_id: "Task (({task_link} {task_id})) has no report".format(task_link=get_task_link(task_id), task_id=task_id) for task_id in task_ids}
    tasks_info = sandbox_client.task.read(id=task_ids, limit=len(task_ids))['items']
    for task_info in tasks_info:
        task_link = general.startrek_hyperlink(get_task_link(task_info["id"]), task_info["id"])
        try:
            reports[task_info["id"]] = "{report}\nComparison task: {task_link}".format(report=task_info["output_parameters"]["st_report"], task_link=task_link)
        except KeyError:
            logger.error("Task %s does not have st_report", task_info["id"])
    return reports


def get_report_data(sandbox_client, interval_sequence, problems_by_interval):
    summary = defaultdict(list)
    interval_table_rows = []

    task_ids = list(interval.task_id for interval in interval_sequence if interval.task_id)
    task_reports = get_reports(task_ids, sandbox_client)

    for interval in interval_sequence:
        diff_status = interval.status
        interval_table_row = {
            "begin": interval.begin,
            "end": interval.end,
        }

        interval_table_row["report"] = task_reports.get(interval.task_id, "No comparison task")
        if interval.database:
            interval_table_row["database"] = "From {}".format(general.startrek_hyperlink(get_testenv_database_link(interval.database), interval.database))

        if interval.status == intervals.DIFF_DATA:
            problem = problems_by_interval.get(interval.end, {})
            interval_table_row["resolved"] = problem.get("status") == "resolved"

            if interval_table_row["resolved"]:
                interval_table_row["resolve_comment"] = problem.get("resolve_comment", "")
                interval_table_row["revision_message"] = problem.get("diff", {}).get("right_revision", {}).get("message", "")
                diff_status = intervals.RESOLVED_DIFF

        interval_table_row["color"] = STATUS_COLOR_MAP.get(diff_status, "red")

        summary[diff_status].append(interval.end)
        interval_table_rows.append(interval_table_row)

    summary_rows = [{
        "color": STATUS_COLOR_MAP.get(status, "red"),
        "status": STATUS_NAME_MAP.get(status, status),
        "revisions": revisions,
    } for status, revisions in summary.items()]

    return summary_rows, interval_table_rows


def create_startrek_report(summary_rows, interval_rows, test_name):
    report_template = textwrap.dedent(u"""
    ==== {{ test_name }} ====
    #|
    || **Status** | **Number of intervals** | **Revisions** ||
    {% for summary_row in summary_rows -%}
    || !!({{ summary_row.color }}){{ summary_row.status }}!! | {{ summary_row.revisions|length }} |
    {% for revision in summary_row.revisions %}((https://a.yandex-team.ru/arc/commit/{{ revision }} {{ revision }})) {% endfor %} ||
    {% endfor -%}
    |#
    <{Table of intervals
    #|
    || **Revisions** | **Status** | **Is resolved** | **Resolve reason** | **Commit message** ||
    {% for row in interval_rows %}
    || !!({{ row.color }}){{ row.begin }}!! - ((https://a.yandex-team.ru/arc/commit/{{ row.end }} {{ row.end }})) | {{ row.report }}
    {{ row.database }} |
    {{ row.resolved }} | {{ row.resolve_comment }} | {{ row.revision_message }} ||
    {%- endfor %}
    |#
    }>
    """)
    env = jinja2.Environment()
    template = env.from_string(report_template)
    return template.render(summary_rows=summary_rows, interval_rows=interval_rows, test_name=test_name)
