import itertools
import logging

from sandbox.common.errors import TaskFailure

from testenv_data import get_test_results, get_values_and_resources

ROLE_TEST_PREFIX = 'YABS_SERVER_40_PERFORMANCE_AVG_'
ROLE_TEST_SUFFIXES = ['BS', 'YABS', 'BSRANK']

ROLE_TESTS_THRESHOLD = 2.0

TOTALS_TEST = 'YABS_SERVER_40_PERFORMANCE_TOTAL_AVG_WEIGHTED'
TOTALS_TEST_ALIAS = 'Weighted'

TOTALS_TEST_THRESHOLD = 1.5


def make_performance_report(first_revision, last_revision):
    return '\n'.join(itertools.chain(
        ['#|'],
        ['||{}||'.format('|'.join(str(_) for _ in line)) for line in _iter_performance_report(first_revision, last_revision)],
        ['|#'],
    ))


def _iter_performance_report(first_revision, last_revision):
    first_revision = int(first_revision)
    last_revision = int(last_revision)

    tests = [ROLE_TEST_PREFIX + s for s in ROLE_TEST_SUFFIXES] + [TOTALS_TEST]
    test_aliases = ROLE_TEST_SUFFIXES + [TOTALS_TEST_ALIAS]
    relative_thresholds = [ROLE_TESTS_THRESHOLD for _ in ROLE_TEST_SUFFIXES] + [TOTALS_TEST_THRESHOLD]

    test_data = get_test_results(tests, 'rps')
    results = {test: get_values_and_resources(points) for test, points in test_data.iteritems()}
    logging.debug(results)

    def iter_filtered_revisions():
        for test, result in results.iteritems():
            for revision in result.iterkeys():
                if revision >= first_revision and revision <= last_revision:
                    yield revision
    revisions = set(iter_filtered_revisions())

    start_revision = min(revisions)
    finish_revision = max(revisions)

    has_resource_switching = lambda rev: any(rev in result and result[rev].switched_res_id is not None for result in results.itervalues())
    all_switching_revisions = sorted(rev for rev in revisions if has_resource_switching(rev))

    def get_revision_values(rev, is_finish_revision=False):
        values = []
        missing_tests = []

        for test in tests:
            test_results = results[test]
            try:
                value = test_results[rev].value if test_results[rev].switched_res_id is None or is_finish_revision else test_results[rev].value_before_switching
                values.append(float(value))
            except KeyError:
                missing_tests.append(test)
        if missing_tests:
            raise TaskFailure("There is no data for r{} for tests: {}".format(rev, ", ".join(missing_tests)))
        return values

    def format_values(vals):
        return ['{val:.1f}'.format(val=val) for val in vals]

    def format_results(start_values, finish_values, thresholds):
        return [
            '**!!({mark}){status}!!** ({percent:.3f}%)'.format(
                mark=('green' if role_finish >= role_thresholds else 'red'),
                status=('OK' if role_finish >= role_thresholds else 'FAIL'),
                percent=(role_finish - role_start) * 100 / role_start
            )
            for role_start, role_finish, role_thresholds in zip(start_values, finish_values, thresholds)
        ]

    raw_start_values = get_revision_values(start_revision)

    yield ['revision'] + ['**{}**'.format(t) for t in test_aliases] + [' ']
    yield [start_revision] + format_values(raw_start_values) + ['(raw)']

    coeffs = [1.0 for _ in tests]

    for rev in all_switching_revisions:
        line = [rev]
        for n, test in enumerate(tests):
            data = results[test].get(rev)
            if data is not None and data.switched_res_id:
                start_values_prev = coeffs[n] * raw_start_values[n]
                multiplier = float(data.value) / float(data.value_before_switching)
                coeffs[n] *= multiplier
                line.append('x{:.2f} ({:.2f}->{:.2f})'.format(multiplier, start_values_prev, coeffs[n] * raw_start_values[n]))
            else:
                line.append(' ')
        line.append(' ')
        yield line

    corr_start_values = [val * coeff for val, coeff in zip(raw_start_values, coeffs)]
    raw_finish_values = get_revision_values(finish_revision, is_finish_revision=True)
    thresholds = [sv * (1.0 - 0.01 * rt) for sv, rt in zip(corr_start_values, relative_thresholds)]

    yield [start_revision] + format_values(corr_start_values) + ['(corrected for resource switching)']
    yield [finish_revision] + format_values(raw_finish_values) + ['(raw)']
    yield [''] + format_results(corr_start_values, raw_finish_values, thresholds) + ['Status (diff in percent)']
