import logging
import hashlib
import os
from collections import namedtuple
from datetime import timedelta
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import ProcessLog
from sandbox.common.types.task import Status as TaskStatus
from sandbox.common.types.misc import NotExists
from sandbox.common.utils import get_task_link
from sandbox.projects.common import constants as common_constants
from sandbox.projects.common.build.YaPackage2 import YaPackage2
from sandbox.projects.common.build.parameters import (
    ArcadiaPatch,
    ArcadiaUrl,
    BuildSystem,
    Sanitize,
    YtStore,
    YtProxy,
    YtDir,
    YtPut,
    YtTokenVaultOwner,
    YtTokenVaultName,
    YtStoreCodec,
    YtReplaceResult,
    YtMaxCacheSize,
)
from sandbox.projects.common.build.YaPackage import (
    TARBALL,
)
from sandbox.projects.yabs.qa.resource_types import BS_RELEASE_TAR, BS_RELEASE_YT, YabsResponseDumpUnpacker, YT_ONESHOTS_PACKAGE
from sandbox.projects.yabs.audit.runtime.monitoring.constants import AUDITABLE_BINARIES


from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace, trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import enqueue_task, new_resource
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import new_resource_data
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2.helpers import subprocess

logger = logging.getLogger(__name__)

ScheduledBuild = namedtuple("ScheduledBuild", ("target", "resource_type", "resource_id", "build_type", "task_id", "sanitize", "is_main"))
ScheduledBuild.__new__.__defaults__ = (False, )
BuildTarget = namedtuple("BuildTarget", ("build_type", "targets", "resource_types", "sanitize", "is_main"))
BuildTarget.__new__.__defaults__ = (None, False)
BinariesInfo = namedtuple("BinariesInfo", ("md5", "base_ver"))

KiB = 2**10
RESOURCES_BY_TARGETS = {
    "yabs/server/packages/yabs-server-bundle.json": BS_RELEASE_TAR.name,
    "yabs/server/cs/packages/yabs-cs.json": BS_RELEASE_YT.name,
}
RESOURCES_BY_TARGETS_DEBUG = {
    "yabs/server/packages/yabs-server-bundle.json": BS_RELEASE_TAR.name,
}
RESOURCES_BY_TOOLS = {
    "yabs/server/tools/dolbilo2json/package.json": YabsResponseDumpUnpacker.name,
}

BUILD_FIXES = [
    # (broken_revision, fixed_revision, arcadia_patch)
    # examples:
    #   (9160412, 9161375, 'https://paste.yandex-team.ru/7456067/text'), # got by svn diff -c 9161375 yabs | ya paste
    #   (9160412, 9161375, 'arc:2003908'), # for pull-request https://a.yandex-team.ru/review/2003908/details
    (9160412, 9161375, 'https://paste.yandex-team.ru/7456067/text'),
    (9272384, 9274771, 'arc:2413742'),
    (9376035, 9379418, 'arc:2489240'),
    (9615785, 9618871, 'https://paste.yandex-team.ru/10058460/text'),
    (9684485, 9685526, 'arc:2711140'),
    (9774841, 9776823, 'https://paste.yandex-team.ru/10825994/text'),
]


def compute_md5(path):
    try:
        hash_md5 = hashlib.md5()
        with open(path, "rb") as bf:
            for chunk in iter(lambda: bf.read(50 * KiB), b""):
                hash_md5.update(chunk)

        return hash_md5.hexdigest()
    except IOError as exc:
        logger.error(exc)

    return None


class BuildYabsServer(sdk2.Task):
    """Builds yabs_server runtime and basegen
    """

    __bs_release_tar_dir = None

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less

        environments = (
            environments.PipEnvironment('retrying'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        push_tasks_resource = True
        kill_timeout = timedelta(minutes=20).total_seconds()
        packages_to_build = sdk2.parameters.Dict("Target packages and target resource types", default=RESOURCES_BY_TARGETS)
        tools_to_build = sdk2.parameters.Dict("Tool packages and target resource types", description="Build only in release mode", default=RESOURCES_BY_TOOLS)
        packages_to_build_debug = sdk2.parameters.Dict("Target packages and target resource types for debug build", default=RESOURCES_BY_TARGETS_DEBUG)
        wait_for_debug_build = sdk2.parameters.Bool("Should the task wait for debug build", default=False)

        with sdk2.parameters.Group("Checkout parameters") as checkout_parameters:
            checkout_arcadia_from_url = ArcadiaUrl()
            arcadia_patch = ArcadiaPatch()

        with sdk2.parameters.Group("Build parameters") as build_parameters:
            build_system = BuildSystem(default=common_constants.YMAKE_BUILD_SYSTEM)
            fallback_build_system = BuildSystem()
            sanitize = Sanitize()
            max_retries = sdk2.parameters.Integer("Max retries to build targets", default=2)
            child_task_cores_requirement = sdk2.parameters.Integer("CPU cores requirement for child tasks", default=0)
            child_task_ram_requirement = sdk2.parameters.Integer("RAM requirement for child tasks (in GB)", default=128)

            with sdk2.parameters.RadioGroup("Build type") as build_type:
                build_type.values.release = build_type.Value(common_constants.RELEASE_BUILD_TYPE, default=True)
                build_type.values.debug = build_type.Value(common_constants.DEBUG_BUILD_TYPE)
                build_type.values.profile = build_type.Value(common_constants.PROFILE_BUILD_TYPE)
                build_type.values.coverage = build_type.Value(common_constants.COVERAGE_BUILD_TYPE)
                build_type.values.release_with_debug_info = build_type.Value(common_constants.RELEASE_WITH_DEBUG_INFO_BUILD_TYPE)
                build_type.values.valgrind_debug = build_type.Value(common_constants.VALGRIND_DEBUG_BUILD_TYPE)
                build_type.values.valgrind_release = build_type.Value(common_constants.VALGRIND_RELEASE_BUILD_TYPE)  # only for build with ya

            parallel_build = sdk2.parameters.Bool("Build all targets in separate tasks", default=True)

        with sdk2.parameters.Group("Build cache parameters") as build_cache_parameters:
            yt_store = YtStore(default=True)
            yt_proxy = YtProxy()
            yt_dir = YtDir()
            yt_token_vault_owner = YtTokenVaultOwner(default="YABS_SERVER_SANDBOX_TESTS")
            yt_token_vault_name = YtTokenVaultName(default="yabscs_yt_token")
            yt_put = YtPut()
            yt_store_codec = YtStoreCodec()
            yt_replace_result = YtReplaceResult()
            yt_max_cache_size = YtMaxCacheSize()

        with sdk2.parameters.Group("Postprocess parameters") as postprocess_parameters:
            compute_md5 = sdk2.parameters.Bool("Compute md5 of files in result archive", default=False)
            compute_version = sdk2.parameters.Bool("Compute version of the artifacts", default=False)

        with sdk2.parameters.Group("Miscellaneous parameters") as misc_parameters:
            resource_attributes = sdk2.parameters.Dict("Add following attributes to result resources", default={})

        with sdk2.parameters.Output:
            md5 = sdk2.parameters.Dict("MD5 hashes of the built targets")
            bs_release_tar_resource = sdk2.parameters.Resource("Yabs server package", resource_type=BS_RELEASE_TAR)
            bs_release_tar_debug_resource = sdk2.parameters.Resource("Yabs server package with debug build", resource_type=BS_RELEASE_TAR)
            bs_release_yt_resource = sdk2.parameters.Resource("Yabs CS package", resource_type=BS_RELEASE_YT)
            common_oneshots_resource = sdk2.parameters.Resource("Common oneshots resource", resource_type=YT_ONESHOTS_PACKAGE)

    @trace_calls(save_arguments=(1, 'bs_release_tar_resource'))
    def bs_release_tar_dir(self, bs_release_tar_resource):
        if self.__bs_release_tar_dir is None:
            resource_data_path = str(new_resource_data(bs_release_tar_resource).path)
            target_dir_path = os.path.join(os.getcwd(), "yabs-server-dir")
            if not os.path.isdir(target_dir_path):
                os.mkdir(target_dir_path)
            cmdline = ['tar', '-xzvf', resource_data_path, '-C', target_dir_path]
            logging.info("Running %s", ' '.join(cmdline))
            with ProcessLog(self, logging.getLogger('yabs_server_unpack')) as process_log:
                subprocess.popen_and_wait(cmdline, stdout=process_log.stdout, stderr=subprocess.STDOUT)
            self.__bs_release_tar_dir = target_dir_path

        return self.__bs_release_tar_dir

    def get_md5(self, bs_release_tar_dir):
        md5 = {}
        for file_name in AUDITABLE_BINARIES:
            md5[file_name] = compute_md5(os.path.join(bs_release_tar_dir, file_name))
        return md5

    def get_base_ver(self, bs_release_tar_dir):
        return subprocess.check_output([os.path.join(bs_release_tar_dir, 'yabs_chkdb')]).strip()

    def get_version(self):
        from sandbox.projects.yabs.release.version.version import get_version_from_arcadia_url
        return get_version_from_arcadia_url(self.Parameters.checkout_arcadia_from_url)

    def _get_common_package_resource_attrs(self, build_type, basic_version):
        return dict(
            self.Parameters.resource_attributes,
            global_key_type='parent_task_id',
            global_key=str(self.id),
            build_mode=build_type,
            major_version=basic_version.major,
            minor_version=basic_version.minor,
        )

    @staticmethod
    def get_targets_by_resources(package_dict, parallel_build=True):
        """
        :param: package_dict: key -- path to package.json, value -- resource_type
        """
        if not package_dict:
            return []
        if parallel_build:
            return [([target], [resource]) for target, resource in package_dict.items()]
        return [map(list, zip(*package_dict.items()))]

    def get_build_targets(self):
        targets_to_build = []
        for targets, resource_types in self.get_targets_by_resources(self.Parameters.packages_to_build, parallel_build=self.Parameters.parallel_build):
            targets_to_build.append(BuildTarget(self.Parameters.build_type, targets, resource_types, self.Parameters.sanitize, True))

        for targets, resource_types in self.get_targets_by_resources(self.Parameters.tools_to_build, parallel_build=False):
            targets_to_build.append(BuildTarget(common_constants.RELEASE_BUILD_TYPE, targets, resource_types))

        for targets, resource_types in self.get_targets_by_resources(self.Parameters.packages_to_build_debug, parallel_build=self.Parameters.parallel_build):
            targets_to_build.append(BuildTarget(common_constants.DEBUG_BUILD_TYPE, targets, resource_types, self.Parameters.sanitize, self.Parameters.wait_for_debug_build))

        return targets_to_build

    def should_propogate_resource(self, build_type):
        return build_type == self.Parameters.build_type or (build_type == "debug" and self.Parameters.wait_for_debug_build)

    @trace_calls(save_arguments='all')
    def schedule_builds(self, targets_to_build, build_system, attempt):
        from sandbox.sdk2 import svn

        arcadia_patch = self.Parameters.arcadia_patch
        parsed_url = svn.Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)

        if 'TESTENV-COMMIT-CHECK' in self.Parameters.tags and not arcadia_patch and parsed_url.trunk:
            current_revision = int(svn.Arcadia.get_revision(self.Parameters.checkout_arcadia_from_url))
            for broken_revision, fixed_revision, patch in BUILD_FIXES:
                if broken_revision <= current_revision < fixed_revision:
                    arcadia_patch = patch
                    break

        scheduled_builds = []
        delegated_builds = []
        basic_version = self.get_version()
        for build_target in targets_to_build:
            attrs = self._get_common_package_resource_attrs(build_target.build_type, basic_version)
            subtask_parameters = {
                # "kill_timeout": timedelta(hours=2).total_seconds(),  # TODO: igorock@ speed up yabs-cs build
                "checkout_arcadia_from_url": self.Parameters.checkout_arcadia_from_url,
                "arcadia_patch": arcadia_patch,
                "package_type": TARBALL,
                "build_system": build_system,
                "build_type": build_target.build_type,
                "use_aapi_fuse": True,
                "aapi_fallback": True,
                "use_arc_instead_of_aapi": True,
                "sanitize": build_target.sanitize,
                "packages": ";".join(build_target.targets) if isinstance(build_target.targets, (list, set, tuple)) else build_target.targets,
                "resource_type": ";".join(build_target.resource_types) if isinstance(build_target.resource_types, (list, set, tuple)) else build_target.resource_types,
                "package_resource_attrs": attrs,
                "custom_version": str(basic_version),
                "tags": ["{}_build".format(build_target.build_type)] + build_target.resource_types + self.Parameters.tags,
                "force_vcs_info_update": True,
                YtStore.name: self.Parameters.yt_store,
                YtProxy.name: self.Parameters.yt_proxy,
                YtDir.name: self.Parameters.yt_dir,
                YtPut.name: self.Parameters.yt_put,
                YtTokenVaultOwner.name: self.Parameters.yt_token_vault_owner,
                YtTokenVaultName.name: self.Parameters.yt_token_vault_name,
                YtStoreCodec.name: self.Parameters.yt_store_codec,
                YtReplaceResult.name: self.Parameters.yt_replace_result,
                YtMaxCacheSize.name: self.Parameters.yt_max_cache_size,
            }
            if self.should_propogate_resource(build_target.build_type):
                resource_ids = []
                for target, resource_type in zip(build_target.targets, build_target.resource_types):
                    resource = new_resource(
                        sdk2.Resource[resource_type],
                        self,
                        "[{build_type}] {resource_type} {basic_version}".format(resource_type=resource_type, build_type=build_target.build_type, basic_version=basic_version),
                        os.path.join(os.getcwd(), target.replace("/", ".") + ".{}.{}.tar.gz".format(build_target.build_type, attempt)),
                    )
                    resource_ids.append(resource.id)

                subtask_parameters.update(resource_id=list(map(str, resource_ids)))
            if build_target.is_main:
                subtask_parameters.update(
                    env_vars="YA_TOKEN_PATH='$(vault:file:robot-yabs-cs-sb:yabs_cs_sb_ya_token)'",
                    distbuild_pool="//man/users/yabs",
                )

            logger.debug("Running subtask with parameters: %s", subtask_parameters)
            build_task = YaPackage2(
                self,
                description="[{build_type}] {resource_type} {basic_version}".format(
                    resource_type=", ".join(build_target.resource_types),
                    build_type=build_target.build_type,
                    basic_version=basic_version,
                ),
                hints=list(self.hints),
                __requirements__={
                    "ram": self.Parameters.child_task_ram_requirement * (1 << 10),
                    "cores": self.Parameters.child_task_cores_requirement,
                    "tasks_resource": self.Requirements.tasks_resource,
                },
                **subtask_parameters
            )
            enqueue_task(build_task)

            if self.should_propogate_resource(build_target.build_type):
                for target, resource_type, resource_id in zip(build_target.targets, build_target.resource_types, resource_ids):
                    scheduled_builds.append(ScheduledBuild(target, resource_type, resource_id, build_target.build_type, build_task.id, build_target.sanitize, is_main=build_target.is_main))
            else:
                for target, resource_type in zip(build_target.targets, build_target.resource_types):
                    delegated_builds.append(ScheduledBuild(target, resource_type, None, build_target.build_type, build_task.id, build_target.sanitize, is_main=build_target.is_main))

        return scheduled_builds, delegated_builds

    @trace_calls(save_arguments='all')
    def postprocess_builds(self, scheduled_builds, compute_md5=False, compute_version=False):
        broken_builds = []
        package_version = None
        for scheduled_build in scheduled_builds:
            build_task = sdk2.Task[scheduled_build.task_id]
            resource = sdk2.Resource[scheduled_build.resource_id]
            if build_task.status == TaskStatus.SUCCESS:
                resource.revision = resource.svn_revision
                if isinstance(resource, BS_RELEASE_TAR):
                    logger.info("BS_RELEASE_TAR resource found: {}".format(scheduled_build.build_type))
                    if scheduled_build.build_type == self.Parameters.build_type:
                        if compute_md5:
                            md5 = self.get_md5(self.bs_release_tar_dir(resource))
                            self.Parameters.md5 = md5
                            for file_name, md5_hash in md5.items():
                                setattr(resource, "md5_{}".format(file_name), md5_hash)
                        if compute_version:
                            base_ver = self.get_base_ver(self.bs_release_tar_dir(resource))
                            package_version = "{revision}.{base_ver}~{branch}".format(
                                revision=resource.revision,
                                base_ver=base_ver,
                                branch=resource.branch,
                            )
                        self.Parameters.bs_release_tar_resource = resource
                    elif scheduled_build.build_type == "debug":
                        self.Parameters.bs_release_tar_debug_resource = resource
                elif isinstance(resource, BS_RELEASE_YT) and scheduled_build.build_type == self.Parameters.build_type:
                    self.Parameters.bs_release_yt_resource = resource
                    sdk2.ResourceData(resource)  # https://st.yandex-team.ru/SAMOGON-815
                elif isinstance(resource, YT_ONESHOTS_PACKAGE) and scheduled_build.build_type == self.Parameters.build_type:
                    self.Parameters.common_oneshots_resource = resource
            else:
                broken_builds.append(scheduled_build)

        if package_version is not None:
            self.Context.version = package_version
            for scheduled_build in scheduled_builds:
                resource = sdk2.Resource[scheduled_build.resource_id]
                resource.package_version = package_version

        return broken_builds

    @property
    def targets_to_build(self):
        return tuple(BuildTarget(*t) for t in self.Context.targets_to_build)

    @property
    def build_system(self):
        if self.Context.attempt > 1:
            return self.Parameters.fallback_build_system
        return self.Parameters.build_system

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        targets_to_build = self.get_build_targets()

        if self.Context.attempt is NotExists:
            self.Context.attempt = 0
        if self.Context.targets_to_build is NotExists:
            self.Context.targets_to_build = targets_to_build

        while self.Context.attempt <= self.Parameters.max_retries and self.Context.targets_to_build:
            attempt = self.Context.attempt
            with self.memoize_stage["schedule_build_{}".format(attempt)](commit_on_entrance=False), trace('schedule_build'):
                scheduled_builds, delegated_builds = self.schedule_builds(self.targets_to_build, self.build_system, attempt)
                logger.info("Scheduled builds: %s", scheduled_builds)
                logger.info("Delegated builds: %s", delegated_builds)
                self.Context.scheduled_builds = scheduled_builds
                self.Context.delegated_builds = delegated_builds
                if delegated_builds:
                    message = "Following builds were delegated to separate tasks and won't be awaited by this one:\n{delegated_builds}".format(
                        delegated_builds="\n".join([
                            "{resource_type} ({build_type}): <a href=\"{task_link}\" target=\"_blank\">{task_id}</a>".format(
                                resource_type=delegated_build.resource_type,
                                build_type=delegated_build.build_type,
                                task_id=delegated_build.task_id,
                                task_link=get_task_link(delegated_build.task_id),
                            ) for delegated_build in delegated_builds
                        ])
                    )
                    self.set_info(message, do_escape=False)

            with self.memoize_stage["wait_build_{}".format(attempt)]:
                raise sdk2.WaitTask(
                    [ScheduledBuild(*t).task_id for t in self.Context.scheduled_builds],
                    TaskStatus.Group.FINISH + TaskStatus.Group.BREAK,
                    wait_all=True,
                )

            broken_builds = self.postprocess_builds(
                [ScheduledBuild(*t) for t in self.Context.scheduled_builds],
                compute_md5=self.Parameters.compute_md5,
                compute_version=self.Parameters.compute_version,
            )
            if broken_builds:
                message = "Some tasks are broken:\n{broken_builds}".format(
                    broken_builds="\n".join([
                        "{resource_type} ({build_type}): <a href=\"{broken_task_link}\" target=\"_blank\">{broken_task_id}</a>".format(
                            resource_type=broken_build.resource_type,
                            build_type=broken_build.build_type,
                            broken_task_id=broken_build.task_id,
                            broken_task_link=get_task_link(broken_build.task_id),
                        ) for broken_build in broken_builds
                    ])
                )
                self.set_info(message, do_escape=False)
                self.server.batch.resources.delete = [b.resource_id for b in broken_builds]
                self.Context.attempt += 1
                self.Context.targets_to_build = [
                    BuildTarget(broken_build.build_type, [broken_build.target], [broken_build.resource_type], broken_build.sanitize, broken_build.is_main)
                    for broken_build in broken_builds
                ]
                logger.warning("There are broken build tasks: %s", broken_builds)
            else:
                self.set_info("All packages were built successfully")
                return

        raise Exception("Some of release builds are broken")
