import logging
import os

from sandbox import sdk2
from sandbox.common.rest import Client
from sandbox.common.proxy import NoAuth
from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.abc.client import AbcClient
from sandbox.projects.common.binary_task.deprecated import LastBinaryTaskRelease, LastBinaryReleaseParameters
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.components.all import get_component
from sandbox.projects.yabs.qa.template_utils import get_template
from sandbox.projects.yabs.release.notifications.jns.client import send_message
from sandbox.projects.yabs.release.notifications.jns.helpers import get_logins
from sandbox.projects.yabs.release.notifications.environment.report_info import BaseReportData

from sandbox.projects.yabs.audit.runtime.monitoring.requirements import check_requirements
from sandbox.projects.yabs.audit.runtime.monitoring.constants import AUDIT_REQUIREMENTS
from sandbox.projects.yabs.audit.runtime.monitoring.startrek import get_release_issue


logger = logging.getLogger(__name__)
TEMPLATES_DIR = os.path.join(os.path.dirname(__file__), '..', '..', '..', 'release', 'notifications', 'templates')
TEMPLATE_NAME = 'check_release_audit.j2'


class ReportData(BaseReportData):
    __slots__ = BaseReportData.__slots__ + ('errors', )

    def __init__(
            self,
            errors=(),
            **kwargs
    ):
        super(ReportData, self).__init__(**kwargs)

        self.tags = ['audit_release'] + self.tags
        self.errors = errors


class YabsServerCheckReleaseAuditStatus(LastBinaryTaskRelease, sdk2.Task):
    """Checks that current release was properly approved
    """
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less

        environments = [
            PipEnvironment('startrek_client', use_wheel=True),
            PipEnvironment('retrying'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        st_token = sdk2.parameters.String("Startrek token vault name", default_value='robot_yabs_releaser_token')
        build_task_id = sdk2.parameters.Integer("Build task id", required=True)
        component_name = sdk2.parameters.String("Component name", default="yabs_server")

        with sdk2.parameters.Group("Debug options") as debug_options:
            debug_mode = sdk2.parameters.Bool("Debug mode", default=False)
            with debug_mode.value[True]:
                use_production_sandbox = sdk2.parameters.Bool("Use production sandbox", default=True)

        ext_params = LastBinaryReleaseParameters()

    def get_recipients(self):
        if self.Context.copy_of:
            return {
                "yachats": {"internal": [{"login": self.author}]},
                "telegram": {"internal": [{"login": self.author}]},
            }
        return {
            "telegram": {"chat_name": ["yabs_server_release_chat"]},
            "yachats": {"chat_name": ["yabs_server_release_chat"]},
        }

    def _notify_event(self, report_data):
        report_template_j2 = get_template(TEMPLATE_NAME, templates_dir=TEMPLATES_DIR)

        tokens = sdk2.yav.Secret("sec-01fx7jcsjevejnypw63tk26nj3").data()
        try:
            spawn_users = list(set([self.author, AbcClient(tokens['abc_token']).get_current_duty_login(179, schedule_slug='yabs_frontend_duty_first')]))
            for transport, recipient in self.get_recipients().items():
                mentions = get_logins(self, spawn_users, transport=transport)
                report = report_template_j2.render(report_data.as_dict(transport=transport, mentions=mentions))
                send_message(report, tokens['juggler_token'], recipients={transport: recipient})
        except Exception:
            self.set_info('Cannot send notification')
            logger.error('Cannot send notification', exc_info=True)

        return report_template_j2.render(report_data.as_dict(transport='html'))

    def on_execute(self):
        if self.Parameters.debug_mode and self.Parameters.use_production_sandbox:
            sandbox_client = Client(base_url=Client.DEFAULT_BASE_URL, auth=NoAuth())
        else:
            sandbox_client = self.server
        startrek_helper = STHelper(token=sdk2.Vault.data(self.owner, self.Parameters.st_token))

        component_info = get_component(self.Parameters.component_name)

        issue = get_release_issue(self.Parameters.build_task_id, sandbox_client, startrek_helper, component_info)
        is_legit, errors = check_requirements(AUDIT_REQUIREMENTS, issue, self.Parameters.build_task_id, sandbox_client)
        if not is_legit:
            report_data = ReportData(
                component_name=self.Parameters.component_name,
                errors=set(errors),
                task_id=self.id,
                task_type=str(self.type)
            )
            report = self._notify_event(report_data)
            self.set_info(report, do_escape=False)
            raise TaskFailure('Audit release failed')
        self.set_info("Release is legit")
