import logging

from enum import Enum

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.types.misc import NotExists
from sandbox.projects.common.nanny.client import NannyClient, NannyApiException
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.yabs.release.tasks.DeployNannySnapshots import YabsServerDeployNannySnapshots, Events

logger = logging.getLogger(__name__)


class FilterType(Enum):
    service_id = "service_id"
    labels = "labels"


class YabsServerDeployNannyDashboard(YabsServerDeployNannySnapshots):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less

        environments = (
            environments.PipEnvironment('retrying'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):

        with sdk2.parameters.Group("Sandbox files update") as sandbox_files_settings:
            build_task = sdk2.parameters.Task('Build task')
            resource_types_to_update = sdk2.parameters.List("Resources to update", default=[])

        with sdk2.parameters.Group("Inline files update") as inline_files_settings:
            inline_file_path_to_update = sdk2.parameters.String("Inline file to update")
            inline_file_content = sdk2.parameters.String("Inline file content")

        with sdk2.parameters.Group("Nanny services filter") as services_filter:
            filter_type = sdk2.parameters.String("Filter type", choices=[(filter_type.name, filter_type.value) for filter_type in FilterType], default=FilterType.labels.value)

            with filter_type.value[FilterType.labels.value]:
                dashboard_id = sdk2.parameters.String("Dashboard identifier", required=True)
                filter_services_by_labels = sdk2.parameters.Dict("Filter dashboard services by labels", description="If empty, all services from dashboard will be updated")
                use_conjunction = sdk2.parameters.Bool("filter1 & filter2 & ...", default=True)
                use_negation = sdk2.parameters.Bool("!filter", default=False)

            with filter_type.value[FilterType.service_id.value]:
                service_ids = sdk2.parameters.List("List of Nanny services to update")

        with sdk2.parameters.Group("Release parameters") as release_parameters:
            component_name = rm_params.ComponentName2.component_name()
            release_type = sdk2.parameters.String(
                'Release type',
                choices=list((t, t) for t in (rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable)),
                default=rm_const.ReleaseStatus.testing,
            )
            major_version = sdk2.parameters.Integer("Major", default=0)
            minor_version = sdk2.parameters.Integer("Minor", default=0)

        with sdk2.parameters.Group("Nanny deployment parameters") as deployment_parameters:
            recipe_names = sdk2.parameters.List("Recipies to deploy (deprecated)", default=[])
            recipe_name = sdk2.parameters.String("Recipie to deploy")
            nanny_oauth_token_name = sdk2.parameters.String("Nanny OAuth token", default=None, description="If empty, token for robot-srch-releaser will be used")

        wait_deploy = sdk2.parameters.Bool("Wait for deploy to finish", default=False)
        with wait_deploy.value[True]:
            with sdk2.parameters.Group("Wait parameters") as wait_parameters:
                wait_deploy_period = sdk2.parameters.Integer("Wait deploy period", default=10 * 60)
                wait_deploy_stalled_warn_time = sdk2.parameters.Integer("Wait deploy timeout", default=4 * 60 * 60)
                wait_deploy_timeout = sdk2.parameters.Integer("Wait deploy timeout", default=6 * 60 * 60)

        notify = sdk2.parameters.Bool("Notify users via telegram", default=True)
        with notify.value[True]:
            with sdk2.parameters.CheckGroup("Notify about stages") as notify_stages:
                for stage_name, stage_value in Events.__members__.items():
                    notify_stages.values[stage_name] = notify_stages.Value("Deployment {}".format(stage_value), checked=True)

        with sdk2.parameters.Output:
            deploy_end_time = sdk2.parameters.Integer("Deploy successfully finished at")

    @staticmethod
    def get_service_ids(nanny_client, dashboard_id, filter_by_labels, use_conjunction=True):
        service_ids = nanny_client.get_dashboard_services(dashboard_id)
        if not isinstance(service_ids, list) or len(service_ids) == 0:
            logger.error("Expected non-empty list of service_ids, got this: %s", service_ids)
            raise NannyApiException("Expected non-empty list of service_ids")
        filtered_service_ids = []
        for service_id in service_ids:
            service_info = nanny_client.get_service(service_id)
            labels = [(label['key'], label.get('value')) for label in service_info['info_attrs']['content']['labels']]
            filter_function = all if use_conjunction else any
            if filter_function([
                label_key_value in labels
                for label_key_value in filter_by_labels
            ]):
                filtered_service_ids.append(service_id)
        return filtered_service_ids

    def create_snapshots(self, nanny_client, dashboard_id, build_task, resource_types_to_update, filter_services_by_labels, comment=""):
        service_ids = self.get_service_ids(nanny_client, dashboard_id, filter_services_by_labels, use_conjunction=self.Parameters.use_conjunction)
        snapshots = {}
        if build_task is not None:
            for service_id in service_ids:
                update_result = nanny_client.update_service_sandbox_file(
                    service_id,
                    unicode(build_task.type),
                    unicode(build_task.id),
                    deploy=False,
                    comment=comment,
                    resource_types=resource_types_to_update,
                    skip_not_existing_resources=True,
                )
                snapshots[service_id] = update_result['runtime_attrs']['_id']
        return snapshots

    @property
    def desired_version(self):
        from sandbox.projects.yabs.release.version.version import BasicVersion

        if self.Context.desired_version is NotExists:
            from sandbox.projects.yabs.release.version.sandbox_helpers import SandboxHelper
            sandbox_helper = SandboxHelper()
            if self.Parameters.build_task:
                basic_version = sandbox_helper.get_basic_version_from_task(self.Parameters.build_task.id)
            else:
                basic_version = BasicVersion(0, 0, 0)
            self.Context.desired_version = tuple(basic_version)

        return BasicVersion(*self.Context.desired_version)

    def on_execute(self):
        if self.Parameters.nanny_oauth_token_name:
            nanny_token = sdk2.Vault.data(self.Parameters.nanny_oauth_token_name)
        else:
            nanny_token = rm_sec.get_rm_token(self)
        nanny_client = NannyClient(rm_const.Urls.NANNY_BASE_URL, nanny_token)

        with self.memoize_stage.check_active(commit_on_entrance=False):
            self.check_active_deployments(nanny_client)
            logger.info("No active deplyments found")

        with self.memoize_stage.create_snapshots(commit_on_entrance=False):
            comment = "{} r{} {} release".format(self.Parameters.component_name, str(self.desired_version), self.Parameters.release_type)
            self.Context.created_snapshots = self.create_snapshots(
                nanny_client,
                self.Parameters.dashboard_id,
                self.Parameters.build_task,
                self.Parameters.resource_types_to_update,
                self.Parameters.filter_services_by_labels.items(),
                comment=comment,
            )

        with self.memoize_stage.deploy_dashboard(commit_on_entrance=False):
            self.Context.deployment_id = self.deploy_dashboard(nanny_client, self.Parameters.dashboard_id, self.Parameters.recipe_name, self.Context.created_snapshots)

        with self.memoize_stage.notify_deploy_dashboard(commit_on_entrance=False):
            self._notify_event(Events.deployment_started, deployment_id=self.Context.deployment_id)

        if self.Parameters.wait_deploy:
            self.wait_deploy(nanny_client, self.Context.deployment_id)
            self._notify_event(Events.deployment_succeeded, deployment_id=self.Context.deployment_id)
        else:
            self._notify_event(Events.deployment_scheduled, deployment_id=self.Context.deployment_id)
