import logging
import os
import time
import traceback

from collections import namedtuple
from enum import Enum

from sandbox import sdk2
from sandbox.common.types.misc import NotExists
from sandbox.common.errors import TaskStop, TaskFailure
from sandbox.projects.abc.client import AbcClient
from sandbox.projects.common.nanny.client import NannyClient, NannyApiException, TaskGroupStatus
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.yabs.qa.template_utils import get_template
from sandbox.projects.yabs.release.notifications.environment.report_info import BaseReportData
from sandbox.projects.yabs.release.notifications.jns.client import send_message
from sandbox.projects.yabs.release.notifications.jns.helpers import get_logins

logger = logging.getLogger(__name__)
TEMPLATES_DIR = os.path.join(os.path.dirname(__file__), '..', '..', 'notifications', 'templates')
Status = namedtuple('Status', ('text', 'icon'))


class Events(Enum):
    deployment_precondition_failed = Status("cannot be started", u"\U000026D4")
    deployment_started = Status("started", u"\U000025B6")
    deployment_scheduled = Status("scheduled", u"\U000023EF")
    deployment_succeeded = Status("succeeded", u"\U00002705")
    deployment_stalled = Status("stalled", u"\U00002757")
    deployment_rejected = Status("rejected", u"\U0001F6AB")
    deployment_timed_out = Status("timed out", u"\U00002757 \U0000231B")


class DeploymentLink(object):
    __slots__ = ('text', 'url')
    __url_template = 'https://nanny.yandex-team.ru/ui/#/services/dashboards/catalog/{dashboard_id}/deployments/catalog/{deployment_id}'

    def __init__(self, recipe_name, deployment_id, dashboard_id):
        self.text = recipe_name
        self.url = self.__url_template.format(dashboard_id=dashboard_id, deployment_id=deployment_id)


class ReportData(BaseReportData):
    __slots__ = BaseReportData.__slots__ + ('active_deployments', 'deployment', )

    def __init__(
            self,
            deployment_id=None,
            dashboard_id=None,
            recipe_name=None,
            active_deployments=None,
            **kwargs
    ):
        super(ReportData, self).__init__(**kwargs)

        self.tags = ['deploy'] + self.tags
        self.deployment = DeploymentLink(recipe_name, deployment_id, dashboard_id) if deployment_id else None
        self.active_deployments = active_deployments


class YabsServerDeployNannySnapshots(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        snapshots_to_deploy = sdk2.parameters.Dict("Snapshots to deploy", description="service_id to snapshot_id mapping")

        with sdk2.parameters.Group("Release parameters") as release_parameters:
            component_name = rm_params.ComponentName2.component_name()
            release_type = sdk2.parameters.String(
                'Release type',
                choices=list((t, t) for t in (rm_const.ReleaseStatus.testing, rm_const.ReleaseStatus.stable)),
                default=rm_const.ReleaseStatus.testing,
            )
            major_version = sdk2.parameters.Integer("Major", default=0)
            minor_version = sdk2.parameters.Integer("Minor", default=0)

        with sdk2.parameters.Group("Nanny deployment parameters") as deployment_parameters:
            recipe_name = sdk2.parameters.String("Recipie to deploy", required=True)
            dashboard_id = sdk2.parameters.String("Dashboard identifier", required=True)
            nanny_oauth_token_name = sdk2.parameters.String("Nanny OAuth token", default=None, description="If empty, token for robot-srch-releaser will be used")

        wait_deploy = sdk2.parameters.Bool("Wait for deploy to finish", default=False)
        with wait_deploy.value[True]:
            with sdk2.parameters.Group("Wait parameters") as wait_parameters:
                wait_deploy_period = sdk2.parameters.Integer("Wait deploy period", default=10 * 60)
                wait_deploy_stalled_warn_time = sdk2.parameters.Integer("Wait deploy timeout", default=4 * 60 * 60)
                wait_deploy_timeout = sdk2.parameters.Integer("Wait deploy timeout", default=6 * 60 * 60)

        notify = sdk2.parameters.Bool("Notify users via telegram", default=True)
        with notify.value[True]:
            with sdk2.parameters.CheckGroup("Notify about stages") as notify_stages:
                for stage_name, stage_value in Events.__members__.items():
                    notify_stages.values[stage_name] = notify_stages.Value("Deployment {}".format(stage_value), checked=True)

        with sdk2.parameters.Output:
            deploy_end_time = sdk2.parameters.Integer("Deploy successfully finished at")

    def get_recipients(self):
        if self.Context.copy_of:
            return {
                "yachats": {"internal": [{"login": self.author}]},
                "telegram": {"internal": [{"login": self.author}]},
            }
        if self.Parameters.release_type == rm_const.ReleaseStatus.testing:
            return {
                "telegram": {"chat_name": ["yabs_server_release_chat"]},
                "yachats": {"chat_name": ["yabs_server_release_chat"]},
            }
        if self.Parameters.release_type == rm_const.ReleaseStatus.stable:
            return {
                "telegram": {"chat_name": ["yabs_server_release_chat", "yabs_server_duty"]},
                "yachats": {"chat_name": ["yabs_server_release_chat", "yabs_server_duty"]},
            }
        return {}

    def _notify_event(self, event, deployment_id=None, active_deployments=(), template_name='deployment_status.j2'):
        report_data = ReportData(
            component_name=self.Parameters.component_name,
            recipe_name=self.Parameters.recipe_name,
            dashboard_id=self.Parameters.dashboard_id,
            deployment_id=deployment_id,
            active_deployments=active_deployments,
            status=event,
            release_type=self.Parameters.release_type,
            major_version=self.Parameters.major_version,
            minor_version=self.Parameters.minor_version,
            task_id=self.id,
            task_type=str(self.type)
        )
        report_template_j2 = get_template(template_name, templates_dir=TEMPLATES_DIR)
        if self.Parameters.notify and event.name in self.Parameters.notify_stages:
            logger.info("Trying to send notification about '%s' event", event.name)
            html_report = report_template_j2.render(report_data.as_dict(transport='html'))
            self.set_info(html_report, do_escape=False)

            tokens = sdk2.yav.Secret("sec-01fx7jcsjevejnypw63tk26nj3").data()
            try:
                spawn_users = list(set([self.author, AbcClient(tokens['abc_token']).get_current_duty_login(179, schedule_slug='yabs_frontend_duty_first')]))
                for transport, recipient in self.get_recipients().items():
                    mentions = get_logins(self, spawn_users, transport=transport)
                    report = report_template_j2.render(report_data.as_dict(transport=transport, mentions=mentions))
                    send_message(report, tokens['juggler_token'], recipients={transport: recipient})
            except Exception:
                self.set_info('Cannot send notification about {} event'.format(event.name))
                logger.error(traceback.format_exc())
        else:
            logger.info("Notification about '%s' event disabled", event.name)

    @staticmethod
    def get_active_deployments_info(nanny_client, dashboard_id, statuses=()):
        active_deployments = nanny_client.get_dashboard_taskgroups(dashboard_id, statuses=statuses)
        deployments_info = [
            DeploymentLink(
                d["meta"]["dashboard_deployment"].get("yaml_recipe_id", d["meta"]["dashboard_deployment"]["recipe_id"]),
                d["meta"]["dashboard_deployment"]["deployment_id"],
                dashboard_id,
            )
            for d in active_deployments
        ]
        return deployments_info

    @staticmethod
    def get_service_ids(nanny_client, dashboard_id, filter_by_labels, use_conjunction=True):
        service_ids = nanny_client.get_dashboard_services(dashboard_id)
        if not isinstance(service_ids, list) or len(service_ids) == 0:
            logger.error("Expected non-empty list of service_ids, got this: %s", service_ids)
            raise NannyApiException("Expected non-empty list of service_ids")
        filtered_service_ids = []
        for service_id in service_ids:
            service_info = nanny_client.get_service(service_id)
            labels = [(label['key'], label.get('value')) for label in service_info['info_attrs']['content']['labels']]
            filter_function = all if use_conjunction else any
            if filter_function([
                label_key_value in labels
                for label_key_value in filter_by_labels
            ]):
                filtered_service_ids.append(service_id)
        return filtered_service_ids

    @staticmethod
    def deploy_dashboard(nanny_client, dashboard_id, recipe_name, snapshots):
        deployment_result = nanny_client.deploy_dashboard_yaml(
            dashboard_id,
            recipe=recipe_name,
            service_snapshots=snapshots,
            pass_additional_service_info=('labels', 'gencfg_groups'),
        )
        logger.debug("Deployed recipe '%s' with result %s", recipe_name, deployment_result)
        deployment_id = deployment_result["_id"]
        return deployment_id

    def check_active_deployments(self, nanny_client):
        active_deployments = self.get_active_deployments_info(nanny_client, self.Parameters.dashboard_id, statuses=(TaskGroupStatus.MERGED, TaskGroupStatus.NEW, TaskGroupStatus.COMMITTED))
        if active_deployments:
            self._notify_event(Events.deployment_precondition_failed, active_deployments=active_deployments)
            raise TaskStop("Found {} deployments in progress, stopping. Please, finalize them manually then restart this task by pressing the 'Run' button.".format(len(active_deployments)))

    def wait_deploy(self, nanny_client, deployment_id):
        if self.Context.wait_deploy_start_time is NotExists:
            self.Context.wait_deploy_start_time = time.time()

        taskgroup_id = nanny_client.get_dashboard_deployment(self.Parameters.dashboard_id, deployment_id)['taskgroup_id']

        deployment_state = nanny_client.get_taskgroup_status(taskgroup_id)['status']

        if deployment_state not in (TaskGroupStatus.DONE, TaskGroupStatus.REJECTED):
            if time.time() - self.Context.wait_deploy_start_time > self.Parameters.wait_deploy_timeout:
                self._notify_event(Events.deployment_timed_out, deployment_id=deployment_id)
                raise TaskStop("Deployments haven't been finalized after {}s".format(self.Parameters.wait_deploy_timeout))

            if time.time() - self.Context.wait_deploy_start_time > self.Parameters.wait_deploy_stalled_warn_time:
                with self.memoize_stage.warn_deployment_stalled(commit_on_entrance=False):
                    self._notify_event(Events.deployment_stalled, deployment_id=deployment_id)

            raise sdk2.WaitTime(self.Parameters.wait_deploy_period)

        if deployment_state == TaskGroupStatus.REJECTED:
            self._notify_event(Events.deployment_rejected, deployment_id=deployment_id)
            raise TaskFailure("Deployment was rejected")

        self.Parameters.deploy_end_time = nanny_client.get_taskgroup_info(taskgroup_id)['finish_timestamp']

    def on_execute(self):
        if self.Parameters.nanny_oauth_token_name:
            nanny_token = sdk2.Vault.data(self.Parameters.nanny_oauth_token_name)
        else:
            nanny_token = rm_sec.get_rm_token(self)
        nanny_client = NannyClient(rm_const.Urls.NANNY_BASE_URL, nanny_token)

        with self.memoize_stage.check_active(commit_on_entrance=False):
            self.check_active_deployments(nanny_client)
            logger.info("No active deplyments found")

        with self.memoize_stage.deploy_dashboard(commit_on_entrance=False):
            self.Context.deployment_id = self.deploy_dashboard(nanny_client, self.Parameters.dashboard_id, self.Parameters.recipe_name, self.Parameters.snapshots_to_deploy)

        with self.memoize_stage.notify_deploy_dashboard(commit_on_entrance=False):
            self._notify_event(Events.deployment_started, deployment_id=self.Context.deployment_id)

        if self.Parameters.wait_deploy:
            self.wait_deploy(nanny_client, self.Context.deployment_id)
            self._notify_event(Events.deployment_succeeded, deployment_id=self.Context.deployment_id)
        else:
            self._notify_event(Events.deployment_scheduled, deployment_id=self.Context.deployment_id)
