import logging
import os
import time
import traceback

from collections import namedtuple, OrderedDict
from enum import Enum

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.errors import (
    TaskStop,
    TaskFailure,
)
from sandbox.common.types.misc import NotExists
from sandbox.common.types.task import Status
from sandbox.projects.abc.client import AbcClient
from sandbox.projects.yabs.qa.resource_types import BS_RELEASE_YT
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.yabs.qa.template_utils import get_template
from sandbox.projects.yabs.release.tasks.DeployYabsCS.util.filter import get_filtered_version
from sandbox.projects.yabs.release.notifications.environment.report_info import BaseReportData
from sandbox.projects.yabs.release.notifications.jns.client import send_message
from sandbox.projects.yabs.release.notifications.jns.helpers import get_logins

from sandbox.projects.samogon.deploy import DeploySamogonServants


logger = logging.getLogger(__name__)
TEMPLATES_DIR = os.path.join(os.path.dirname(__file__), '..', '..', 'notifications', 'templates')
CLUSTERAPI_URL_TEMPLATE = "https://clusterapi-{project}{namespace}.{balancer}.yandex-team.ru"
BALANCERS = ("n", "qloud")
DeploymentStatus = namedtuple('DeploymentStatus', ('text', 'icon'))


class Events(Enum):
    deployment_succeeded = DeploymentStatus("succeeded", u"\U00002705")
    already_performed = DeploymentStatus("already performed without any action made by this task", u"\U00002705")
    deployment_scheduled = DeploymentStatus("scheduled", u"\U000023EF")
    deployment_started = DeploymentStatus("started", u"\U000025B6")
    deployment_failed = DeploymentStatus("failed", u"\U0001F6AB")
    deployment_timed_out = DeploymentStatus("timed out", u"\U00002757 \U0000231B")
    version_syncronization_failed = DeploymentStatus("still cannot be started due to mismatched versions", u"\U000026D4")
    version_syncronization_temporary_failed = DeploymentStatus("cannot be started due to mismatched versions", u"\U000026D4")
    namespace_selection_failed = DeploymentStatus("cannot be started because correct namespace cannot be determined", u"\U000026D4")


class ServiceLabels(object):
    def __init__(self, server_version_helper):
        self.helper = server_version_helper

    def get(self, service_id, default):
        return self.helper.get_service_labels(service_id) or default


class NamespaceLink(object):
    __slots__ = ('text', 'url')

    def __init__(self, project, namespace):
        self.text = '{}{}'.format(project, namespace)
        self.url = 'https://{}{}.n.yandex-team.ru/'.format(project, namespace)


class Versions(object):
    __slots__ = ('deployed_cs_versions', 'deployed_server_versions')

    def __init__(self, deployed_server_versions, deployed_cs_versions):
        self.deployed_server_versions = OrderedDict()
        for v, services in sorted(deployed_server_versions.items(), key=lambda x: len(x[1])):
            self.deployed_server_versions[v] = len(services)

        self.deployed_cs_versions = OrderedDict()
        for namespace, versions in sorted(deployed_cs_versions.items(), key=lambda x: (x[0], len(x[1]))):
            self.deployed_cs_versions[namespace] = sorted(versions)


class ReportData(BaseReportData):
    __slots__ = BaseReportData.__slots__ + ('namespace', 'versions', )

    def __init__(
            self,
            project='yabscs',
            namespace=0,
            deployed_server_versions=None,
            deployed_cs_versions=None,
            **kwargs
    ):
        super(ReportData, self).__init__(**kwargs)

        self.tags = ['deploy'] + self.tags
        self.namespace = NamespaceLink(project, namespace) if project and namespace else None
        self.versions = Versions(deployed_server_versions, deployed_cs_versions) if deployed_server_versions and deployed_cs_versions else None


class DeployYabsCS(sdk2.Task):
    """YaBS Content-System deployment
    """
    name = "DEPLOY_YABS_CS"

    class Parameters(sdk2.Parameters):
        component_name = rm_params.ComponentName2.component_name(default_value="yabs_server")
        resource = sdk2.parameters.Resource("Resource to deploy", resource_type=BS_RELEASE_YT)
        project = sdk2.parameters.String("Samogon project", default="yabscs")
        namespaces = sdk2.parameters.List("Namespaces to check", default=["1", "2"])

        with sdk2.parameters.Group("Wait for cluster consistency") as consistency_settings:
            version_sync_period = sdk2.parameters.Integer("Versions sync period", default=10 * 60)
            server_versions_sync_timeout = sdk2.parameters.Integer("Timeout for server version sync", default=2 * 60 * 60)
            cs_versions_sync_timeout = sdk2.parameters.Integer("Timeout for CS version sync", default=2 * 60 * 60)
            ignored_server_services = sdk2.parameters.List(
                "Ignore following services in version detection",
                default=[
                    "experiment_yabs_frontend_server_bs_msk_iva_meta",
                    "experiment_yabs_frontend_server_bs_msk_iva_stat",
                    "experiment_yabs_frontend_server_yabs_sas_meta",
                    "experiment_yabs_frontend_server_yabs_sas_stat",
                    "prestable_yabs_frontend_server_bs_msk_iva_meta",
                    "prestable_yabs_frontend_server_bs_msk_iva_stat",
                    "prestable_yabs_frontend_server_yabs_sas_meta",
                    "prestable_yabs_frontend_server_yabs_sas_stat",
                    "preproduction_yabs_frontend_server_bs_dev1_meta",
                    "preproduction_yabs_frontend_server_bs_dev1_stat",
                    "mirror_asan_yabs_frontend_server_bs_sas1_meta",
                    "mirror_asan_yabs_frontend_server_bs_sas1_stat",
                    "mirror_yabs_frontend_server_bs_sas1_meta",
                    "mirror_yabs_frontend_server_bs_sas1_stat",
                    "mirror_yabs_frontend_server_yabs_sas1_meta",
                    "mirror_yabs_frontend_server_yabs_sas1_stat",
                ]
            )
            whitelist_nanny_labels = sdk2.parameters.Dict(
                "Check only service with that labels",
                default={},
            )

        with sdk2.parameters.Group("Deploy settings") as deploy_settings:
            wait_deploy = sdk2.parameters.Bool("Wait for deploy task to finish", default=False)
            with wait_deploy.value[True]:
                wait_deploy_timeout = sdk2.parameters.Integer("Wait for deploy no more than", default=30 * 60)
                wait_deploy_period = sdk2.parameters.Integer("Wait for deploy no more than", default=10 * 60)

            with sdk2.parameters.Group("Vault") as vault_parameters:
                samogon_oauth_token_vault_name = sdk2.parameters.String("Samogon OAuth token Vault name", required=True)

        with sdk2.parameters.Group("Notification settings") as notification_settings:
            notify = sdk2.parameters.Bool("Notify users via telegram", default=True)
            with notify.value[True]:
                with sdk2.parameters.CheckGroup("Notify about stages (deprecated)") as notify_stages:
                    for stage_name, stage_value in Events.__members__.items():
                        notify_stages.values[stage_name] = notify_stages.Value("Deployment {}".format(stage_value), checked=True)

        debug_mode = sdk2.parameters.Bool("Do no harm, perform only read-only actions", default=False)

    class Requirements(sdk2.Requirements):
        cores = 1

        environments = (
            environments.PipEnvironment('retrying'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def schedule_deploy(self, project, namespace, resource_id):
        deploy_task = DeploySamogonServants(
            self,
            project=project,
            namespace=namespace,
            auto=False,
            resource=sdk2.Resource[resource_id],
            samogon_oauth_token_vault_name=self.Parameters.samogon_oauth_token_vault_name,
        )
        deploy_task.enqueue()
        return deploy_task

    def get_recipients(self):
        if self.Context.copy_of:
            return {
                "yachats": {"internal": [{"login": self.author}]},
                "telegram": {"internal": [{"login": self.author}]},
            }
        return {
            "telegram": {"chat_name": ["yabs_server_release_chat"]},
            "yachats": {"chat_name": ["yabs_server_release_chat"]},
        }

    def _notify_event(self, event, deployed_server_versions=None, deployed_cs_versions=None, namespace=0, template_name='cs_deployment_status.j2'):
        from sandbox.projects.yabs.release.version.version import ServerVersion
        desired_version = ServerVersion(*self.Context.desired_version)

        report_data = ReportData(
            task_type=str(self.type),
            task_id=self.id,
            component_name=self.Parameters.component_name,
            major_version=desired_version.major,
            minor_version=desired_version.minor,
            project=self.Parameters.project,
            namespace=namespace,
            status=event,
            deployed_cs_versions=deployed_cs_versions,
            deployed_server_versions=deployed_server_versions,
        )

        report_template_j2 = get_template(template_name, templates_dir=TEMPLATES_DIR)
        if self.Parameters.notify:
            logger.info("Trying to send notification about '%s' event", event.name)
            html_report = report_template_j2.render(report_data.as_dict(transport='html'))
            self.set_info(html_report, do_escape=False)

            tokens = sdk2.yav.Secret("sec-01fx7jcsjevejnypw63tk26nj3").data()
            try:
                spawn_users = list(set([self.author, AbcClient(tokens['abc_token']).get_current_duty_login(179, schedule_slug='yabs_frontend_duty_first')]))
                for transport, recipient in self.get_recipients().items():
                    mentions = get_logins(self, spawn_users, transport=transport)
                    report = report_template_j2.render(report_data.as_dict(transport=transport, mentions=mentions))
                    send_message(report, tokens['juggler_token'], recipients={transport: recipient})
            except Exception:
                self.set_info('Cannot send notification about {} event'.format(event.name))
                logger.error(traceback.format_exc())
        else:
            logger.info("Notification about '%s' event disabled", event.name)

    def on_execute(self):
        from sandbox.projects.yabs.release.version.cs import CSVersionHelper
        from sandbox.projects.yabs.release.version.server import ServerVersionHelper
        from sandbox.projects.yabs.release.version.version import (
            ServerVersion,
            generate_versions_description,
            get_namespace_to_deploy,
            versions_synchronized,
        )
        nanny_token = rm_sec.get_rm_token(self)

        cs_version_helper = CSVersionHelper()
        server_version_helper = ServerVersionHelper(nanny_token=nanny_token)

        if self.Context.desired_version is NotExists:
            self.Context.desired_version = tuple(cs_version_helper.get_full_version_from_resource(self.Parameters.resource.id))

        if self.Context.sync_start_time is NotExists:
            self.Context.sync_start_time = time.time()

        deployed_cs_versions = cs_version_helper.get_deployed_cs_versions_by_namespace(self.Parameters.project, self.Parameters.namespaces)
        deployed_server_versions = get_filtered_version(
            versions_info=server_version_helper.get_deployed_versions('bsfront_production', 'BS_RELEASE_TAR'),
            whitelist_labels=self.Parameters.whitelist_nanny_labels or {},
            ignored_server_services=self.Parameters.ignored_server_services or [],
            service_labels=ServiceLabels(server_version_helper),
        )

        if self.Context.deploy_task_id is NotExists:
            for namespace, cs_versions in deployed_cs_versions.items():
                if len(cs_versions) == 1 and ServerVersion(*self.Context.desired_version) in cs_versions:
                    self.Context.namespace_to_deploy = namespace
                    self._notify_event(Events.already_performed, namespace=self.Context.namespace_to_deploy)
                    desired_version = ServerVersion(*self.Context.desired_version)
                    self.set_info("Version {v.major}-{v.minor}.{v.basever} already deployed at {project}_{namespace}".format(
                        v=desired_version, project=self.Parameters.project, namespace=namespace))
                    return

        versions_are_synced = versions_synchronized(deployed_server_versions, deployed_cs_versions)

        if not versions_are_synced:
            if time.time() - self.Context.sync_start_time > self.Parameters.cs_versions_sync_timeout:
                self._notify_event(
                    Events.version_syncronization_failed,
                    namespace=self.Context.namespace_to_deploy,
                    deployed_server_versions=deployed_server_versions,
                    deployed_cs_versions=deployed_cs_versions,
                )
                raise TaskStop("Frontend and CS versions weren't syncronized after {}s".format(self.Parameters.cs_versions_sync_timeout))
            with self.memoize_stage.notify_deployment_precondition_failed(commit_on_entrance=False):
                self._notify_event(
                    Events.version_syncronization_temporary_failed,
                    namespace=self.Context.namespace_to_deploy,
                    deployed_server_versions=deployed_server_versions,
                    deployed_cs_versions=deployed_cs_versions,
                )
            raise sdk2.WaitTime(self.Parameters.version_sync_period)

        with self.memoize_stage.select_namespace(commit_on_entrance=False):
            try:
                deployed_server_version = max(deployed_server_versions.items(), key=lambda x: len(x[1]))[0]
                self.Context.namespace_to_deploy = get_namespace_to_deploy(deployed_server_version, deployed_cs_versions)
                logger.info("Will deploy %s to %s_%s (detected by version)", self.Parameters.resource.id, self.Parameters.project, self.Context.namespace_to_deploy)
            except Exception as exc:
                logger.error("Cannot detect namespace to deploy because of %s", exc)
                self._notify_event(Events.namespace_selection_failed, deployed_server_versions=deployed_server_versions, deployed_cs_versions=deployed_cs_versions)
                raise TaskStop("Cannot detect namespace to deploy")

        if self.Parameters.debug_mode:
            self.set_info(generate_versions_description(deployed_server_versions, deployed_cs_versions))
            return

        with self.memoize_stage.schedule_deploy(commit_on_entrance=False):
            deploy_task = self.schedule_deploy(self.Parameters.project, self.Context.namespace_to_deploy, self.Parameters.resource.id)
            self._notify_event(Events.deployment_started, namespace=self.Context.namespace_to_deploy)

            self.Context.deploy_task_id = deploy_task.id

        desired_version = ServerVersion(*self.Context.desired_version)

        if self.Parameters.wait_deploy:
            with self.memoize_stage.wait_deploy:
                raise sdk2.WaitTask([self.Context.deploy_task_id], Status.Group.FINISH + Status.Group.BREAK)

            if sdk2.Task[self.Context.deploy_task_id].status != Status.SUCCESS:
                self._notify_event(Events.deployment_failed, namespace=self.Context.namespace_to_deploy)
                raise TaskFailure("Deploy task is in {} status".format(sdk2.Task[self.Context.deploy_task_id].status))
            else:
                if self.Context.wait_deploy_start_time is NotExists:
                    self.Context.wait_deploy_start_time = time.time()

                deployed_cs_versions = cs_version_helper.get_deployed_cs_versions(self.Parameters.project, self.Context.namespace_to_deploy)
                if set(deployed_cs_versions) != {desired_version}:
                    if time.time() - self.Context.wait_deploy_start_time > self.Parameters.wait_deploy_timeout:
                        self._notify_event(Events.deployment_timed_out, namespace=self.Context.namespace_to_deploy)
                        raise TaskStop("CS versions weren't syncronized after {}s".format(self.Parameters.wait_deploy_timeout))
                    raise sdk2.WaitTime(self.Parameters.wait_deploy_period)

                self.set_info("Version {v.major}-{v.minor}.{v.basever} successfully deployed at {project}_{namespace}".format(
                    v=desired_version, project=self.Parameters.project, namespace=self.Context.namespace_to_deploy))
                self._notify_event(Events.deployment_succeeded, namespace=self.Context.namespace_to_deploy)
        else:
            self.set_info("Version {v.major}-{v.minor}.{v.basever} successfully scheduled for deploy at {project}_{namespace}".format(
                v=desired_version, project=self.Parameters.project, namespace=self.Context.namespace_to_deploy))
